"""reST directive for syntax-highlighting itango interactive sessions."""

# -----------------------------------------------------------------------------
# Needed modules

# Standard library
import copy
import re

import pygments.styles

# Third party
from pygments.lexer import Lexer, do_insertions
from pygments.lexers.agile import PythonLexer
from pygments.token import Comment, Generic
from sphinx import highlighting

# -----------------------------------------------------------------------------
# Global constants
line_re = re.compile(".*?\n")

DftStyle = pygments.styles.get_style_by_name("default")


class TangoStyle(DftStyle):
    styles = copy.copy(DftStyle.styles)
    styles[Generic.Prompt] = "bold #00AA00"


class TangoConsoleLexer(Lexer):
    """
    For itango console output or doctests, such as:

    .. sourcecode:: itango

      ITango [1]: a = 'foo'

      ITango [2]: a
                   Result [2]: 'foo'

      ITango [3]: print a
      foo

      ITango [4]: 1 / 0

    Notes:

      - Tracebacks are not currently supported.

      - It assumes the default itango prompts, not customized ones.
    """

    name = "ITango console session"
    aliases = ["itango"]
    mimetypes = ["text/x-itango-console"]
    input_prompt = re.compile("(ITango \[(?P<N>[0-9]+)\]: )|(   \.\.\.+:)")
    output_prompt = re.compile("(\s*Result \[(?P<N>[0-9]+)\]: )|(   \.\.\.+:)")
    continue_prompt = re.compile("   \.\.\.+:")
    tb_start = re.compile("\-+")

    def get_tokens_unprocessed(self, text):
        pylexer = PythonLexer(**self.options)

        curcode = ""
        insertions = []
        for match in line_re.finditer(text):
            line = match.group()
            input_prompt = self.input_prompt.match(line)
            continue_prompt = self.continue_prompt.match(line.rstrip())
            output_prompt = self.output_prompt.match(line)
            if line.startswith("#"):
                insertions.append((len(curcode), [(0, Comment, line)]))
            elif input_prompt is not None:
                insertions.append(
                    (len(curcode), [(0, Generic.Prompt, input_prompt.group())])
                )
                curcode += line[input_prompt.end() :]
            elif continue_prompt is not None:
                insertions.append(
                    (len(curcode), [(0, Generic.Prompt, continue_prompt.group())])
                )
                curcode += line[continue_prompt.end() :]
            elif output_prompt is not None:
                # Use the 'error' token for output.  We should probably make
                # our own token, but error is typicaly in a bright color like
                # red, so it works fine for our output prompts.
                insertions.append(
                    (len(curcode), [(0, Generic.Error, output_prompt.group())])
                )
                curcode += line[output_prompt.end() :]
            else:
                if curcode:
                    for item in do_insertions(
                        insertions, pylexer.get_tokens_unprocessed(curcode)
                    ):
                        yield item
                        curcode = ""
                        insertions = []
                yield match.start(), Generic.Output, line
        if curcode:
            for item in do_insertions(
                insertions, pylexer.get_tokens_unprocessed(curcode)
            ):
                yield item


def setup(app):
    """Setup as a sphinx extension."""

    # This is only a lexer, so adding it below to pygments appears sufficient.
    # But if somebody knows that the right API usage should be to do that via
    # sphinx, by all means fix it here.  At least having this setup.py
    # suppresses the sphinx warning we'd get without it.
    pass


# -----------------------------------------------------------------------------
# Register the extension as a valid pygments lexer
highlighting.lexers["itango"] = TangoConsoleLexer()
