/* ----------------------------------------------------------------

SmillaEnlarger  -  resize, especially magnify bitmaps in high quality
    selectField.cpp: the select-widget within the EnlargerDialog

Copyright (C) 2009 Mischa Lusteck

This program is free software;
you can redistribute it and/or modify it under the terms of the
GNU General Public License as published by the Free Software Foundation;
either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <http://www.gnu.org/licenses/>.

---------------------------------------------------------------------- */

#include "selectField.h"
#include <QtGui>
#include <iostream>
using namespace std;

const int selectBorderWidth = 14;
SelectField::SelectField( QWidget *parent ) {
    setAttribute(Qt::WA_StaticContents);
    setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
    myImage = QImage( 200, 200, QImage::Format_ARGB32 );
    myImage.fill( qRgb(0,0,255) );
    selectW = 50.0;
    selectH = 50.0;

    baseClipX0 = 0.0;                     baseClipY0 = 0.0;
    baseClipX1 = float(myImage.width());  baseClipY1 = float(myImage.height());
    UpdateSizeAndZoom();
    sX =  baseClipX0 + 0.5*( baseClipX1 - baseClipX0 ) - 0.5*selectW;
    sY =  baseClipY0 + 0.5*( baseClipY1 - baseClipY0 ) - 0.5*selectH;
    CorrectSelect();
    selectBox = QRect( ScreenX( sX ), ScreenY( sY ), int( selectW*zoom + 0.5 ), int( selectH*zoom + 0.5 ) );
    emit selectionChanged( sX, sY );

    clipX0 = baseClipX0; clipY0 = baseClipY0;
    clipX1 = baseClipX1; clipY1 = baseClipY1;
    selectMoveMode = false;

 }

SelectField::~SelectField( void ) {
}


void SelectField::setTheImage( const QImage & newImage ) {
    if (newImage != myImage) {
        myImage = newImage;
        baseClipX0 = 0.0;                     baseClipY0 = 0.0;
        baseClipX1 = float(myImage.width());  baseClipY1 = float(myImage.height());
        UpdateSizeAndZoom();
        UpdateScreenImage();
        float nX,nY;
        nX =  baseClipX0 + 0.5*( baseClipX1 - baseClipX0 ) - 0.5*selectW;
        nY =  baseClipY0 + 0.5*( baseClipY1 - baseClipY0 ) - 0.5*selectH;
        moveSelection( nX, nY );
        emit selectionChanged( sX, sY );

        clipX0 = baseClipX0; clipY0 = baseClipY0;
        clipX1 = baseClipX1; clipY1 = baseClipY1;
        update();
        updateGeometry();
    }
 }


QSize SelectField::sizeHint( void ) const {
    return mySize;
}


QSize SelectField::minimumSizeHint( void ) const {
    return mySize;
}

void SelectField::setSelectSize( float sW, float sH ) {
    if( sW != selectW || sH != selectH ) {
        // move box that center stays unchanged
        float xCenter = sX + 0.5*selectW;
        float yCenter = sY + 0.5*selectH;
        sX = xCenter - 0.5*sW;
        sY = yCenter - 0.5*sH;

        selectW = sW;
        selectH = sH;
        CorrectSelect();   // have we left the bounds? move us back
        UpdateSelectBox();
        emit selectionChanged( sX, sY );
    }
}

 // try to move selectBox within image
void SelectField::moveSelection ( float nX, float nY ) {
    if( nX != sX || nY != sY ) {
        float xOld = sX, yOld = sY;
        sX = nX;
        sY = nY;
        CorrectSelect();   // have we left the bounds? move us back
        if( sX != xOld || sY != yOld ) {  // real change ?
            UpdateSelectBox();
            emit selectionChanged( sX, sY );
        }
    }
}

// set the baseClipRect, which selects the used and displayed part of the source
void SelectField::setClipRect( float cx0, float cy0, float cx1, float cy1 ) {
    baseClipX0 = cx0;    baseClipY0 = cy0;
    baseClipX1 = cx1;    baseClipY1 = cy1;
    clipX0 = cx0;    clipY0 = cy0;
    clipX1 = cx1;    clipY1 = cy1;
    UpdateSizeAndZoom();
    UpdateScreenImage();
    float xOld = sX, yOld = sY;
    CorrectSelect();   // have we left the bounds? move us back
    if( sX != xOld || sY != yOld ) {  // real change ?
        emit selectionChanged( sX, sY );
    }
    UpdateSelectBox();
    updateGeometry();
    update();
}

void SelectField::mousePressEvent( QMouseEvent *event ) {
    if( event->button() == Qt::LeftButton ) {
        this->setFocus(); // only a simple means to remove focus from textedits ( and thus allow image paste in the EnlargerDialog )
        int px  = event->pos().x(),        py = event->pos().y();
        int sx0 = selectBox.x(),           sy0 = selectBox.y();
        int sx1 = sx0 + selectBox.width(), sy1 = sy0 + selectBox.height();
        const int bR = selectBorderWidth/2;

        if( ( px >= sx0-bR && px <= sx1+bR && py >= sy0-bR && py <= sy1+bR ) &&
            !( px >  sx0+bR && px <  sx1-bR && py >  sy0+bR && py <  sy1-bR )   ) {
            // start moving select-box
            selectMoveMode = true;
            grabX = float( event->pos().x() - selectBox.x() );
            grabY = float( event->pos().y() - selectBox.y() );
            grabX /= zoom; grabY /= zoom;
        }
        else {   // start changing clip-box
            clipX0 = SrcX( event->pos().x() );
            clipY0 = SrcY( event->pos().y() );
            clipX1 = clipX0;
            clipY1 = clipY0;
            update();
        }
    }
}


void SelectField::mouseMoveEvent( QMouseEvent *event ) {
    if( event->buttons() & Qt::LeftButton ) {
        if( selectMoveMode ) { // move select-box
            float newx,newy;
            newx = SrcX( event->pos().x() ) - grabX;
            newy = SrcY( event->pos().y() ) - grabY;
            moveSelection( newx, newy );
        }
        else { // change clip-box
            float newx,newy;
            newx = SrcX( event->pos().x() );
            newy = SrcY( event->pos().y() );
            if( newx < baseClipX0 )
                newx = baseClipX0;
            else if( newx > baseClipX1 )
                newx = baseClipX1;
            if( newy < baseClipY0 )
                newy = baseClipY0;
            else if( newy > baseClipY1 )
                newy = baseClipY1;
            if( newx<clipX0 )
                clipX0 = newx;
            if( newy<clipY0 )
                clipY0 = newy;
            clipX1 = newx;
            clipY1 = newy;
            update();

        }
    }
}

void SelectField::mouseReleaseEvent( QMouseEvent *event ) {
    if( event->button() == Qt::LeftButton ) {   // finish moving select-box
        if( selectMoveMode ) {
            selectMoveMode = false;
        }
        else {  // finish changing clip-box
            if( clipX0<clipX1 && clipY0<clipY1 )
                emit clippingChanged( clipX0, clipY0, clipX1, clipY1 );
            else {
                clipX0 = baseClipX0; clipY0 = baseClipY0;
                clipX1 = baseClipX1; clipY1 = baseClipY1;
            }
            update();
        }
    }
}

void SelectField::paintEvent( QPaintEvent *event ) {
    QPainter painter(this);
    painter.setClipping( true );
    painter.setClipRect( 0, 0, mySize.width(), mySize.height() );

    painter.drawImage( 0, 0, screenImage );

    // draw clip-rect
    int cx0 = ScreenX( clipX0 ), cy0 = ScreenY( clipY0 );
    int cx1 = ScreenX( clipX1 ), cy1 = ScreenY( clipY1 );
    painter.fillRect( 0,   0,   mySize.width(),     cy0,                      QColor(0,0,0,100) );
    painter.fillRect( 0,   cy0, cx0,                cy1-cy0,                  QColor(0,0,0,100) );
    painter.fillRect( cx1, cy0, mySize.width()-cx1, cy1-cy0,                  QColor(0,0,0,100) );
    painter.fillRect( 0,   cy1, mySize.width(),     mySize.height() - cy1,    QColor(0,0,0,100) );

    QPen rectPen( QColor( 0, 0, 0, 100 ) );
    rectPen.setWidth( selectBorderWidth );
    rectPen.setJoinStyle( Qt::MiterJoin );
    painter.setPen( rectPen );
    painter.drawRect( selectBox );

    rectPen.setColor( QColor(255,255,255, 100 ) );
    rectPen.setWidth( 0 );
    painter.setPen( rectPen );
    painter.drawRect( selectBox );
}

// ---- ------------- ----

void SelectField::UpdateSizeAndZoom( void ) {

    float newW,newH;
    zoom = MAXHEIGHT / ( baseClipY1 - baseClipY0 );
    newH = MAXHEIGHT;
    newW = ( baseClipX1 - baseClipX0 ) * zoom;
    if( newW > MAXWIDTH ) {
        zoom = zoom * MAXWIDTH / newW;
        newW = MAXWIDTH;
        newH = ( baseClipY1 - baseClipY0 ) * zoom;
    }
    mySize = QSize( int( newW ), int( newH ) );
}

void SelectField::CorrectSelect( void ) {
    if( sX > baseClipX1 - selectW )
        sX = baseClipX1 - selectW ;
    if( sY > baseClipY1 - selectH )
        sY = baseClipY1 - selectH ;
    if( sX < baseClipX0 )
        sX = baseClipX0;
    if( sY < baseClipY0 )
        sY = baseClipY0;
}

void SelectField::UpdateSelectBox( void ) {
    const long bW = selectBorderWidth + 1;
    QRect oldBox = selectBox;
    selectBox.moveTo( ScreenX( sX ), ScreenY( sY ) );
    selectBox.setWidth ( int( selectW*zoom ) );
    selectBox.setHeight( int( selectH*zoom ) );

    update( oldBox.x()-bW/2, oldBox.y()-bW/2, oldBox.width()+1+bW, oldBox.height()+1+bW );
    update( selectBox.x()-bW/2, selectBox.y()-bW/2, selectBox.width()+1+bW, selectBox.height()+1+bW );
}

// instead of drawing the source image into the field, which may be slow for big sizes,
// use a screenImage, into which the needed clip of the source is drawn once at every clipping-change
void SelectField::UpdateScreenImage( void ) {
    screenImage = QImage( mySize, QImage::Format_ARGB32 );
    QPainter painter( &screenImage );
    painter.setRenderHint(QPainter::Antialiasing, true );

    int sx = int( SrcX(0) )-1,   sy = int( SrcY(0) )-1;
    int sw = int( baseClipX1 - baseClipX0 ) + 3;
    int sh = int( baseClipY1 - baseClipY0 ) + 3;

    int tx = int( ( float(sx) - SrcX(0) )*zoom );
    int ty = int( ( float(sy) - SrcY(0) )*zoom );
    int tw = int( float(sw)*zoom );
    int th = int( float(sh)*zoom );
    QRect sourceR, targetR;
    sourceR = QRect( sx, sy, sw, sh );
    targetR = QRect( tx, ty, tw, th );
    painter.drawImage( targetR,  myImage, sourceR );

}

