/* ----------------------------------------------------------------

SmillaEnlarger  -  resize, especially magnify bitmaps in high quality
    EnlargerDialog.cpp: the Qt Dialog

Copyright (C) 2009 Mischa Lusteck

This program is free software;
you can redistribute it and/or modify it under the terms of the
GNU General Public License as published by the Free Software Foundation;
either version 3 of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY;
without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, see <http://www.gnu.org/licenses/>.

---------------------------------------------------------------------- */

#include <QFileDialog>
#include <QDesktopServices>    // for getting std. picture-directory
#include <QDir>
#include <QSettings>
#include <QtGui>

#include <iostream>
#include "EnlargerDialog.h"
#include "ui_enlargerdialog.h"
#include "previewField.h"
#include "EnlargerThread.h"
#include "ImageEnlargerCode/FractTab.h"
#include "CalcQueue.h"

const int historyClipMax = 100;
using namespace std;

EnlargerDialog::EnlargerDialog(QWidget *parent)
    : QMainWindow(parent), ui(new Ui::EnlargerDialog),
      previewThread( new EnlargerThread )
{
    setUpdatesEnabled( false );
    ui->setupUi(this);
    MenuBarSetup();
    setCentralWidget( ui->centralwidget );
    zoomFact = 1.0;
    aspectX = aspectY = 1.0;
    freeAspectRatio = false;
    zoomInt = 100;
    zoomUpdateMod = useZoom;
    ui->zoomRButton->setChecked( true );

    // avoid indirect self-calls of setZoom via diverse value-changes
    // recursionBlocker set true at begin of SetZoom, false at end
    setZoom_recursionBlocker = false;

    // create the calculation queue
    theCalcQueue = new CalcQueue();
    ui->queueListView->setModel( theCalcQueue->DisplayModel() );

    // set up clipping history
    historyClipIdx = 0;
    historyClipX0 = new float[ historyClipMax + 1 ];
    historyClipY0 = new float[ historyClipMax + 1 ];
    historyClipX1 = new float[ historyClipMax + 1 ];
    historyClipY1 = new float[ historyClipMax + 1 ];

    QString picDir = QDesktopServices::storageLocation ( QDesktopServices::PicturesLocation );
    srcDir.setPath( picDir );
    SetDestDir( picDir );

    ui->sourceDirLabel->setText( " < none > ");
    ui->destFileEdit->setText( "enlarged.jpg" );
    ui->srcCheckBox->setChecked( true );
    ui->load2Button->setEnabled( false );

    connect( ui->load1Button,   SIGNAL(clicked()), this, SLOT(slot_load1()) );
    connect( ui->load2Button,   SIGNAL(clicked()), this, SLOT(slot_load2()) );
    connect( ui->comboBox,      SIGNAL(activated ( const QString & )),
             this, SLOT(slot_comboChanged( const QString & )) );
    connect( ui->srcCheckBox,   SIGNAL(stateChanged( int ) ), this, SLOT(slot_checkBoxChanged( int ) ) );
    connect( ui->previewButton, SIGNAL(clicked()), this, SLOT(DoPreview()) );

    connect( ui->calcButton,    SIGNAL(clicked()), this, SLOT(slot_queueCalc()) );

    connect( ui->previewField,   SIGNAL(selectionChanged(float,float)),
             ui->selectField,  SLOT(moveSelection( float, float ) ) );
    connect( ui->selectField,   SIGNAL(selectionChanged(float,float)),
             ui->previewField,  SLOT(moveSelection( float, float ) ) );
    connect( ui->selectField,   SIGNAL(selectionChanged(float,float)),
             this,  SLOT(stopPreview() ) );
    connect( ui->selectField,   SIGNAL(clippingChanged( float, float, float, float )),
             this,  SLOT(slot_setClipping( float, float, float, float ) ) );
    connect( ui->clipResetButton, SIGNAL(clicked()), this, SLOT(slot_resetClipping()) );

    //--------- ToolTips ---------------------------------
    QString tipStr;

    tipStr =  QString(" All you see here will be enlarged. \n");
    tipStr += QString(" With the left mouse-button you can select a new \n");
    tipStr += QString(" cropping rectangle or grab the frame of the preview-square ");
    SetToolTip( ui->sourceTabWidget, ui->selectField, tipStr );

    tipStr =  QString(" Preview - shows a ( small ) part of the result \n");
    tipStr += QString(" in the desired magnification. \n" );
    tipStr += QString(" Move it with left mouse-button ");
    SetToolTip( ui->previewField, tipStr );

    //--------- End of ToolTips ---------------------------------

    // calc & preview Threads
    connect( theCalcQueue,      SIGNAL( tellProgress(int) ), ui->progressBar, SLOT( setValue(int) ) );
    connect( theCalcQueue,      SIGNAL(tellJobCount(int,int)), this, SLOT(slot_ShowJobCount(int,int)) );

    connect( previewThread,        SIGNAL( tellProgress(int) ), this, SLOT( slot_showPreviewProgress(int) ) );
    connect( previewThread,        SIGNAL( enlargedImage(QImage) ), this, SLOT(slot_showPreview(QImage)) );

    // Sliders & Boxes
    const int zoomMin     = 1,   zoomMax      = 3000;
    const int sharpMin    = 0,   sharpMax     = 100;
    const int flatMin     = 0,   flatMax      = 100;
    const int ditherMin   = 0,   ditherMax    = 100;
    const int deNoiseMin  = 0,   deNoiseMax   = 100;
    const int preSharpMin = 0,   preSharpMax  = 100;
    const int fractNoiseMin=0,   fractNoiseMax= 100;
    ui->newWidthBox ->setRange( 1, 100000 );
    ui->newHeightBox->setRange( 1, 100000 );
    ui->zoomSpinBox->setRange( zoomMin, zoomMax );
    ui->zoomSlider ->setRange( zoomMin, zoomMax );
    ui->zoomSpinBox->setSuffix( "%" );
    connect( ui->zoomSlider,  SIGNAL(valueChanged(int)), ui->zoomSpinBox, SLOT(setValue(int)) );
    connect( ui->zoomSpinBox, SIGNAL(valueChanged(int)), ui->zoomSlider,  SLOT(setValue(int)) );
    connect( ui->zoomSpinBox, SIGNAL(valueChanged(int)), this,  SLOT(SetZoomInt(int)) );
    connect( ui->newWidthBox, SIGNAL(valueChanged(int)), this,  SLOT(SetNewWidth(int)) );
    connect( ui->newHeightBox, SIGNAL(valueChanged(int)), this,  SLOT(SetNewHeight(int)) );
    connect( ui->zoomRButton,   SIGNAL(pressed()), this, SLOT(slot_ZoomRadioButton()) );
    connect( ui->widthRButton,  SIGNAL(pressed()), this, SLOT(slot_WidthRadioButton()) );
    connect( ui->heightRButton, SIGNAL(pressed()), this, SLOT(slot_HeightRadioButton()) );
    ui->zoomSpinBox->setValue( 1100 );

    ui->sharpSpinBox->setRange( sharpMin, sharpMax );
    ui->sharpSlider ->setRange( sharpMin, sharpMax );
    connect( ui->sharpSlider,  SIGNAL(valueChanged(int)), ui->sharpSpinBox, SLOT(setValue(int)) );
    connect( ui->sharpSpinBox, SIGNAL(valueChanged(int)), ui->sharpSlider,  SLOT(setValue(int)) );
    ui->sharpSlider->setValue( 80 );

    ui->flatSpinBox->setRange( flatMin, flatMax );
    ui->flatSlider ->setRange( flatMin, flatMax );
    connect( ui->flatSlider,  SIGNAL(valueChanged(int)), ui->flatSpinBox, SLOT(setValue(int)) );
    connect( ui->flatSpinBox, SIGNAL(valueChanged(int)), ui->flatSlider,  SLOT(setValue(int)) );
    ui->flatSlider->setValue( 20 );

    ui->ditherSpinBox->setRange( ditherMin, ditherMax );
    ui->ditherSlider ->setRange( ditherMin, ditherMax );
    connect( ui->ditherSlider,  SIGNAL(valueChanged(int)), ui->ditherSpinBox, SLOT(setValue(int)) );
    connect( ui->ditherSpinBox, SIGNAL(valueChanged(int)), ui->ditherSlider,  SLOT(setValue(int)) );
    ui->ditherSlider->setValue( 20 );

    ui->deNoiseSpinBox->setRange( deNoiseMin, deNoiseMax );
    ui->deNoiseSlider ->setRange( deNoiseMin, deNoiseMax );
    connect( ui->deNoiseSlider,  SIGNAL(valueChanged(int)), ui->deNoiseSpinBox, SLOT(setValue(int)) );
    connect( ui->deNoiseSpinBox, SIGNAL(valueChanged(int)), ui->deNoiseSlider,  SLOT(setValue(int)) );
    ui->deNoiseSlider->setValue( 50 );

    ui->preSharpSpinBox->setRange( preSharpMin, preSharpMax );
    ui->preSharpSlider ->setRange( preSharpMin, preSharpMax );
    connect( ui->preSharpSlider,  SIGNAL(valueChanged(int)), ui->preSharpSpinBox, SLOT(setValue(int)) );
    connect( ui->preSharpSpinBox, SIGNAL(valueChanged(int)), ui->preSharpSlider,  SLOT(setValue(int)) );
    ui->preSharpSlider->setValue( 0 );

    ui->fractNoiseSpinBox->setRange( fractNoiseMin, fractNoiseMax );
    ui->fractNoiseSlider ->setRange( fractNoiseMin, fractNoiseMax );
    connect( ui->fractNoiseSlider,  SIGNAL(valueChanged(int)), ui->fractNoiseSpinBox, SLOT(setValue(int)) );
    connect( ui->fractNoiseSpinBox, SIGNAL(valueChanged(int)), ui->fractNoiseSlider,  SLOT(setValue(int)) );
    ui->fractNoiseSlider->setValue( 0 );

    SetSource( QImage(":/smilla.bmp") );

    ui->progressBar->setRange( 0, 100 );
    ui->progressBar->setValue( 100 );

    ui->comboBox->setSizeAdjustPolicy( QComboBox::AdjustToContentsOnFirstShow );
    ui->comboBox->setMinimumContentsLength( 30 );
    ui->comboBox->clear();
    currentSrcName = " < Smilla > ";
    currentSrcPath = ":/smilla.bmp";

    ui->comboBox->addItem( currentSrcName );

    //----- Parameter Edit -----
    ui->paramCombo->setEditable( true );
    ui->paramCombo->setSizeAdjustPolicy( QComboBox::AdjustToContentsOnFirstShow );
    ui->paramCombo->setMinimumContentsLength( 15 );
    ui->paramCombo->clear();
    ui->paramCombo->setCompleter( 0 );
    ui->paramCombo->setInsertPolicy( QComboBox::InsertAtCurrent );

    connect( ui->paramAddButton,   SIGNAL( clicked() ), this, SLOT(slot_AddParam()) );
    connect( ui->paramDelButton,   SIGNAL( clicked() ), this, SLOT(slot_DelParam()) );
    connect( ui->paramCombo,       SIGNAL( activated(int) ), this, SLOT(slot_SetParamSlidersFromCombo(int)) );
    connect( ui->paramChangeCheckBox, SIGNAL(stateChanged( int ) ), this, SLOT(slot_ParamCheckBoxChanged( int ) ) );

    //----- CalcQueue -----
    connect( ui->clearQueueButton,  SIGNAL( clicked() ), this, SLOT(slot_QueueClear()) );
    connect( ui->removeJobButton,   SIGNAL( clicked() ), this, SLOT(slot_QueueRemoveSelected()) );
    connect( ui->cleanUpQueueButton,SIGNAL( clicked() ), this, SLOT(slot_QueueRemoveEnded()) );

    //help browser
    ui->helpBrowser->setSource( QUrl("qrc:/help/helpMain.html") );
    connect( ui->helpBackButton,    SIGNAL(clicked()), ui->helpBrowser, SLOT(backward()) );
    connect( ui->helpForwardButton, SIGNAL(clicked()), ui->helpBrowser, SLOT(forward()) );
    connect( ui->helpHomeButton,    SIGNAL(clicked()), ui->helpBrowser, SLOT(home()) );

    ResetDialog();

    setAcceptDrops(true);  // Drag and Drop: accept drops ( of source files )

    PrintStatusText(" Welcome to SmillaEnlarger 0.8.9.  Drop a file onto the window to open it." );

    theSettings = 0;    // Pointer to settings, none loaded yet
    ReadSettings();

    SetZoom( 5.0 );
    DoPreview();
    AdjustDestName();   // if dstName exists already: add or inc number at end of name

    resize( minimumSize() );
    setUpdatesEnabled( true );
}

EnlargerDialog::~EnlargerDialog()
{
    delete[] historyClipX0;
    delete[] historyClipY0;
    delete[] historyClipX1;
    delete[] historyClipY1;
    delete previewThread;
    if( theSettings != 0 )
       delete theSettings;
    delete theCalcQueue;
    delete ui;
}

void EnlargerDialog::MenuBarSetup( void ) {
   QMenu *fileMenu, *helpMenu;
   QAction *aboutA, *quitA, *loadA, *saveA, *pasteA;
   QAction *helpShowA, *helpMoreA;

   fileMenu = menuBar()->addMenu(tr("File"));
   helpMenu = menuBar()->addMenu(tr("Help"));

   loadA  = fileMenu->addAction(tr("&Open..."));
   saveA  = fileMenu->addAction(tr("Enlarge && Save"));
   pasteA = fileMenu->addAction(tr("Paste"));

   // miniModeA is EnlargerDialog  class member
   miniModeA  = fileMenu->addAction(tr("Mini Mode"));
   miniModeA->setCheckable( true );
   miniModeA->setChecked( false );

   quitA  = fileMenu->addAction(tr("Quit"));

   aboutA      = helpMenu->addAction(tr("About"));
   helpShowA   = helpMenu->addAction(tr("Show Help"));
   helpMoreA   = helpMenu->addAction(tr("more Help..."));

   loadA->setShortcut( QKeySequence::Open );
   saveA->setShortcut( QKeySequence::Save );
   pasteA->setShortcut( QKeySequence::Paste );

   connect( aboutA, SIGNAL(triggered()), this, SLOT( InfoMessage()) );
   connect( loadA,  SIGNAL(triggered()), this, SLOT( slot_load1())  );
   connect( saveA,  SIGNAL(triggered()), this, SLOT( slot_queueCalc())   );
   connect( pasteA, SIGNAL(triggered()), this, SLOT( slot_paste())   );
   connect( miniModeA, SIGNAL(toggled(bool)), this, SLOT( slot_MiniMode( bool ))   );
   connect( quitA,  SIGNAL(triggered()), this, SLOT( close())        );

   connect( helpShowA,  SIGNAL(triggered()), this, SLOT(slot_ShowHelp()) );
   connect( helpMoreA,   SIGNAL(triggered()), this, SLOT( HelpMessageMore()) );
}

void EnlargerDialog::slot_load1( void ) {
   QString path,body,type;
   QString fileName = QFileDialog::getOpenFileName(this,
       tr("Open File"), srcDir.absolutePath(),
       tr("Image Files (*.bmp *.jpg *.png *.jpeg *.ppm *.tif *.tiff *.gif )") );
   if (!fileName.isEmpty()) {
      TryOpenSource( fileName );
   }
}

void EnlargerDialog::slot_load2( void ) {
    QString path,body,type;
    QString fileName = QFileDialog::getExistingDirectory(this, tr("Open Directory"), dstDir.absolutePath() );
    if (!fileName.isEmpty()) {
        SetDestDir( fileName );
    }
    AdjustDestName();   // if dstName exists already: add or inc number at end of name
}

void EnlargerDialog::slot_comboChanged( const QString & txt ) {
   if( txt == currentSrcName )
       return;
   if( srcDir.exists( txt ) ) {
      TryOpenSource( srcDir.absoluteFilePath( txt ) );
      stopPreview();
   }
}

void EnlargerDialog::slot_checkBoxChanged( int state ) {
    if( ui->srcCheckBox->isChecked() ) {
        ui->load2Button->setEnabled( false );
        SetDestDir( srcDir.absolutePath() );
        AdjustDestName();   // if dstName exists already: add or inc number at end of name
    }
    else {
        ui->load2Button->setEnabled( true );
    }
}

void EnlargerDialog::stopPreview( void ) {
    previewThread->StopEnlarge();
    ui->previewProgressLabel->setText("  " );
}

void EnlargerDialog::DoPreview( void ) {
    if( ZoomX() < 1.0 && ZoomY() < 1.0 )
        return;

    QRect dstRect;
    QImage srcImage;
    QImage preImage;
    EnlargeFormat format;
    EnlargeParamInt enlargeParam;

    dstRect = ui->previewField->DstRect();
    srcImage = ui->previewField->theImage();

    format.srcWidth  = srcImage.width();
    format.srcHeight = srcImage.height();
    format.SetScaleFact( ZoomX(), ZoomY() );
    format.SetDstClip( dstRect.x(), dstRect.y(), dstRect.x() + dstRect.width(), dstRect.y() + dstRect.height() );

    ReadParameters( enlargeParam );
    previewThread->Enlarge( srcImage, format, enlargeParam.FloatParam() );
    previewThread->setPriority( QThread::NormalPriority );

}

void EnlargerDialog::slot_queueCalc( void ) {
   if( dstDir.exists( ui->destFileEdit->text() ) )  {
      if( !FileExistsMessage( ui->destFileEdit->text(), dstDir.dirName() ) )
         return;
   }

   SingleCalcJob *newJob;
   newJob = new SingleCalcJob();

   newJob->srcName = currentSrcName;
   newJob->srcPath = currentSrcPath;
   if( currentSrcPath.isEmpty() ) {   // indicates, that the image itself has to be attached to the job
      newJob->AttachImage( ui->previewField->theImage() );
   }
   newJob->dstName = ui->destFileEdit->text();
   newJob->dstPath = dstDir.absoluteFilePath( ui->destFileEdit->text() );
   newJob->hasClipping = true;
   newJob->srcClipX0 = baseClipX0;
   newJob->srcClipY0 = baseClipY0;
   newJob->srcClipX1 = baseClipX1;
   newJob->srcClipY1 = baseClipY1;

   // give ZoomUpdateMod to job indirectly
   // w<0 && h<0 -> use zoom
   // w>0 && h<0 -> use width
   // w<0 && h>0 -> use height
   // w>0 && h>0 -> use width&height

   newJob->dstWidth  = -1;
   newJob->dstHeight = -1;
   if( zoomUpdateMod == useZoom ) {
      newJob->zoom = zoomFact;
   }
   else if( zoomUpdateMod == useWidth ) {
      newJob->dstWidth = newWidth;
   }
   else if( zoomUpdateMod == useHeight ) {
      newJob->dstHeight = newHeight;
   }
   else if( zoomUpdateMod == useWidthAndHeight ) {
      newJob->dstWidth  = newWidth;
      newJob->dstHeight = newHeight;
   }

   ReadParameters( newJob->param );

   connect( newJob, SIGNAL(StatusMessage(QString)), this, SLOT(slot_MessageToLog(QString)) );
   connect( newJob, SIGNAL(ErrorMessage(QString)), this, SLOT(slot_MessageToLog(QString)) );
   connect( newJob, SIGNAL(BadAlloc(QString)), this, SLOT(BadAllocMessage(QString)) );

   QString statusText;
   statusText =  "Calculating '" + ui->destFileEdit->text() + "' . ";
   statusText += " Source was '" + ui->comboBox->currentText() + "' . ";
   statusText += " Destination directory is '" + ui->destDirLabel->text() + "' . ";
   PrintStatusText( statusText );

   theCalcQueue->ResetProgress();
   theCalcQueue->AddJob( newJob );
   AdjustDestName();
   ui->queueListView->clearFocus();
   ui->queueListView->setFocus();
}

void EnlargerDialog::slot_showPreview( const QImage & result ) {
    ui->previewField->setPreview( result );
}

void EnlargerDialog::slot_showPreviewProgress( int val ) {
   if( val<100 ) {
      ui->previewProgressLabel->setText(" [ " + QString::number( val ) + "% ]   " );
   }
   else {
      ui->previewProgressLabel->setText("  " );
   }
}

void EnlargerDialog::SetZoomInt( int v ) {
    if( zoomInt != v ) {
        if( ui->zoomSlider->hasFocus() || ui->zoomSpinBox->hasFocus() ) {
           ZoomUseZoom();
        }
        SetZoom( float( v )*0.01 );
        stopPreview();
    }
}

void EnlargerDialog::SetNewWidth( int v ) {
   if( setZoom_recursionBlocker )  // called from within SetZoom -> nothing to do
      return;
   if( v != newWidth ) {
      if( zoomUpdateMod == useWidthAndHeight ){
         ChangeAspectRatio( v, newHeight );
      }
      else {
         if( ui->newWidthBox->hasFocus() ) {
            ZoomUseWidth();
         }
         SetZoom( float( v ) / (baseClipX1-baseClipX0) );
      }
      stopPreview();
   }
}

void EnlargerDialog::SetNewHeight( int v ) {
   if( setZoom_recursionBlocker )  // called from within SetZoom -> nothing to do
      return;
   if( v != newHeight ) {
      if( zoomUpdateMod == useWidthAndHeight ){
         ChangeAspectRatio( newWidth, v );
      }
      else {
         if( ui->newHeightBox->hasFocus() ) {
            ZoomUseHeight();
         }
         SetZoom( float( v ) / (baseClipY1-baseClipY0) );
      }
      stopPreview();
   }
}

void EnlargerDialog::slot_ZoomRadioButton( void ) {
   ZoomUseZoom();
   ui->zoomRButton->setChecked( false ); // will be toggled after releasing the button
}

void EnlargerDialog::slot_WidthRadioButton( void ) {
   if( ui->widthRButton->isChecked() && ui->heightRButton->isChecked()) {
      ZoomUseWidth();
   }
   else if( !ui->widthRButton->isChecked() && !ui->heightRButton->isChecked() ) {
      ZoomUseWidth();
   }
   else if( !ui->widthRButton->isChecked() && ui->heightRButton->isChecked() ) {
      ZoomUseWidthAndHeight();
   }
   ui->widthRButton->setChecked( false ); // will be toggled after releasing the button
}

void EnlargerDialog::slot_HeightRadioButton( void ) {
   if( ui->widthRButton->isChecked() && ui->heightRButton->isChecked()) {
      ZoomUseHeight();
   }
   else if( !ui->widthRButton->isChecked() && !ui->heightRButton->isChecked() ) {
      ZoomUseHeight();
   }
   else if( ui->widthRButton->isChecked() && !ui->heightRButton->isChecked() ) {
      ZoomUseWidthAndHeight();
   }
   ui->heightRButton->setChecked( false ); // will be toggled after releasing the button
}

void EnlargerDialog::slot_applyClipping( void ) {
    float cx0, cy0, cx1, cy1;

    ui->selectField->GetMarkedClipRect( cx0, cy0, cx1, cy1 );
    if( cx0 >= cx1 || cy0 >= cy1 )
        return;
    if( cx0 < 0.0 || cy0 < 0.0 )
        return;
    if( cy1 > float( srcHeight ) || cx1 > float( srcWidth ) )
        return;

    ui->selectField ->setClipRect( cx0, cy0, cx1, cy1 );
    ui->previewField->setClipRect( cx0, cy0, cx1, cy1 );
    baseClipX0 = cx0; baseClipY0 = cy0;
    baseClipX1 = cx1; baseClipY1 = cy1;

    UpdateZoom();   // format changed -> maybe zoom changed ( if newWidth or newHeight held const. )
    //CenterSelectBox();
}

void EnlargerDialog::slot_setClipping( float cx0, float cy0, float cx1, float cy1 ) {
    if( cx0 >= cx1 || cy0 >= cy1 )
        return;
    if( cx0 < 0.0 || cy0 < 0.0 )
        return;
    if( cy1 > float( srcHeight ) || cx1 > float( srcWidth ) )
        return;

    ui->selectField ->setClipRect( cx0, cy0, cx1, cy1 );
    ui->previewField->setClipRect( cx0, cy0, cx1, cy1 );

    baseClipX0 = cx0; baseClipY0 = cy0;
    baseClipX1 = cx1; baseClipY1 = cy1;
    ClippingHistoryAdd();

    UpdateZoom();   // format changed -> maybe zoom changed ( if newWidth or newHeight held const. )
    //CenterSelectBox();
}

void EnlargerDialog::slot_resetClipping( void ) {
    ClippingHistoryUndo();
    ui->selectField ->setClipRect( baseClipX0, baseClipY0, baseClipX1, baseClipY1 );
    ui->previewField->setClipRect( baseClipX0, baseClipY0, baseClipX1, baseClipY1 );

    UpdateZoom();   // format changed -> maybe zoom changed ( if newWidth or newHeight held const. )
    //CenterSelectBox();
}

void EnlargerDialog::slot_paste( void )
{
   const QClipboard *clipboard = QApplication::clipboard();
   SourceFromMimeData( clipboard->mimeData() );
}

void EnlargerDialog::slot_ShowJobCount( int endedJobs, int totalJobs ) {
   int tabIndex;
   tabIndex = ui->tabWidgetRight->indexOf( ui->jobTab );
   if( tabIndex != -1 ) {
      ui->tabWidgetRight->setTabText( tabIndex, "Jobs: ( " + QString::number( endedJobs ) + "/" + QString::number( totalJobs ) + " )" );
   }
}

void EnlargerDialog::slot_MessageToLog( const QString & message ) {
   ui->logDisplay->append( message );
   ui->logDisplay->append( " " );
   ui->logDisplay->moveCursor ( QTextCursor::End );
   ui->logDisplay->ensureCursorVisible();
}

void EnlargerDialog::slot_MiniMode( bool switchToMini ) {
//   hide();
   setUpdatesEnabled( false );
   if( switchToMini ) {
      ui->sourceTabWidget->hide();
      ui->tabWidgetTop->hide();
      ui->tabWidgetLeft->hide();
      ui->tabWidgetRight->setParent( ui->centralwidget );
      ui->tabWidgetLeft->removeTab( 0 );
      ui->tabWidgetRight->insertTab( 0, ui->enlargeTab, "Enlarge" );
      ui->tabWidgetRight->setCurrentIndex( 0 );

      // workaround ?!?: force correct resize ?!?
      setUpdatesEnabled( true );
      ui->centralwidget->setParent( 0 );
      setCentralWidget( ui->centralwidget );
      ui->centralwidget->show();
   }
   else {
      ui->tabWidgetRight->removeTab( 0 );
      ui->tabWidgetLeft->insertTab( 0, ui->enlargeTab, "Enlarge"  );
      ui->tabWidgetRight->setCurrentIndex( 2 );
      ui->tabWidgetLeft->setCurrentIndex( 0 );
      ui->tabWidgetLeft->show();
      ui->tabWidgetRight->show();
      ui->sourceTabWidget->show();
      ui->tabWidgetTop->show();
   }
   resize( minimumSize() );
   setUpdatesEnabled( true );
}

void EnlargerDialog::closeEvent(QCloseEvent *event)
{
   WriteSettings();
   if ( theCalcQueue->UnfinishedJobs() == 0 ) {
      theCalcQueue->Clear();
      event->accept();
      return;
   }
   if( AbortCalcMessage( theCalcQueue->UnfinishedJobs() ) ) {
      theCalcQueue->Clear();
      event->accept();
   } else {
      event->ignore();
   }
 }

// ---- Drag and Drop ----

void EnlargerDialog::dragEnterEvent(QDragEnterEvent *event)
 {
 /*
    QStringList formatsList;

    cout<<" ED DragEnter "<<" \n"<<flush;
    formatsList = event->mimeData()->formats();
    for (int i = 0; i < formatsList.size(); i++) {
       cout << "MimeType ("<<i<<"): "<<formatsList.at(i).toLocal8Bit().constData()<<" \n"<<flush;
    }
    if (event->mimeData()->hasFormat("text/uri-list")) {
       cout<<"    text/uri-list "<<" \n"<<flush;
       QList<QUrl>  uriList;
       uriList = event->mimeData()->urls();
       for (int i = 0; i < uriList.size(); i++) {
          cout << "URI ("<<i<<"): "<<uriList.at(i).toString().toLocal8Bit().constData()<<" \n"<<flush;
          cout << "   Path: "<<uriList.at(i).path().toLocal8Bit().constData()<<" \n"<<flush;
       }
    }
    cout << "Text: "<<event->mimeData()->text().toLocal8Bit().constData()<<" \n"<<flush;
    event->acceptProposedAction();

    if( event->mimeData()->hasImage() )
       cout << "Has Image!\n"<<flush;
    else
       cout << "No Image.\n"<<flush;
*/
    event->acceptProposedAction();

}

void EnlargerDialog::dropEvent(QDropEvent *event)
{
   SourceFromMimeData( event->mimeData() );
   event->acceptProposedAction();
}

//-----------------------------------------------------------

void EnlargerDialog::ZoomUseWidth ( void ) {
   zoomUpdateMod  = useWidth;
   ui->zoomRButton->setChecked( false );
   ui->widthRButton->setChecked( true );
   ui->heightRButton->setChecked( false );
   aspectX = aspectY = 1.0;
   UpdateZoom();
}

void EnlargerDialog::ZoomUseHeight( void ) {
   zoomUpdateMod = useHeight;
   ui->zoomRButton->setChecked( false );
   ui->widthRButton->setChecked( false );
   ui->heightRButton->setChecked( true );
   aspectX = aspectY = 1.0;
   UpdateZoom();
}

void EnlargerDialog::ZoomUseWidthAndHeight( void ) {
   zoomUpdateMod = useWidthAndHeight;
   ui->zoomRButton->setChecked( false );
   ui->widthRButton->setChecked( true );
   ui->heightRButton->setChecked( true );
}

void EnlargerDialog::ZoomUseZoom  ( void ) {
   zoomUpdateMod  = useZoom;
   ui->zoomRButton->setChecked( true );
   ui->widthRButton->setChecked( false );
   ui->heightRButton->setChecked( false );
   aspectX = aspectY = 1.0;
   UpdateZoom();
}

void EnlargerDialog::UpdateZoom( void ) {
   float newZoom;

   if( zoomUpdateMod == useWidthAndHeight ) {  //  width and height const
      ChangeAspectRatio( ui->newWidthBox->value(), ui->newHeightBox->value() );
      return;
   }

   if( zoomUpdateMod == useWidth ) {  // newWidth spinbox has focus -> width const
      newZoom = float( ui->newWidthBox->value() ) / (baseClipX1-baseClipX0) ;
   }
   else if( zoomUpdateMod == useHeight ) {  // newWidth spinbox has focus -> width const
      newZoom = float( ui->newHeightBox->value() ) / (baseClipY1-baseClipY0) ;
   }
   else {
      newZoom = float( ui->zoomSpinBox->value() )*0.01 ;
   }
   SetZoom( newZoom );
}

void EnlargerDialog::SetZoom( float zoomF ) {
    if( !setZoom_recursionBlocker ) {
        setZoom_recursionBlocker = true;
        zoomFact = zoomF;
        newWidth  = int( (baseClipX1-baseClipX0)*ZoomX() + 0.5);
        newHeight = int( (baseClipY1-baseClipY0)*ZoomY() + 0.5);
        zoomInt   = int(zoomF*100.0 + 0.5);
        ui->previewField->setZoom( ZoomX(), ZoomY() );
        ui->selectField->setSelectSize( preWidth/ZoomX(), preHeight/ZoomY() );
        ui->newWidthBox->setValue(newWidth);
        ui->newHeightBox->setValue(newHeight);
        ui->zoomSpinBox->setValue(zoomInt);

        setZoom_recursionBlocker = false;
    }
}

void EnlargerDialog::ChangeAspectRatio( int dstWidth, int dstHeight ) {
   float zx, zy , zoomF;
   zx = float( dstWidth ) / (baseClipX1-baseClipX0);  // zoom in X
   zy = float( dstHeight) / (baseClipY1-baseClipY0);  // zoom in Y
   zoomF = sqrt( zx*zy );   // zoomFact
   aspectX = zx/zoomF;
   aspectY = zy/zoomF;
   SetZoom( zoomF );
}

void EnlargerDialog::ResetDialog( void ) {
    ui->previewButton->setFocus();
}

void EnlargerDialog::SetSource(const  QImage & src ) {
    srcWidth  = src.width();
    srcHeight = src.height();
    baseClipX0 = 0.0;               baseClipY0 = 0.0;
    baseClipX1 = float( srcWidth ); baseClipY1 = float( srcHeight );
    ClippingHistoryClear();

//    UpdateZoom();

    ui->selectField->setTheImage( src );
    ui->selectField->setClipRect( baseClipX0, baseClipY0, baseClipX1, baseClipY1 );

    ui->previewField->setTheImage( src );
    ui->previewField->setClipRect( baseClipX0, baseClipY0, baseClipX1, baseClipY1 );
    //newWidth  = int( float( srcWidth  )*zoomFact );
    //newHeight = int( float( srcHeight )*zoomFact );
    //ui->newWidthBox ->setValue( newWidth  );
    //ui->newHeightBox->setValue( newHeight );
    stopPreview();

    UpdateZoom();   // format changed -> maybe zoom changed ( if newWidth or newHeight held const. )
    CenterSelectBox();
}

void EnlargerDialog::CenterSelectBox( void ) {
/*
     // change pos&zoom of selectBox to put it into middle of selectField
    float dx = 0.3*(baseClipX1-baseClipX0), dy = 0.4*(baseClipY1-baseClipY0);
    float newZoom,z2;
    newZoom = preWidth/dx;
    z2 = preHeight/dy;
    if( newZoom<z2 )
        newZoom=z2;
    if( newZoom>30.0 )
        newZoom = 30.0;
    SetZoom( newZoom );
*/

    float nX =  baseClipX0 + 0.5*( baseClipX1 - baseClipX0 ) - 0.5*preWidth/ZoomX();
    float nY =  baseClipY0 + 0.5*( baseClipY1 - baseClipY0 ) - 0.5*preHeight/ZoomY();
    ui->selectField->moveSelection( nX, nY );
}

void EnlargerDialog::FillComboBox( void ) {
    QString path,body,type;
    QStringList filters;
    filters << "*.jpg" << "*.jpeg" << "*.bmp" << "*.png" << "*.tif" << "*.tiff" << "*.gif" << "*.ppm";
    srcDir.setNameFilters(filters);
    srcDir.setSorting ( QDir::Name | QDir::IgnoreCase );
    QStringList entries;
    entries = srcDir.entryList();
    ui->comboBox->clear();
    ui->comboBox->addItems( entries );
}

void EnlargerDialog::ClippingHistoryUndo( void ) {
    if( historyClipIdx > 0 ) {
        historyClipIdx--;
    }
    baseClipX0 = historyClipX0[ historyClipIdx ];
    baseClipY0 = historyClipY0[ historyClipIdx ];
    baseClipX1 = historyClipX1[ historyClipIdx ];
    baseClipY1 = historyClipY1[ historyClipIdx ];
}

void EnlargerDialog::ClippingHistoryAdd( void ) {
    if( historyClipIdx < historyClipMax ) {
        historyClipIdx++;
    }
    historyClipX0[ historyClipIdx ] = baseClipX0;
    historyClipY0[ historyClipIdx ] = baseClipY0;
    historyClipX1[ historyClipIdx ] = baseClipX1;
    historyClipY1[ historyClipIdx ] = baseClipY1;
}

void EnlargerDialog::ClippingHistoryClear( void ) {
    historyClipIdx = 0;
    historyClipX0[ 0 ] = 0.0;               historyClipY0[ 0 ] = 0.0;
    historyClipX1[ 0 ] = float( srcWidth ); historyClipY1[ 0 ] = float( srcHeight );
}

void EnlargerDialog::PrintStatusText( QString statusText ) {
   ui->logDisplay->append( statusText );
   ui->logDisplay->append( " " );
   ui->logDisplay->moveCursor ( QTextCursor::End );
   ui->logDisplay->ensureCursorVisible();

   SetToolTip( ui->statusTextLabel, statusText );
   statusText.truncate( 180 );
   ui->statusTextLabel->setText ( statusText );
}

void EnlargerDialog::slot_ShowHelp( void ) {
   // to show help, deactivate mini mode
   if( miniModeA->isChecked() )
      miniModeA->toggle();
   ui->tabWidgetTop->setCurrentIndex( 1 );
}

void EnlargerDialog::InfoMessage( void ) {
    QMessageBox msgBox;
    QString mainText, infoText;
    msgBox.setTextFormat( Qt::RichText );

    mainText =  "<p><b>SmillaEnlarger version 0.8.9</b></p>";
    infoText = "<p>Copyright (C) 2009 Mischa Lusteck</p>";
    infoText += "<p>Project site: <a href=\"http://sourceforge.net/projects/imageenlarger/\">";
    infoText += "http://sourceforge.net/projects/imageenlarger/</a></p>";

    infoText +=  "<p></p>";
    infoText += "<p>This program is free software; </p>";
    infoText += "<p> you can redistribute it and/or modify it under the terms of the ";
    infoText += " GNU General Public License as published by the Free Software Foundation; ";
    infoText += " either version 3 of the License, or (at your option) any later version. ";
    infoText += "</p>";
    infoText += "<p>This program is distributed in the hope that it will be useful, ";
    infoText += " but WITHOUT ANY WARRANTY; ";
    infoText += " without even the implied warranty of MERCHANTABILITY ";
    infoText += " or FITNESS FOR A PARTICULAR PURPOSE. ";
    infoText += " See the GNU General Public License for more details. ";
    infoText += "</p>";
    infoText += "<p>You should have received a copy of the GNU General Public License ";
    infoText += " along with this program; if not, see ";
    infoText += " <a href=\"http://www.gnu.org/licenses/\">http://www.gnu.org/licenses/</a>. </p>";

    msgBox.setText( mainText );
    msgBox.setInformativeText( infoText );
    msgBox.exec();
}

void EnlargerDialog::HelpMessageMore( void ) {
    QMessageBox msgBox;
    msgBox.setText("<b>More Help:</b>");
    msgBox.setTextFormat( Qt::RichText );
    QString helpText;
    helpText += "<p>Visit the <a href=\"http://sourceforge.net/projects/imageenlarger/\"> ";
    helpText += " Project Page</a> ,</p>";
    helpText += "<p>especially the <a href=\"http://sourceforge.net/forum/forum.php?forum_id=974929\">Support Forum</a>.</p>";
    helpText += "<p>Most interface elemets have tool tips ( they pop up when you don't move the mouse some time ).</p> ";


    msgBox.setInformativeText( helpText );
    msgBox.exec();
}

void EnlargerDialog::BadAllocMessage(  const QString & dstName  ) {
    QMessageBox msgBox;
    msgBox.setTextFormat( Qt::RichText );
    msgBox.setText("<b>Result too big !?</b>");
    QString helpText;
    helpText =  "<p> An error occured:</p>";
    helpText += "<p> SmillaEnlarger was not able to allocate enough memory for '" + dstName + "', calculation was aborted.</p>";

    msgBox.setInformativeText( helpText );
    msgBox.exec();

    ResetDialog();
}

bool EnlargerDialog::FileExistsMessage( const QString & fileName, const QString & dirName ) {
   QMessageBox msgBox;
   msgBox.setText("The file '" + fileName + "' exists. Do you want to replace it? " );
   QString helpText;
   helpText =  "In the directory '" + dirName +"' exists a file with the same name. ";
   helpText += " Click 'OK' to replace it. ";
   msgBox.setInformativeText( helpText );
   msgBox.setStandardButtons( QMessageBox::Ok | QMessageBox::Cancel);
   int ret = msgBox.exec();

   return ret == QMessageBox::Ok;
}

bool EnlargerDialog::AbortCalcMessage( int unfinishedJobs  ) {
   QMessageBox msgBox;
   QString helpText;

   if( unfinishedJobs == 0 )
      return true;

   if( unfinishedJobs == 1 ) {
      msgBox.setText("Do you want to abort the current calculation? " );
      helpText =  "Currently there is a unfinished job in the calculation queue. ";
      helpText += " Click 'Abort' to stop the calculation. ";
   }
   else {
      msgBox.setText("Do you want to abort the current calculations? " );
      helpText =  "Currently there are " + QString::number( unfinishedJobs ) + " unfinished jobs in the calculation queue. ";
      helpText += " Click 'Abort' to stop the calculations. ";
   }

   msgBox.setInformativeText( helpText );
   msgBox.setStandardButtons( QMessageBox::Abort | QMessageBox::Cancel);
   int ret = msgBox.exec();

   return ret == QMessageBox::Abort;
}

void EnlargerDialog::SetToolTip( QWidget *widget, const QString & tipStr ) {
    widget->setToolTip(   tipStr );
    widget->setWhatsThis( tipStr );
}

void EnlargerDialog::SetToolTip( QWidget *w1,  QWidget *w2, const QString & tipStr ) {
    w1->setToolTip(   tipStr );
    w1->setWhatsThis( tipStr );
    w2->setToolTip(   tipStr );
    w2->setWhatsThis( tipStr );
}

void EnlargerDialog::slot_QueueRemoveSelected( void ) {
   QModelIndex jobIdx = ui->queueListView->currentIndex();
   if( !jobIdx.isValid() )
      return;
   if( !ui->queueListView->selectionModel()->isSelected( jobIdx ) )
      return;
   theCalcQueue->RemoveJob( jobIdx );
   ui->queueListView->clearFocus();
   ui->queueListView->setFocus();
}

void EnlargerDialog::slot_QueueClear( void ) {
   theCalcQueue->Clear();
   ui->queueListView->clearFocus();
   ui->queueListView->setFocus();
}

void EnlargerDialog::slot_QueueRemoveEnded( void ) {
   theCalcQueue->RemoveEnded();
   ui->queueListView->clearFocus();
   ui->queueListView->setFocus();
}

