/*
 * Copyright (C) 2005-2006 Ruslan Popov <radz@yandex.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <math.h>

#define MIDDLE 159

#include "crtbrt.h"

// Widgets' handlers
static void on_bReset_clicked( GtkButton *button, gpointer p_this )
{
  ( ( CrtBrtPlugin * ) p_this )->ResetValues();
  GetSelectedFramesForFX().Repaint();
}

static void on_hsBrightness_changed( GtkRange *range, gpointer p_this )
{
  ( ( CrtBrtPlugin * ) p_this )->Scale2Spin( "hscaleBrightness", "sbBrightness" );
  GetSelectedFramesForFX().Repaint();
}

static void on_sbBrightness_changed( GtkRange *range, gpointer p_this )
{
  ( ( CrtBrtPlugin * ) p_this )->Spin2Scale( "sbBrightness", "hscaleBrightness" );
  GetSelectedFramesForFX().Repaint();
}

static void on_hsContrast_changed( GtkRange *range, gpointer p_this )
{
  ( ( CrtBrtPlugin * ) p_this )->Scale2Spin( "hscaleContrast", "sbContrast" );
  GetSelectedFramesForFX().Repaint();
}

static void on_sbContrast_changed( GtkRange *range, gpointer p_this )
{
  ( ( CrtBrtPlugin * ) p_this )->Spin2Scale( "sbContrast", "hscaleContrast" );
  GetSelectedFramesForFX().Repaint();
}

// Constructor
CrtBrtPlugin::CrtBrtPlugin()
{
  // create a widget for plugin
  glade = glade_xml_new( DATADIR "/" PACKAGE "/brightcontrast.glade", NULL, NULL );
  plugin_window = glade_xml_get_widget( glade, "pluginBrtCrt" );
  // connect widgets' handlers
  GtkWidget* widget;
  widget = glade_xml_get_widget( glade, "bReset" );
  if ( widget ) {
    g_signal_connect( G_OBJECT( widget ), "clicked", 
		      G_CALLBACK( on_bReset_clicked ), this );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "hscaleBrightness" );
  if ( widget ) {
    g_signal_connect( G_OBJECT( widget ), "value-changed", 
		      G_CALLBACK( on_hsBrightness_changed ), this );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "sbBrightness" );
  if ( widget ) {
    g_signal_connect( G_OBJECT( widget ), "value-changed", 
		      G_CALLBACK( on_sbBrightness_changed ), this );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "hscaleContrast" );
  if ( widget ) {
    g_signal_connect( G_OBJECT( widget ), "value-changed", 
		      G_CALLBACK( on_hsContrast_changed ), this );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "sbContrast" );
  if ( widget ) {
    g_signal_connect( G_OBJECT( widget ), "value-changed", 
		      G_CALLBACK( on_sbContrast_changed ), this );
    widget = NULL;
  }
}

// Destructor
CrtBrtPlugin::~CrtBrtPlugin()
{
  // remove our widget
  gtk_widget_destroy( plugin_window );
}

char * CrtBrtPlugin::GetDescription( ) const
{
  return "Brightness & Contrast";
}

/** Here we filter our frame. This function converts frame pixels
    (represented here as array) using tranform table. */
void CrtBrtPlugin::FilterFrame( uint8_t *pixels, int width, int height,
				    double position, double delta_frame )
{
  uint8_t * length = pixels + ( width * height * 3 );
  if ( 0 != bright || 0 != contrast )
    {
      while ( length >= pixels )
	{
	  *length = transform[*length];
	  --length;
	}
    }
}

void CrtBrtPlugin::AttachWidgets( GtkBin *bin )
{
  gtk_widget_reparent( ( GTK_BIN( plugin_window ) )->child, GTK_WIDGET( bin ) );
}

void CrtBrtPlugin::DetachWidgets( GtkBin *bin )
{
  gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( plugin_window ) );
}

/** This function is called after user is pressed on 'Preview' or
    'Render' button. This is useful place to make some initializing
    thing if it needs. */
void CrtBrtPlugin::InterpretWidgets( GtkBin *bin )
{
  GtkRange * range;
  GtkAdjustment * adjust;
  // get the brightness value from GUI control
  range = GTK_RANGE( glade_xml_get_widget( glade, "hscaleBrightness" ) );
  adjust = gtk_range_get_adjustment( range );
  bright = ( int ) gtk_adjustment_get_value( adjust );
  // get the contrast value from GUI control
  range = GTK_RANGE( glade_xml_get_widget( glade, "hscaleContrast" ) );
  adjust = gtk_range_get_adjustment( range );
  contrast = ( int ) gtk_adjustment_get_value( adjust );
  // build table
  buildTransform( bright, contrast );
}

/** Building of a transform table. */
void CrtBrtPlugin::buildTransform( char br_value, char ct_value )
{
  // brightness part
  for ( int i=255; i>=0; --i )
    {
      int value = i + br_value;
      if ( 255 < value ) { transform[i] = 255; continue; }
      if ( 0 > value )   { transform[i] = 0; continue; }
      transform[i] = value;
    }
  // contrast part
  int offset, limit_b, limit_t, idx_b, idx_t;
  double col, step;
  for ( int i=255; i>=0; --i )
    {
      if ( 0 > ct_value )
	{
	  if ( MIDDLE > transform[i] )
	    {
	      offset = ( MIDDLE - transform[i] ) * ct_value / 128;
	      if ( MIDDLE < transform[i] - offset )
		transform[i] = MIDDLE;
	      else
		transform[i] -= offset;
	    }
	  else
	    {
	      offset = ( transform[i] - MIDDLE ) * ct_value / 128;
	      if ( MIDDLE > transform[i] + offset )
		transform[i] = MIDDLE;
	      else
		transform[i] += offset;
	    }
	}
      else
	{
	  limit_b = ct_value * MIDDLE / 128;
	  for ( idx_b=0; idx_b<256; ++idx_b )
	    {
	      if ( limit_b > transform[idx_b] )
		transform[idx_b] = 0;
	      else
		break;
	    }
	  limit_t = ct_value * 128 / MIDDLE;
	  for ( idx_t=0; idx_t<256; ++idx_t )
	    {
	      if ( 255 < limit_t + transform[idx_t] )
		transform[idx_t] = 255;
	      else
		break;
	    }
	  step = 256.0 / ( ( double ) ( 256 - limit_b + limit_t ) );
	  col = 0.0;
	  for ( int i=idx_b; i<idx_t; ++i )
	    {
	      if ( transform[i] >= limit_b || transform[i] < 256 - limit_t )
		{
		  col = ( double ) ( transform[i] - limit_b ) * step + 0.5;
		  transform[i] = ( col > 255.0 ) ? 255 : ( uint8_t ) col;
		}
	    }
	}
    }
}

void CrtBrtPlugin::ResetValues( void ) const
{
  GtkWidget * widget = NULL;
  // brightness
  widget = glade_xml_get_widget( glade, "hscaleBrightness" );
  if ( widget ) {
    gtk_adjustment_set_value( gtk_range_get_adjustment( GTK_RANGE( widget ) ), 0.0 );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "sbBrightness" );
  if ( widget ) {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), 0.0 );
    widget = NULL;
  }
  // constrast
  widget = glade_xml_get_widget( glade, "hscaleContrast" );
  if ( widget ) {
    gtk_adjustment_set_value( gtk_range_get_adjustment( GTK_RANGE( widget ) ), 0.0 );
    widget = NULL;
  }
  widget = glade_xml_get_widget( glade, "sbContrast" );
  if ( widget ) {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( widget ), 0.0 );
    widget = NULL;
  }
}

void CrtBrtPlugin::Spin2Scale( const gchar * spinname, const gchar * scalename ) const
{
  // get spin value and set it to scale
  GtkWidget * scale = glade_xml_get_widget( glade, scalename );
  GtkWidget * spin = glade_xml_get_widget( glade, spinname );
  if ( scale && spin ) {
    gtk_adjustment_set_value( gtk_range_get_adjustment( GTK_RANGE( scale ) ), 
			      gtk_spin_button_get_value( GTK_SPIN_BUTTON( spin ) ) );
  }
}

void CrtBrtPlugin::Scale2Spin( const gchar * scalename, const gchar * spinname ) const
{
  // get scale value and set it to spin
  GtkWidget * scale = glade_xml_get_widget( glade, scalename );
  GtkWidget * spin = glade_xml_get_widget( glade, spinname );
  if ( scale && spin ) {
    gtk_spin_button_set_value( GTK_SPIN_BUTTON( spin ),
			       gtk_adjustment_get_value( gtk_range_get_adjustment( GTK_RANGE( scale ) ) ) );
  }
}

// Export filter

extern "C" {

  GDKImageFilter *GetImageFilter( int index )
  {
    switch( index )
      {
      case 0: 
	return new CrtBrtPlugin();
      }
    return NULL;
  }
  
}

