% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/surprisals_models.R
\name{surprisals.lm}
\alias{surprisals.lm}
\alias{surprisals_prob.lm}
\alias{surprisals.gam}
\alias{surprisals_prob.gam}
\title{Surprisals and surprisal probabilities computed from a model}
\usage{
\method{surprisals}{lm}(object, loo = FALSE, ...)

\method{surprisals_prob}{lm}(
  object,
  approximation = c("none", "gpd", "rank"),
  threshold_probability = 0.1,
  loo = FALSE,
  ...
)

\method{surprisals}{gam}(object, ...)

\method{surprisals_prob}{gam}(
  object,
  approximation = c("none", "gpd", "rank"),
  threshold_probability = 0.1,
  ...
)
}
\arguments{
\item{object}{A model object such as returned by \code{\link[stats]{lm}},
\code{\link[stats]{glm}}, or \code{\link[mgcv]{gam}}.
This includes a specified conditional probability
distribution which is used to compute surprisal values.}

\item{loo}{Should leave-one-out surprisals be computed? For computational
reasons, this is only available for \code{lm} objects.}

\item{...}{Other arguments are ignored.}

\item{approximation}{Character string specifying the method to use in
computing the surprisal probabilities. See Details below.}

\item{threshold_probability}{Probability threshold when computing the GPD
approximation. This is the probability below which the GPD is fitted. Only
used if \code{approximation = "gpd"}.}
}
\value{
A numerical vector containing the surprisals or surprisal probabilities.
}
\description{
A surprisal is given by \eqn{s = -\log f(y)} where \eqn{f} is the
density or probability mass function of the estimated or assumed distribution,
and \eqn{y} is an observation. This is returned by \code{surprisals()}.
A surprisal probability is the probability of a surprisal at least as extreme
as \eqn{s}. This is returned by \code{surprisals_prob()}
}
\details{
The surprisal probabilities may be computed in three different ways.
\enumerate{
\item When \code{approximation = "none"} (the default), the surprisal probabilities are computed
using the same distribution that was used to compute the surprisal values.
Under this option, surprisal probabilities are equal to 1 minus the
coverage probability of the largest HDR that contains each value. Surprisal
probabilities smaller than 1e-6 are returned as 1e-6.
\item When \code{approximation = "gdp"}, the surprisal probabilities are
computed using a Generalized Pareto Distribution fitted to the most extreme
surprisal values (those with probability less than \code{threshold_probability}).
For surprisal probabilities greater than \code{threshold_probability}, the value of
\code{threshold_probability} is returned. Under this option, the distribution is
used for computing the surprisal values but not for determining their
probabilities. Due to extreme value theory, the resulting probabilities should
be relatively insensitive to the distribution used in computing the surprisal
values.
\item When \code{approximation = "rank"}, the surprisal probability of each
observation is estimated using the proportion of observations with
greater surprisal values; i.e., 1 - rank(s)/n where \code{rank(s)} is the rank
of the surprisal value \code{s} among all surprisal values, and \code{n} is the
number of observations. This is a nonparametric approach that is also
insensitive to the distribution used in computing the surprisal values.
}
}
\examples{
# A linear model (i.e., a conditional Gaussian distribution)
lm_of <- lm(waiting ~ duration, data = oldfaithful)
oldfaithful |>
  mutate(
    fscore = surprisals_prob(lm_of),
    prob = surprisals_prob(lm_of, loo = TRUE),
  ) |>
  ggplot(aes(
    x = duration, y = waiting,
    color = prob < 0.01
  )) +
  geom_point()
# A Poisson GLM
glm_breaks <- glm(breaks ~ wool + tension, data = warpbreaks, family = poisson)
warpbreaks |>
  mutate(prob = surprisals_prob(glm_breaks)) |>
  filter(prob < 0.05)
}
\references{
Rob J Hyndman (2026) "That's weird: Anomaly detection using R", Chapter 6,
\url{https://OTexts.com/weird/}.
}
\seealso{
For specific methods, see \code{\link[=surprisals.numeric]{surprisals.numeric()}} and \code{\link[=surprisals.lm]{surprisals.lm()}},
}
\author{
Rob J Hyndman
}
