% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/optimize_gps.R
\name{select_opt}
\alias{select_opt}
\title{Select Optimal Parameter Combinations from Optimization Results}
\usage{
select_opt(
  x,
  smd_groups = NULL,
  smd_variables = NULL,
  smd_type = c("mean", "max"),
  perc_matched = NULL
)
}
\arguments{
\item{x}{An object of class \code{best_opt_result}, produced by the
\code{optimize_gps()} function.}

\item{smd_groups}{A \code{list} of pairwise comparisons (as \code{character} vectors of
length 2) specifying which treatment group comparisons should be
prioritized in SMD evaluation. Each element must be a valid pair of
treatment levels. If \code{NULL}, all pairwise comparisons are used. Example:
\code{list(c("adenoma", "crc_malignant"), c("controls", "adenoma"))}}

\item{smd_variables}{A \code{character} vector of covariate names to include in
the SMD evaluation. Must match variables listed in \code{attr(x, "model_covs")}.}

\item{smd_type}{A \code{character} string (\code{"mean"} or \code{"max"}), defining how to
aggregate SMDs across covariates and comparisons. \code{"max"} selects
combinations with the lowest maximum SMD; \code{"mean"} uses the average SMD.}

\item{perc_matched}{A \code{character} vector of treatment levels for which the
matching rate should be maximized. If \code{NULL}, overall \code{perc_matched} is
used. If specified, only the sum of matching percentages for the listed
groups is used for selection within each SMD category.}
}
\value{
An S3 object of class \code{select_result}, containing the filtered and
prioritized optimization results. The object includes:
\itemize{
\item A \code{data.frame} with selected parameter combinations and performance
metrics.
\item \strong{Attribute \code{param_df}}: A \code{data.frame} with full parameter specifications
(\code{iter_ID}, GPS/matching parameters, etc.), useful for manually refitting or
reproducing results.
}

The object also includes a custom \code{print()} method that summarizes:
\itemize{
\item Number of selected combinations per SMD bin
\item Corresponding aggregated SMD (mean or max)
\item Overall or group-specific percentage matched
}
}
\description{
\code{select_opt()} is a helper function to filter and prioritize
results from \code{optimize_gps()} based on the specific goals of a study.
Depending on the research design, certain pairwise comparisons or treatment
groups may be more important than others. For example:
\itemize{
\item You may want to prioritize matching between a specific groups (e.g.
specific disease vs. controls), while ignoring other group comparisons
during SMD evaluation.
\item You may wish to retain as many samples as possible from a critical group or
set of groups, regardless of matching rates in other groups.
}

This function enables targeted selection of optimal parameter combinations
by:
\itemize{
\item Evaluating SMDs for specific pairwise treatment group comparisons,
\item Selecting key covariates to assess balance,
\item Prioritizing matched sample size in selected treatment groups.
}

By combining these criteria, \code{select_opt()} allows you to tailor the
optimization output to your study's focus - whether it emphasizes covariate
balance in targeted group comparisons or maximizing sample retention for
specific subgroups.
}
\details{
Optimization results are grouped into bins based on the
\strong{maximum SMD} observed for each parameter combination. These bins follow
the same structure as in \code{optimize_gps()}:
\itemize{
\item 0.00-0.05
\item 0.05-0.10
\item 0.10-0.15
\item 0.15-0.20
\item 0.20-0.25
\item 0.25-0.30
\item 0.30-0.35
\item 0.35-0.40
\item 0.40-0.45
\item 0.45-0.50
\item more than 0.50
}

Within each bin, models are first filtered based on their aggregated SMD
across the specified \code{smd_groups} and \code{smd_variables}, using the method
defined by \code{smd_type}. Then, among the remaining models, the best-performing
one(s) are selected based on the percentage of matched samples - either
overall or in the specified treatment groups (\code{perc_matched}).
}
\examples{
\donttest{
# Define formula and set up optimization
formula_cancer <- formula(status ~ age * sex)
opt_args <- make_opt_args(cancer, formula_cancer, gps_method = "m1")
withr::with_seed(8252, {
  opt_results <- optimize_gps(
    data = cancer,
    formula = formula_cancer,
    opt_args = opt_args,
    n_iter = 2000
  )
})
# Select optimal combinations prioritizing SMD balance and matching in key
# groups
select_opt(
  x = opt_results,
  smd_groups = list(
    c("adenoma", "control"),
    c("control", "crc_benign"),
    c("crc_malignant", "control")
  ),
  smd_variables = "age",
  smd_type = "max",
  perc_matched = c("adenoma", "crc_malignant")
)
}
}
