% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/filterResults.R
\docType{class}
\name{FilterResults-class}
\alias{FilterResults-class}
\alias{FilterResults}
\title{FilterResults}
\description{
Class for estimated Dynamic Gompertz Curve model and contains
methods to extract smoothed/filtered estimates of the states, the level of
the incidence variable \eqn{y}, and forecasts of \eqn{y}.
}
\section{Methods}{

\describe{
\item{\code{get_growth_y(smoothed = FALSE, return.components = FALSE)}}{Returns the growth rate of the incidence (\eqn{y}) of the cumulated
variable (\eqn{Y}). Computed as
\deqn{g_t = \exp\{\delta_t\}+\gamma_t.}
 \subsection{Parameters}{\itemize{
  \item{\code{smoothed} Logical value indicating whether to use the
  smoothed estimates of \eqn{\delta} and \eqn{\gamma} to compute the
  growth rate (\code{TRUE}), or the contemporaneous filtered estimates
  (\code{FALSE}). Default is \code{FALSE}.}
  \item{\code{return.components} Logical value indicating whether to
  return the estimates of \eqn{\delta} and \eqn{\gamma} as well as
  the estimates of the growth rate, or just the growth rate. Default is
  \code{FALSE}.}
}}
\subsection{Return Value}{\code{xts} object containing
smoothed/filtered growth rates and components (\eqn{\delta} and
\eqn{\gamma}), where applicable.}}

\item{\code{get_gy_ci(smoothed = FALSE, confidence_level = 0.68)}}{Returns the growth rate of the incidence (\eqn{y}) of the cumulated
variable (\eqn{Y}). Computed as
\deqn{g_t = \exp\{\delta_t\}+\gamma_t.}
 \subsection{Parameters}{\itemize{
  \item{\code{smoothed} Logical value indicating whether to use the
  smoothed estimates of \eqn{\delta} and \eqn{\gamma} to compute the
  growth rate (\code{TRUE}), or the contemporaneous filtered estimates
  (\code{FALSE}). Default is \code{FALSE}.}
  \item{\code{confidence_level} Confidence level for the confidence
  interval.  Default is \eqn{0.68}, which is one standard deviation for
  a normally distributed random variable.}
}}
\subsection{Return Value}{\code{xts} object containing smoothed/filtered
 growth rates and upper and lower bounds for the confidence intervals.}}

\item{\code{predict_all(n.ahead, sea.on = FALSE, return.all = FALSE)}}{Returns forecasts of the incidence variable \eqn{y}, the state variables
 and the conditional covariance matrix
for the states.
 \subsection{Parameters}{\itemize{
  \item{\code{n.ahead} The number of forecasts you wish to create from
  the end of your sample period.}
  \item{\code{sea.on} Logical value indicating whether seasonal
  components should be included in the
  state-space model or not. Default is \code{TRUE}.}
  \item{\code{return.all} Logical value indicating whether to return
  all filtered estimates and forecasts
  (\code{TRUE}) or only the forecasts (\code{FALSE}). Default is
  \code{FALSE}.}
}}
\subsection{Return Value}{\code{xts} object containing the forecast
(and filtered, where applicable) level
of \eqn{y} (\code{y.hat}), \eqn{\delta} (\code{level.t.t}),
\eqn{\gamma} (\code{slope.t.t}), vector of states including the
seasonals where applicable (\code{a.t.t}) and covariance matrix of all
states including seasonals where applicable (\code{P.t.t}).}}

\item{\code{predict_level(
  y.cum,
  n.ahead,
  confidence_level,
  sea.on = FALSE,
  return.diff = FALSE
)}}{Forecast the cumulated variable or the incidence of it. This function returns
the forecast of the cumulated variable \eqn{Y}, or the forecast of the incidence of the cumulated variable, \eqn{y}. For
example, in the case of an epidemic, \eqn{y} might be daily new cases of
the disease and
 \eqn{Y} the cumulative number of recorded infections.
 \subsection{Parameters}{\itemize{
  \item{\code{y.cum} The cumulated variable.}
  \item{\code{n.ahead} The number of periods ahead you wish to forecast from
  the end of the estimation window.}
  \item{\code{confidence_level} The confidence level for the log growth
   rate that should be used to compute
  the forecast intervals of \eqn{y}.}
  \item{\code{return.diff} Logical value indicating whether to return the cumulated variable,
  \eqn{Y}, or the incidence of it,
  \eqn{y} (i.e., the first difference of the cumulated variable). Default is
  \code{FALSE}.}
}}
\subsection{Return Value}{\code{xts} object containing the point
forecasts and upper and lower bounds of
the forecast interval.}}

\item{\code{print_estimation_results()}}{Prints a table of estimated parameters in a format ready to paste into
LaTeX.}
}}

\examples{
library(tsgc)
data(gauteng,package="tsgc")
idx.est <- zoo::index(gauteng) <= as.Date("2020-07-20")

# Specify a model
model <- SSModelDynamicGompertz$new(Y = gauteng[idx.est], q = 0.005)
# Estimate a specified model
res <- model$estimate()
# Print estimation results
res$print_estimation_results()
# Forecast 7 days ahead from the end of the estimation window
res$predict_level(y.cum = gauteng[idx.est], n.ahead = 7,
  confidence_level = 0.68)
# Forecast 7 days ahead from the model and return filtered states
res$predict_all(n.ahead = 7, return.all = TRUE)
# Return the filtered growth rate and its components
res$get_growth_y(return.components = TRUE)
# Return smoothed growth rate of incidence variable and its confidence
# interval
res$get_gy_ci(smoothed = TRUE, confidence_level = 0.68)

}
\references{
Harvey, A. C. and Kattuman, P. (2021). A Farewell to R:
Time Series Models for Tracking and
Forecasting Epidemics, Journal of the Royal Society Interface, vol 18(182):
20210179
}
