#' Plot a Correlation Circle (Unit Circle)
#'
#' This function generates a correlation circle plot from two-dimensional coordinates,
#' commonly used in principal component analysis (PCA) or other multivariate methods.
#'
#' Arrows are drawn from the origin to each point specified in \code{points}. A reference circle
#' with radius 1 is displayed. You can also show the percentage of explained inertia and point labels.
#'
#' The \code{inertia} argument is flexible and can be passed as the second or third parameter
#' if argument names are omitted.
#'
#' @param points A matrix or data frame with two numeric columns (x, y) representing the coordinates of the vectors.
#' @param inertia A number between 0 and 100 representing the percentage of explained inertia. It is displayed in the title.
#' @param labels A character vector with labels for the points (optional). If not specified, labels are left blank.
#' @param title Optional text used as the main title of the plot.
#'
#' @return A \code{ggplot} object with the generated plot.
#'
#' @import ggplot2
#'
#' @seealso \code{\link{statis}}
#'
#' @examples
#' data(expert1, expert2, expert3)
#'
#' labels <- c("Expert 1", "Expert 2", "Expert 3")
#'
#' # If you want to select an specific table or row just set the parameters in the statis function.
#'
#' res <- statis(list(expert1, expert2, expert3), table.labels = labels)
#'
#' # Circle of correlations of all the tables
#' inter <- res$circle.inter
#' plot.statis.circle(inter$points, inter$inertia, inter$labels, inter$title)
#'
#' # Circle of correlations of all variables evolution
#' intra <- res$circle.intra
#' plot.statis.circle(intra$points, intra$inertia, intra$labels, intra$title)
#'
#' @rawNamespace export(plot.statis.circle)
plot.statis.circle <- function(points, inertia = 100, labels = NULL, title = "") {
  # Checks reorder values when the second argument is mistakenly interpreted as `labels` or `title`.
  if (is.numeric(labels) && length(labels) == 1 && is.character(title)) {
    inertia <- as.numeric(labels)
    labels <- NULL
  }
  if (is.numeric(title) && length(title) == 1) {
    inertia <- as.numeric(title)
    title <- ""
  }

  # ----------------------------------------------------------------------
  # Validate and constrain inertia (percentage of explained variance)
  #   - Must be numeric
  #   - Single value
  #   - Clamped between 0 and 100
  if (!is.numeric(inertia) || length(inertia) != 1 || is.na(inertia)) {
    stop("The 'inertia' parameter must be a number (percentage).")
  }
  inertia <- max(0, min(100, inertia))

  # ----------------------------------------------------------------------
  # Build the data frame with coordinates of the variables
  df <- as.data.frame(points)
  names(df) <- c("x", "y")
  if (is.null(labels)) labels <- rep("", nrow(df)) # If labels are missing, fill with empty strings
  df$lab <- labels

  # ----------------------------------------------------------------------
  # Build the unit circle (used in correlation circles)
  ang <- seq(0, 2*pi, length.out = 361)
  circ <- data.frame(cx = cos(ang), cy = sin(ang))

  # Padding (Extended plot limits to avoid clipping label text)
  lim <- 1.1

  # Factor to shorten the arrow (do not reach full radius.)
  factor.arrow <- 0.95
  df$xend <- df$x * factor.arrow
  df$yend <- df$y * factor.arrow

  # ----------------------------------------------------------------------
  # Construct the ggplot graph
  ggplot(df, aes(x, y)) +
    # Draw the correlation circle
    geom_path(data = circ, aes(cx, cy), inherit.aes = FALSE) +
    # Draw x = 0 and y = 0 axes
    geom_hline(yintercept = 0, linewidth = 0.3) +
    geom_vline(xintercept = 0, linewidth = 0.3) +
    # Draw red arrows from the origin to each variable direction
    geom_segment(aes(x = 0, y = 0, xend = xend, yend = yend),
                 arrow = arrow(length = unit(0.2, "cm")),
                 color = "red", linewidth = 0.6) +
    # Plot the variable points
    geom_point(size = 2.4, color = "red") +
    # Labels positioned left/right depending on sign of x
    geom_text(aes(label = lab,
                  hjust = ifelse(x >= 0, -0.1, 1.1)),
              vjust = 0.5) +
    # Preserve aspect ratio and set limits
    coord_fixed(ratio = 1,
                xlim = c(-lim, lim),
                ylim = c(-lim, lim),
                clip = "off") +
    # Title includes inertia percentage
    labs(
      title = paste0(title, "\nInertia Explained: ", sprintf("%.1f", inertia), "%"),
      x = "x", y = "y"
    ) +
    # Minimal theme with visible border
    theme_minimal(base_size = 12) +
    theme(
      panel.border = element_rect(colour = "black", fill = NA, linewidth = 0.5),
      panel.grid.minor = element_blank(),
      plot.margin = margin(5.5, 40, 5.5, 5.5)
    )
}

#' Plot a Plane of Observations or 2D Projections
#'
#' This function generates a two-dimensional scatter plot with centered axes,
#' useful for representing the results of multivariate analyses.
#'
#' The plot includes points, optional labels, Cartesian axes (centered at 0),
#' and a title indicating the percentage of explained inertia.
#'
#' @param points A matrix, data frame, or a length-2 vector with coordinates (x, y). Must have exactly two columns.
#' @param inertia A number between 0 and 100 indicating the percentage of explained inertia (optional, defaults to 100).
#' @param labels A character vector with labels for the points (optional). Must match the number of rows in \code{points}.
#' @param title A text string to be used as the main title of the plot (optional).
#'
#' @return A \code{ggplot} object with the generated plot.
#'
#' @import ggplot2
#'
#' @seealso \code{\link{statis}}
#'
#' @examples
#' data(expert1, expert2, expert3)
#'
#' labels <- c("Expert 1", "Expert 2", "Expert 3")
#'
#' # If you want to select an specific table or row just set the parameters in the statis function.
#'
#' res <- statis(list(expert1, expert2, expert3), table.labels = labels)
#'
#' # Main Plane of Average Individuals
#' individuals <- res$plane.individuals
#' plot.statis.plane(individuals$points, individuals$inertia, individuals$labels, individuals$title)
#'
#' # Main Plane of the Evolution of individuals
#' evolution <- res$plane.evolution
#' plot.statis.plane(evolution$points, evolution$inertia, evolution$labels, evolution$title)
#'
#' @rawNamespace export(plot.statis.plane)
plot.statis.plane <- function(points, inertia = 100, labels = NULL, title = "") {
  # ----------------------------------------------------------------------
  # Ensure that the input `points` is a 2-column structure:
  #
  # - It may be provided as a numeric vector of length 2 → convert to 1-row matrix.
  # - Otherwise it must be a matrix or data frame with two columns (x, y).
  if (is.vector(points) && length(points) == 2) {
    points <- matrix(points, ncol = 2, byrow = TRUE)
  }
  if (!is.matrix(points) && !is.data.frame(points)) {
    stop("`points` must be a matrix or data.frame with 2 columns (x, y).")
  }
  if (ncol(points) != 2) stop("`points` must have exactly 2 columns.")

  # ----------------------------------------------------------------------
  # Convert to data frame and assign column names.
  df <- as.data.frame(points)
  names(df) <- c("x", "y")
  m <- nrow(df)

  # ----------------------------------------------------------------------
  # Labels:
  #   - If not provided, use empty strings.
  #   - Must match the number of points.
  if (is.null(labels)) labels <- rep("", m)
  if (length(labels) != m) stop("`labels` must be the same length as the number of points.")
  df$label <- labels

  # ----------------------------------------------------------------------
  # Inertia validation:
  #   - Must be a single numeric value
  #   - Clamp between 0 and 100 (percentage)
  if (!is.numeric(inertia) || length(inertia) != 1 || is.na(inertia)) {
    stop("The 'inertia' parameter must be a number (percentage).")
  }
  inertia <- max(0, min(100, inertia))

  # ----------------------------------------------------------------------
  # Plot limits
  xr <- range(df$x, na.rm = TRUE)
  yr <- range(df$y, na.rm = TRUE)
  span.x <- diff(xr)
  span.y <- diff(yr)
  pad <- 0.04 * max(span.x, span.y)  # 4% of the highest range
  xr <- xr + c(-pad, pad)
  yr <- yr + c(-pad, pad)

  # ----------------------------------------------------------------------
  # Start building the ggplot
  p <- ggplot(df, aes(x, y)) +
    # Draw axis lines (origin axes)
    geom_hline(yintercept = 0, linewidth = 0.4, linetype = "dashed", color = "gray40") +
    geom_vline(xintercept = 0, linewidth = 0.4, linetype = "dashed", color = "gray40") +
    # Scatter plot of individuals or elements
    geom_point(size = 2.6, color = "black", fill = "red", shape = 21, stroke = 0.6) +
    # Add labels only if at least one is non-empty
    { if (any(nzchar(df$label)))
      geom_text(aes(x + 0.02*max(span.x, span.y), y, label = label), hjust = 0)
    } +
    # Axis limits without additional expansion
    scale_x_continuous(limits = xr, expand = expansion(mult = 0)) +
    scale_y_continuous(limits = yr, expand = expansion(mult = 0)) +
    # Preserve aspect ratio: same scale on both axes
    coord_fixed(ratio = 1, clip = "off") +
    # Title includes inertia percentage
    labs(
      title = paste0(title, "\nInertia Explained: ", sprintf("%.1f", inertia), "%"),
      x = "X Axis", y = "Y Axis"
    ) +
    # Minimal theme with border
    theme_minimal(base_size = 12) +
    theme(
      aspect.ratio = 1,
      panel.border = element_rect(colour = "black", fill = NA, linewidth = 0.5),
      panel.grid.minor = element_blank()
    )

  return(p)
}
