% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/outbreakNB.R
\name{outbreakNB}
\alias{outbreakNB}
\title{Fit an outbreak detection negative binomial outbreak model}
\usage{
outbreakNB(
  cases,
  pop = NULL,
  covariates = NULL,
  beta_init = NULL,
  r_init = NULL,
  beta_prior_mean = 0,
  beta_prior_sd = 10,
  r_prior_shape = 1,
  r_prior_rate = 1,
  p_priors = 1,
  n_iter = 1e+05,
  n_burnin = 10000,
  n_chains = 3,
  n_thin = 1,
  save_params = c("beta", "r", "Z"),
  dates = NULL,
  plot_Z = FALSE
)
}
\arguments{
\item{cases}{Integer or numeric vector of observed case counts (length N).}

\item{pop}{(Optional) Numeric vector of population offsets (length N). If NULL, offset = 1.}

\item{covariates}{(Optional) Data.frame or matrix of covariates for the count model (N x p_c).}

\item{beta_init}{(Optional) List of length \code{n_chains} giving initial values for beta (each a vector of length p_c+1).}

\item{r_init}{(Optional) Numeric vector of length \code{n_chains} giving initial values for the NB dispersion parameter.}

\item{beta_prior_mean}{Prior mean for beta coefficients of the Negative binomial part (default = 0).}

\item{beta_prior_sd}{Prior SD   for beta coefficients of the Negative binomial part (default = 10).}

\item{r_prior_shape}{Shape parameter of a prior on r (default = 1).}

\item{r_prior_rate}{Rate  parameter of b prior on r (default = 1).}

\item{p_priors}{Alpha parameters for the binomial priors on p00 and p11 (default = 1).}

\item{n_iter}{Total number of MCMC iterations per chain (default = 100000).}

\item{n_burnin}{Number of burn-in iterations (default = 10000).}

\item{n_chains}{Number of MCMC chains (default = 3).}

\item{n_thin}{Thinning interval for MCMC samples (default = 1).}

\item{save_params}{Character vector of parameter names to save (must include "Z").}

\item{dates}{(Optional) Vector of Date or POSIX dates for plotting Z; if NULL, uses index 1:N.}

\item{plot_Z}{Logical; if TRUE, returns a ggplot2 object of the posterior mean Z over time.}
}
\value{
A list with MCMC summary, samples, DIC, WAIC, and plot of the probability of being in an epidemic state.
}
\description{
Fit an outbreak detection negative binomial outbreak model
}
\examples{
# ---- tiny example for users & CRAN (< 5s) ----
set.seed(12)
n <- 120
# baseline NB counts with an injected "outbreak" window
cases <- rnbinom(n, size = 6, mu = 8)
cases[70:74] <- cases[70:74] + rnbinom(5, size = 6, mu = 25)
dates <- as.Date("2020-01-01") + seq_len(n) - 1L

\dontshow{
# checks that run on CRAN but are hidden from users
stopifnot(length(cases) == n, all(cases >= 0), inherits(dates, "Date"))
}

# ---- actually run the detector, but only when JAGS is available ----
\dontshow{if (nzchar(Sys.which("jags")) && requireNamespace("R2jags", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
fit <- outbreakNB(
  cases   = cases,
  dates   = dates,
  n_iter  = 10,   # keep fast for examples
  n_burnin= 1,
  n_chains= 1,
  n_thin  = 1,
  plot_Z  = FALSE  # avoid plotting in examples (rename/omit if not applicable)
)
print(fit)
}

\donttest{
# ---- longer user-facing demo (skipped on checks) ----
# Increase iterations a bit for a stabler run (still JAGS-gated by @examplesIf above)
# fit2 <- outbreakNB(
#   cases   = cases,
#   dates   = dates,
#   n_iter  = 1500,
#   n_burnin= 500,
#   n_chains= 2,
#   n_thin  = 2,
#   plot_Z  = FALSE
# )
# print(fit2)
}

\dontrun{
# ---- time-consuming / full demo (not run anywhere) ----
# Here you might use larger MCMC and produce figures/tables of alerts.
# fit_full <- outbreakNB(
#   cases   = cases,
#   dates   = dates,
#   n_iter  = 10000,
#   n_burnin= 5000,
#   n_chains= 4,
#   n_thin  = 5,
#   plot_Z  = TRUE
# )
# print(fit_full)
}

if (interactive()) {
  # e.g., if a plot method exists:  # plot(fit)
}
\dontshow{\}) # examplesIf}
}
