% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/01_the_sdl_distribution.R
\name{sdl}
\alias{sdl}
\alias{dsdl}
\alias{psdl}
\alias{qsdl}
\alias{rsdl}
\title{The Modified Skew Discrete Laplace Distribution}
\usage{
dsdl(x, mu, phi, xi = 0, log = FALSE)

psdl(q, mu, phi, xi = 0, lower.tail = TRUE)

qsdl(p, mu, phi, xi = 0, lower.tail = TRUE)

rsdl(n, mu, phi, xi = 0)
}
\arguments{
\item{x, q}{vector of integer quantiles.}

\item{mu}{vector of means.}

\item{phi}{vector of dispersion parameters (greater than \code{abs(mu - xi)}).}

\item{xi}{the mode of the distribution, an integer value.}

\item{log}{logical; if \code{TRUE}, probabilities \code{p} are given as \code{log(p)}.}

\item{lower.tail}{logical; if \code{TRUE} (default), probabilities are \code{P(X <= x)}, otherwise, \code{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{number of random values to return.}
}
\value{
\code{dsdl} returns the probability mass function, \code{psdl}
    gives the distribution function, \code{qsdl} gives the quantile function,
    and \code{rsdl} generates random observations.
}
\description{
Probability mass function, distribution function, quantile function, and a random
    generation for the modified skew discrete Laplace (SDL) distribution with mean \code{mu},
    dispersion parameter \code{phi}, and mode \code{xi}.
}
\details{
The SDL distribution was introduced by Kozubowski and Inusah (2006) as the discrete part
    of the continuous skew Laplace distribution centered at zero  (Kotz et al., 2001, Ch. 3).
    Although the SDL distribution has attractive properties, the discrete version of the
    zero-centered skew Laplace distribution induces that the mode of the resulting model
    is always equal to zero.

    To overcome this limitation, Medeiros and Bourguignon (2025) proposed to obtain the discrete
    version of the Laplace skew distribution without setting its location parameter to zero,
    defining a new probability model that generalizes the SDL distribution.

    This set of functions represents the probability mass function, the cumulative distribution
    function, the quantile function, and a random number generator for the modified SDL
    distribution parameterized in terms of \code{mu} (mean), \code{phi} (a dispersion parameter),
    and \code{xi} (the mode of the distribution).

    Let \eqn{X} be a discrete random variable following a SDL distribution with mean \eqn{\mu},
    dispersion parameter \eqn{\phi}, and mode \eqn{\xi}. The probability mass function of X is

    \eqn{
    \textrm{P}(X = x) =  \left\{\begin{array}{ll}
    \dfrac{1}{1 + \phi}\left(\dfrac{\phi - \mu + \xi}{2+ \phi - \mu + \xi}\right)^{-(x - \xi)}, & x \in \{\xi - 1, \xi - 2, \ldots\}, \\ \\
    \dfrac{1}{1 + \phi}\left(\dfrac{\phi + \mu - \xi}{2+ \phi + \mu - \xi}\right)^{x - \xi}, & x \in \{\xi, \xi + 1, \xi + 2, \ldots\}.
    \end{array}\right.
    }

    The parametric space of this parameterization satisfies the constraint \eqn{\mu \in \mathbb{R}},
    \eqn{\phi > |\mu - \xi| }, and \eqn{\xi \in \mathbb{Z}}. Additionally, the expected value and
    the variance of \eqn{X} are given, respectively, by

    \eqn{
    \textrm{E}(Y) = \mu \quad \mbox{ and } \quad \textrm{Var}(Y) = \dfrac{\phi(\phi + 2) + (\mu - \xi)^2}{2}.
    }
}
\examples{
### Probability function ###

# Parameters
mu <- c(-4, 2, 4)
phi <- 6.5
xi <- 2

xvals <- -30:30

# Skewed-left distribution (mu < xi)
plot(xvals, dsdl(xvals, mu[1], phi, xi),
     type = "h", xlab = "x", ylab = "Pmf")

# Symmetric distribution (mu = xi)
plot(xvals, dsdl(xvals, mu[2], phi, xi),
     type = "h", xlab = "x", ylab = "Pmf")

# Skewed-right distribution (mu > 0)
plot(xvals, dsdl(xvals, mu[3], phi, xi),
     type = "h", xlab = "x", ylab = "Pmf")

### Difference between paired samples of non-negative observations  ###

# Parameters
mu <- 3
phi <- 4
xi <- 0

# Paired samples of a pre-post treatment experimental study
before <- rgeom(1000, 2 / (2 + phi - mu))
after <- rgeom(1000, 2 / (2 + phi + mu))

# Response variable
y <- after - before

# Barplot
obj <- barplot(prop.table(table(y)),
               xlab = "Response",
               ylab = "Proportion",
               col = "white",
               ylim = c(0, mean(y == 0) + 0.01))

# Sdl model for the differences
points(obj, dsdl(sort(unique(y)), mu, phi, xi), col = "red", pch = 16)
}
\references{
Kotz, S., Kozubowski, T. J., and Podgórski, K. (2001). \emph{The Laplace Distribution
    and Generalizations: A Revisit with Applications to Communications, Economics, Engineering,
    and Finance}. Birkhauser, Boston.

    Kozubowski, T. J., and Inusah, S. (2006). A skew Laplace distribution on integers.
    \emph{Annals of the Institute of Statistical Mathematics}, \bold{58}, 555---571.

    Medeiros, R. M. R., and Bourguignon, M. (2025). Modified skew discrete Laplace
    regression models for integer valued data with applications to paired samples.
    \emph{Manuscript submitted for publication.}
}
\author{
Rodrigo M. R. de Medeiros <\email{rodrigo.matheus@ufrn.br}>
}
