% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/editLayers.R
\name{editLayer}
\alias{editLayer}
\title{Edit a color patch using morphological operations}
\usage{
editLayer(
  recolorize_obj,
  layer_idx,
  operation = "clean",
  px_size = 2,
  plotting = TRUE
)
}
\arguments{
\item{recolorize_obj}{A recolorize object from \code{\link[=recolorize]{recolorize()}},
\code{\link[=recluster]{recluster()}}, or \code{\link[=imposeColors]{imposeColors()}}.}

\item{layer_idx}{A single index value (numeric) indicating which
layer to edit. Corresponds to the order of the colors in the \code{centers}
attribute of the recolorize object, and to the indices in the
\code{pixel_assignments} attribute of the same.}

\item{operation}{The name of an imager morphological operation to perform on
the layer, passed as a string. See details.}

\item{px_size}{The size (in pixels) of the elements to filter. If
\code{operation = "shrink"} and \code{px_size = 2}, for example, the color
patch will be shrunk by a 2-pixel radius.}

\item{plotting}{Logical. Plot results?}
}
\value{
A \code{recolorize} object. The \code{sizes}, \verb{pixel_assignments,}, and
\code{recolored_img} attributes will differ from the input object for the
relevant color patch (layer) to reflect the edited layer.
}
\description{
Applies one of several morphological operations from \code{imager} to a layer of a
recolorize object. Convenient for cleaning up a color patch without affecting
other layers of the recolorized image. This can be used to despeckle, fill in
holes, or uniformly grow or shrink a color patch.
}
\details{
Current imager operations are:
\itemize{
\item \code{\link[imager:grow]{imager::grow()}}: Grow a pixset
\item \code{\link[imager:grow]{imager::shrink()}}: Shrink a pixset
\item \code{\link[imager:clean]{imager::fill()}}: Remove holes in an pixset. Accomplished by
growing and then shrinking a pixset.
\item \code{\link[imager:clean]{imager::clean()}}: Remove small isolated elements (speckle).
Accomplished by shrinking and then growing a pixset.
}
}
\examples{
# load image and recolorize it
img <- system.file("extdata/corbetti.png", package = "recolorize")

# first do a standard color binning
init_fit <- recolorize(img, bins = 2, plotting = FALSE)

# then cluster patches by similarity
re_fit <- recluster(init_fit, cutoff = 40)

# to reset graphical parameters:
current_par <- graphics::par(no.readonly = TRUE)

# examine individual layers:
layout(matrix(1:6, nrow = 2))
layers <- splitByColor(re_fit, plot_method = "color")

# notice patch 2 (cream) - lots of stray pixels
edit_cream_layer <- editLayer(re_fit,
                              layer_idx = 2,
                              operation = "clean",
                              px_size = 3)

# shrinking and growing by the same element size gives us less flexibility, so
# we can also shrink and then grow, using different px_size arguments:
edit_green_1 <- editLayer(re_fit,
                          layer_idx = 4,
                          operation = "shrink",
                          px_size = 2)
edit_green_2 <- editLayer(edit_green_1,
                          layer_idx = 4,
                          operation = "grow",
                          px_size = 3)

# we can get pleasingly mondrian about it:
new_fit <- re_fit
for (i in 1:nrow(new_fit$centers)) {
  new_fit <- editLayer(new_fit,
                       layer_idx = i,
                       operation = "fill",
                       px_size = 5, plotting = FALSE)
}
plot(new_fit)

graphics::par(current_par)

}
\seealso{
\link{editLayers} for editing multiple layers (with multiple operations)
at once; a wrapper for this function.
}
