% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/solve_MDP.R
\name{solve_MDP}
\alias{solve_MDP}
\alias{solve_MDP_DP}
\alias{solve_MDP_TD}
\title{Solve an MDP Problem}
\usage{
solve_MDP(model, method = "value", ...)

solve_MDP_DP(
  model,
  method = "value_iteration",
  horizon = NULL,
  discount = NULL,
  N_max = 1000,
  error = 0.01,
  k_backups = 10,
  U = NULL,
  verbose = FALSE
)

solve_MDP_TD(
  model,
  method = "q_learning",
  horizon = NULL,
  discount = NULL,
  alpha = 0.5,
  epsilon = 0.1,
  N = 100,
  U = NULL,
  verbose = FALSE
)
}
\arguments{
\item{model}{an MDP problem specification.}

\item{method}{string; one of the following solution methods: \code{'value_iteration'},
\code{'policy_iteration'}, \code{'q_learning'}, \code{'sarsa'}, or \code{'expected_sarsa'}.}

\item{...}{further parameters are passed on to the solver function.}

\item{horizon}{an integer with the number of epochs for problems with a
finite planning horizon. If set to \code{Inf}, the algorithm continues
running iterations till it converges to the infinite horizon solution. If
\code{NULL}, then the horizon specified in \code{model} will be used.}

\item{discount}{discount factor in range \eqn{(0, 1]}. If \code{NULL}, then the
discount factor specified in \code{model} will be used.}

\item{N_max}{maximum number of iterations allowed to converge. If the
maximum is reached then the non-converged solution is returned with a
warning.}

\item{error}{value iteration: maximum error allowed in the utility of any state
(i.e., the maximum policy loss) used as the termination criterion.}

\item{k_backups}{policy iteration: number of look ahead steps used for approximate policy evaluation
used by the policy iteration method.}

\item{U}{a vector with initial utilities used for each state. If
\code{NULL}, then the default of a vector of all 0s is used.}

\item{verbose}{logical, if set to \code{TRUE}, the function provides the
output of the solver in the R console.}

\item{alpha}{step size in \verb{(0, 1]}.}

\item{epsilon}{used for \eqn{\epsilon}-greedy policies.}

\item{N}{number of episodes used for learning.}
}
\value{
\code{solve_MDP()} returns an object of class POMDP which is a list with the
model specifications (\code{model}), the solution (\code{solution}).
The solution is a list with the elements:
\itemize{
\item \code{policy} a list representing the policy graph. The list only has one element for converged solutions.
\item \code{converged} did the algorithm converge (\code{NA}) for finite-horizon problems.
\item \code{delta} final \eqn{\delta} (value iteration and infinite-horizon only)
\item \code{iterations} number of iterations to convergence (infinite-horizon only)
}
}
\description{
Implementation of value iteration, modified policy iteration and other
methods based on reinforcement learning techniques to solve finite
state space MDPs.
}
\details{
Implemented are the following dynamic programming methods (following
Russell and Norvig, 2010):
\itemize{
\item \strong{Modified Policy Iteration}
starts with a random policy and iteratively performs
a sequence of
\enumerate{
\item approximate policy evaluation (estimate the value function for the
current policy using \code{k_backups} and function \code{\link[=MDP_policy_evaluation]{MDP_policy_evaluation()}}), and
\item policy improvement (calculate a greedy policy given the value function).
The algorithm stops when it converges to a stable policy (i.e., no changes
between two iterations).
}
\item \strong{Value Iteration} starts with
an arbitrary value function (by default all 0s) and iteratively
updates the value function for each state using the Bellman equation.
The iterations
are terminated either after \code{N_max} iterations or when the solution converges.
Approximate convergence is achieved
for discounted problems (with \eqn{\gamma < 1})
when the maximal value function change for any state \eqn{\delta} is
\eqn{\delta \le error (1-\gamma) / \gamma}. It can be shown that this means
that no state value is more than
\eqn{error} from the value in the optimal value function. For undiscounted
problems, we use \eqn{\delta \le error}.

The greedy policy
is calculated from the final value function. Value iteration can be seen as
policy iteration with truncated policy evaluation.
}

Note that the policy converges earlier than the value function.

Implemented are the following temporal difference control methods
described in Sutton and Barto (2020).
Note that the MDP transition and reward models are only used to simulate
the environment for these reinforcement learning methods.
The algorithms use a step size parameter \eqn{\alpha} (learning rate) for the
updates and the exploration parameter \eqn{\epsilon} for
the \eqn{\epsilon}-greedy policy.

If the model has absorbing states to terminate episodes, then no maximal episode length
(\code{horizon}) needs to
be specified. To make sure that the algorithm does finish in a reasonable amount of time,
episodes are stopped after 10,000 actions with a warning. For models without absorbing states,
a episode length has to be specified via \code{horizon}.
\itemize{
\item \strong{Q-Learning} is an off-policy temporal difference method that uses
an \eqn{\epsilon}-greedy behavior policy and learns a greedy target
policy.
\item \strong{Sarsa} is an on-policy method that follows and learns
an \eqn{\epsilon}-greedy policy. The final \eqn{\epsilon}-greedy policy
is converted into a greedy policy.
\item \strong{Expected Sarsa}: We implement an on-policy version that uses
the expected value under the current policy for the update.
It moves deterministically in the same direction as Sarsa
moves in expectation. Because it uses the expectation, we can
set the step size \eqn{\alpha} to large values and even 1.
}
}
\examples{
data(Maze)
Maze

# use value iteration
maze_solved <- solve_MDP(Maze, method = "value_iteration")
maze_solved
policy(maze_solved)

# plot the value function U
plot_value_function(maze_solved)

# Maze solutions can be visualized
gridworld_plot_policy(maze_solved)

# use modified policy iteration
maze_solved <- solve_MDP(Maze, method = "policy_iteration")
policy(maze_solved)

# finite horizon
maze_solved <- solve_MDP(Maze, method = "value_iteration", horizon = 3)
policy(maze_solved)
gridworld_plot_policy(maze_solved, epoch = 1)
gridworld_plot_policy(maze_solved, epoch = 2)
gridworld_plot_policy(maze_solved, epoch = 3)

# create a random policy where action n is very likely and approximate
#  the value function. We change the discount factor to .9 for this.
Maze_discounted <- Maze
Maze_discounted$discount <- .9
pi <- random_MDP_policy(Maze_discounted, 
        prob = c(n = .7, e = .1, s = .1, w = 0.1))
pi

# compare the utility function for the random policy with the function for the optimal
#  policy found by the solver.
maze_solved <- solve_MDP(Maze)

MDP_policy_evaluation(pi, Maze, k_backup = 100)
MDP_policy_evaluation(policy(maze_solved), Maze, k_backup = 100)

# Note that the solver already calculates the utility function and returns it with the policy
policy(maze_solved)

# Learn a Policy using Q-Learning
maze_learned <- solve_MDP(Maze, method = "q_learning", N = 100)
maze_learned

maze_learned$solution
policy(maze_learned)
plot_value_function(maze_learned)
gridworld_plot_policy(maze_learned)
}
\references{
Russell, S., Norvig, P. (2021). Artificial Intelligence: A Modern Approach.
Fourth edition. Prentice Hall.

Sutton, R. S., Barto, A. G. (2020). Reinforcement Learning: An Introduction.
Second edition. The MIT Press.
}
\seealso{
Other solver: 
\code{\link{solve_POMDP}()},
\code{\link{solve_SARSOP}()}

Other MDP: 
\code{\link{MDP}()},
\code{\link{MDP2POMDP}},
\code{\link{MDP_policy_functions}},
\code{\link{accessors}},
\code{\link{actions}()},
\code{\link{add_policy}()},
\code{\link{gridworld}},
\code{\link{reachable_and_absorbing}},
\code{\link{regret}()},
\code{\link{simulate_MDP}()},
\code{\link{transition_graph}()},
\code{\link{value_function}()}
}
\author{
Michael Hahsler
}
\concept{MDP}
\concept{solver}
