#' @title One-Compartment Oral Pharmacokinetic Model (Linear, First-Order Absorption)
#' @name one_compartment_oral
#' @description
#' Fits plasma concentration-time data to the one-compartment oral pharmacokinetic model
#' using a linearized approach. The model assumes first-order absorption and first-order
#' elimination.
#'
#' Model:
#' C(t) = (F * Dose * ka / (Vd * (ka - kel))) * (exp(-kel * t) - exp(-ka * t))
#'
#' Linearized approximation:
#' Using log-transformed data in the elimination phase (t >> t_max), log(C) ~= log(C0) - kel * t
#'
#' Parameters:
#' \itemize{
#'   \item ka: absorption rate constant
#'   \item kel: elimination rate constant
#'   \item C0: apparent initial concentration for elimination phase
#'   \item t_half: elimination half-life
#'   \item Vd: apparent volume of distribution
#'   \item CL: clearance
#' }
#'
#' @param data A data frame containing plasma concentration-time data.
#' @param time_col Character string specifying the column name for time.
#' @param conc_col Character string specifying the column name for plasma concentration.
#' @param dose Numeric value specifying the administered oral dose.
#' @param group_col Optional character string specifying a grouping variable
#'   (e.g., formulation, subject).
#' @param plot Logical; if TRUE, generates a concentration-time plot with fitted lines.
#' @param annotate Logical; if TRUE, annotates the plot with PK parameters
#'   (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats na.omit lm
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth geom_text scale_y_log10
#' labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with C0, kel, t_half, Vd, CL, and R^2.}
#'   \item{\code{data}}{Processed data used for fitting and plotting.}
#' }
#' @examples
#' # Example I: Single subject oral data
#' df <- data.frame(
#'   time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
#'   concentration = c(5.1, 9.8, 14.2, 13.5, 10.2, 6.8, 4.5, 2.1)
#' )
#' one_compartment_oral(
#'   data = df,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100
#' )
#'
#' # Example II: Condition-dependent kinetics
#' df_cond <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
#'   concentration = c(
#'     4.8, 9.5, 13.7, 12.8, 9.2, 6.4, 3.9,   # Condition A
#'     5.2, 10.1, 14.0, 12.5, 8.7, 5.8, 3.5   # Condition B
#'   ),
#'   condition = rep(c("Condition A", "Condition B"), each = 7)
#' )
#' one_compartment_oral(
#'   data = df_cond,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "condition"
#' )
#'
#' # Example III: Multiple subjects
#' df_subjects <- data.frame(
#'   time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 10),
#'   concentration = c(
#'     5.0, 9.7, 14.0, 13.2, 10.0, 6.6, 4.2,  # Subject 1
#'     4.9, 9.5, 13.8, 12.9, 9.5, 6.3, 4.0,   # Subject 2
#'     5.1, 9.9, 14.1, 13.5, 10.3, 6.9, 4.3,  # Subject 3
#'     4.8, 9.6, 13.9, 13.1, 9.8, 6.5, 4.1,   # Subject 4
#'     5.2, 10.0, 14.3, 13.6, 10.5, 7.1, 4.4, # Subject 5
#'     5.1, 9.8, 14.0, 13.3, 10.1, 6.7, 4.3,  # Subject 6
#'     4.9, 9.6, 13.7, 12.8, 9.4, 6.2, 3.9,   # Subject 7
#'     5.0, 9.9, 14.2, 13.4, 10.2, 6.8, 4.1,  # Subject 8
#'     5.2, 10.1, 14.5, 13.7, 10.7, 7.2, 4.5, # Subject 9
#'     4.8, 9.5, 13.6, 12.7, 9.1, 6.0, 3.8    # Subject 10
#'   ),
#'   subject = rep(paste0("S", 1:10), each = 7)
#' )
#' one_compartment_oral(
#'   data = df_subjects,
#'   time_col = "time",
#'   conc_col = "concentration",
#'   dose = 100,
#'   group_col = "subject"
#' )
#' @references Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
#' 2nd Edition. Marcel Dekker, New York.
#' @references Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
#' Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
#' Swedish Pharmaceutical Press, Stockholm.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "conc", "log_conc", "C0", "kel", "t_half", "Vd",
                         "CL", "R2", "label", "x_pos", "y_pos", "hjust", "vjust",
                         "group", "head", "tail"))

one_compartment_oral <- function(data,
                                 time_col = "time",
                                 conc_col = "concentration",
                                 dose,
                                 group_col = NULL,
                                 plot = TRUE,
                                 annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE))
    stop("Package 'ggplot2' is required.")

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, conc_col, group_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "conc")

  # Remove non-positive concentrations
  df <- df[df$time >= 0 & df$conc > 0, ]
  df$log_conc <- log(df$conc)

  # -------------------------
  # Grouping
  # -------------------------
  if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else {
    df$group <- "Experimental"
  }
  df$group <- as.factor(df$group)

  # -------------------------
  # Linear regression on elimination phase
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {

    fit <- stats::lm(log_conc ~ time, data = d)
    s <- summary(fit)

    kel <- -coef(fit)[2]
    logC0 <- coef(fit)[1]
    C0 <- exp(logC0)
    t_half <- log(2) / kel

    # AUC trapezoidal rule
    auc <- sum(diff(d$time) * (head(d$conc, -1) + tail(d$conc, -1)) / 2)
    CL <- dose / auc
    Vd <- CL / kel

    data.frame(
      group = unique(d$group),
      C0 = C0,
      kel = kel,
      t_half = t_half,
      Vd = Vd,
      CL = CL,
      R2 = s$r.squared
    )
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {

    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = conc, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1) +
      ggplot2::geom_smooth(
        ggplot2::aes(y = exp(log_conc)),
        method = "lm",
        se = FALSE,
        color = "black",
        linewidth = 1
      ) +
      ggplot2::scale_y_log10() +
      ggplot2::labs(
        title = "One-Compartment Oral Pharmacokinetic Model",
        subtitle = "Log-linear elimination phase",
        x = "Time (hours)",
        y = "Plasma Concentration (log scale)",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    # -------------------------
    # Annotations: lower-left and upper-right
    # -------------------------
    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {

      ann <- fit_results
      ann$label <- paste0(
        "C0 = ", round(ann$C0, 2), "\n",
        "k_el = ", signif(ann$kel, 3), "\n",
        "t1/2 = ", round(ann$t_half, 2), "\n",
        "Vd = ", round(ann$Vd, 2), "\n",
        "CL = ", round(ann$CL, 2), "\n",
        "R^2 = ", round(ann$R2, 3)
      )

      x_min <- min(df$time)
      x_max <- max(df$time)
      y_min <- min(df$conc)
      y_max <- max(df$conc)

      ann$x_pos <- c(
        x_min + 0.05 * (x_max - x_min),  # lower left
        x_max - 0.05 * (x_max - x_min)   # upper right
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_min * 0.5,  # lower left
        y_max * 0.95  # upper right
      )[seq_len(nrow(ann))]

      ann$hjust <- c(0, 1)[seq_len(nrow(ann))]
      ann$vjust <- c(0, 1)[seq_len(nrow(ann))]

      p <- p + ggplot2::geom_text(
        data = ann,
        ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
        hjust = ann$hjust,
        vjust = ann$vjust,
        size = 4,
        show.legend = FALSE
      )
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
