# Functions to support the log-linear models found in chapter 2 of
# von Eye, A. & Mun, E. Y. (2005), Analyzing rater agreement: Manifest variable methods.
# Mahwah, NJ: Lawrence Erlbaum.


#' Fits the base model with only independent row and column effects.
#'
#' @param n the matrix of observed counnts
#' @param n the matrix of observed counts
#' @returns a list containing
#'    beta: the regression parameters
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_main_effect <- function(n) {
  x <- log_linear_main_effect_design(n)
  log_linear_fit(n, x)
}


#' Fits the equal weighted diagonal model, where the diagonals all have an additional
#' parameter delta, with the constraint that delta is equal across all categories.
#'
#' @param n the matrix of observed counts
#' @returns a list containing
#'    beta: the regression parameters
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_equal_weighted_diagonal <- function(n) {
  x <- log_linear_equal_weight_agreement_design(n)
  effect_names <- c("agreement")
  log_linear_fit(n, x, effect_names=effect_names)
}


#' Fits the diagonal effects model, where each category has its own parameter delta[k].
#'
#' @param n the matrix of observed counts
#' @returns a list containing
#'    beta: the regression parameters. delta parameters are the final elements of beta
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_diagonal <- function(n) {
  x <- log_linear_main_effect_design(n)
  x_prime <- log_linear_add_all_diagonals(n, x)
  names <- paste0("delta_", 1:nrow(n))
  log_linear_fit(n, x_prime, effect_names=names)
}


#' Fits the basic independent rows and columns model incorporating
#' a linear-by-linear term.
#'
#' @param n matrix of observed counts
#' @param center should the linear-by-linear components be centered to have mean 0?
#' Default is TRUE
#' @returns a list containing
#'    beta: the regression parameters. The linear-by-linear parameter is last
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_linear_by_linear <- function(n, center=TRUE) {
  x <- log_linear_main_effect_design(n)
  x1 <- log_linear_create_linear_by_linear(n, center)
  x_prime <- log_linear_append_column(x, x1)
  names <- c("linear-by-linear")
  log_linear_fit(n, x_prime, effect_names=names)
}


#' Fits the diagonal effects model, where there is a single delta parameter
#' for all categories, while also incorporating a linear-by-linear term.
#'
#' @param n the matrix of observed counts
#' @param center should the linear-by-linear components be centered to have mean 0?
#' Default is TRUE
#' @returns a list containing
#'    beta: the regression parameters. delta parameters come after rows and columns
#'    and finally the linear-by-linear term
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_equal_weight_diagonal_linear <- function(n, center=TRUE) {
  x <- log_linear_equal_weight_agreement_design(n)
  x_linear <- log_linear_create_linear_by_linear(n, center)
  x_prime <- log_linear_append_column(x, x_linear)
  names <- c("delta", "linear-by-linear")
  log_linear_fit(n, x_prime, effect_names=names)
}


#' Fits the diagonal effects model, where each category has its own parameter delta[k],
#' while also incorporating a linear-by-linear term.
#'
#' @param n the matrix of observed counts
#' @param center should the linear-by-linear components be centered to have mean 0?
#' Default is TRUE
#' @returns a list containing
#'    beta: the regression parameters. delta parameters come after rows and columns
#'    and finally the linear-by-linear term
#'    g_squared: G^2 fit measure
#'    chisq: X^2 fit measure
#'    df: degrees of freedom
#'    expected: matrix of expected frequencies
#' @export
von_Eye_diagonal_linear_by_linear <- function(n, center=TRUE) {
  x <- log_linear_main_effect_design(n)
  x_1 <- log_linear_add_all_diagonals(n, x)
  x_linear <- log_linear_create_linear_by_linear(n)
  x_prime <- log_linear_append_column(x_1, x_linear)
  names <- paste0("delta_", 1:nrow(n))
  names <- c(names, "linear-by-linear")
  log_linear_fit(n, x_prime, effect_names=names)
}


#' Creates design matrix for weight be response category model.
#'
#' The model specifies main effects for row and column, and a parameter for
#' the agreement (diagonal) cells.  This takes a design matrix for that model and
#' applies domain-specific weights to the agreement parameters.
#' @param n the matrix of cell counts
#' @param x the original design matrix.
#' @param w the vector of weights to apply to the agreement cells. Should have same number
#' of entries as the number of diagonal elements (number of rows & of columns)
#' @param n_raters number of raters. Currently only 2 (the default) are supported. This is
#' an extension point for future work.
#' @returns new design matrix with weights applied to the agreement cells.
#' @export
von_Eye_weight_by_response_category_design <- function(n, x, w, n_raters=2) {
  if (length(w) != nrow(n)) {
    stop(paste0("length of w (", length(w), ") not equal to number of rows in n(", nrow(n), ")"))
  }

  column <- ncol(x)
  for (i in 1:nrow(n)) {
    index <- (i - 1) * ncol(n) + i
    x[index, column] <- w[i]
  }
  x
}
