#ifndef INTCALIB_HPP
#define INTCALIB_HPP

#include <RcppArmadillo.h>

using namespace arma;
using namespace Rcpp;


// Calculate the number of fitted targets outside the boundaries
int countOOB(const colvec& L, const colvec& U, const colvec& e) {
    int i, oob = 0;
    for (i = 0; (size_t) i < e.size(); i++) {
        if (L[i] > e[i] || U[i] < e[i]) oob++;
    }
    return oob;
}

#include "Losses/Losses.h"
#include "Penalties/Penalties.h"

template <typename Type> colvec IntProgCalib(const Type& A, const colvec& y, colvec& w, const colvec& ref,
                                             const vec& lower, const vec& upper, const mat& Bnds,
                                             const colvec& scale, const std::string lossType,
                                             const std::string penType, const double lambdaPen) {
    /* Integer Programming Calibration algorithm
     * INPUT:
     *   A, matrix of comodities per each farm
     *   y, vector of targets
     *   w, vector of initial weights
     *   ref, vector of weights as reference vector (used only on the penalty)
     *   lower, vector of lower bounds for Restricted weights
     *   upper, vector of lower bounds for Restricted weights
     *   Bnds, matrix of target-bounds centered for zeros point targets
     *   scale, vector of rescaling factors
     *   lossType, string indicating which loss function to optimize:
     *     * "L1", summation of absolute errors
     *     * "aL1", asymmetric summation of absolute errors
     *     * "rL1", summation of absolute relative errors
     *     * "LB1", summation of absolute errors if outside the boundaries
     *     * "rB1", summation of absolute relative errors if outside the boundaries
     *     * "L2", summation of square errors
     *     * "aL2", asymmetric summation of square errors
     *     * "rL2", summation of square relative errors
     *     * "LB2", summation of square errors if outside the boundaries
     *     * "rB2", summation of square relative errors if outside the boundaries
     *   penType, string indicating which penalty function to include in the objective function:
     *     * "l0norm", number of adjustments that are different from zero
     *     * "lasso", summation of the absolute value of the adjustments
     *     * "ridge", summation of the adjustments squared
     *     * "raking", using the sum of raking ratios
     *     * "minentropy", based on the minimum entropy
     *     * "quadrat", using the Euclidean distance
     *     * "quadmod", using the modified Euclidean distance
     *     * "hellinger", using the Hellinger distance
     *     * "mcp", using the minimax concave penalty
     *     * "scad", using the smoothly clipped absolute deviations
     *     * "relasso", using the summation of the absolute value of the relative adjustments 
     *     * "modrelasso", using the summation of the absolute value of the modified relative adjustments 
     *     * "rehuber", using the Huber loss on the relative adjustmnets 
     *     * "modrehuber", using the Huber loss on the modified relative adjustmnets 
     *   lambdaPen, a double floating point number used to rescale the value returned by the choosen penality function
     *
     * OUTPUT:
     *   w, vector of calibrated integer weights
     */
#if _DEBUG
#if _POSCHK
    int df, pos;
#endif // _POSCHK
    time_t now;
#endif // _DEBUG
    int i, j, ac = 0;
    colvec cc(w.size());
    colvec grad(w.size());
    colvec ee(y.size());
    colvec tmpE(y.size());
    double ffnew, ffcc;
    int oob, oobo;
    colvec s(y.size());
    colvec sp(w.size());
    umat ord, nord;
    colvec tau, lambda, iscale;

    int lt = 0;
    if(lossType == "L1") lt = LOSS_L1;
    if(lossType == "aL1") lt = LOSS_aL1;
    if(lossType == "rL1") lt = LOSS_rL1;
    if(lossType == "LB1") lt = LOSS_LB1;
    if(lossType == "rB1") lt = LOSS_rB1;
    if(lossType == "L2") lt = LOSS_L2;
    if(lossType == "aL2") lt = LOSS_aL2;
    if(lossType == "rL2") lt = LOSS_rL2;
    if(lossType == "LB2") lt = LOSS_LB2;
    if(lossType == "rB2") lt = LOSS_rB2;
    if(lossType == "LC") lt = LOSS_LOGCOSH;
    if(lossType == "aLC") lt = LOSS_ALOGCOSH;
    if(lossType == "rLC") lt = LOSS_RLOGCOSH;
    if(lossType == "SE") lt = LOSS_SUMEXP;
    if(lossType == "aSE") lt = LOSS_ASUMEXP;
    if(lossType == "rSE") lt = LOSS_RSUMEXP;

    int pn = 0;
    if(penType == "l0norm") pn = PEN_L0NORM;
    if(penType == "lasso")  pn = PEN_LASSO;
    if(penType == "ridge")  pn = PEN_RIDGE;
    if(penType == "raking") pn = PEN_RACK;
    if(penType == "minentropy") pn = PEN_MINENT;
    if(penType == "quadrat") pn = PEN_QUADRAT;
    if(penType == "quadmod") pn = PEN_QUADMOD;
    if(penType == "hellinger") pn = PEN_HELL;
    if(penType == "mcp") pn = PEN_MCP;
    if(penType == "scad") pn = PEN_SCAD;
    if(penType == "relasso") pn = PEN_RELASSO;
    if(penType == "modrelasso") pn = PEN_MODRELASSO;
    if(penType == "rehuber") pn = PEN_REHUBER;
    if(penType == "modrehuber") pn = PEN_MODREHUBER;

    /* Processing based on the loss function */
    switch (lt) {
    case LOSS_aL1: // bounds asymmetry factorLOSS_RLOGCOSH
    case LOSS_aL2:
    case LOSS_ALOGCOSH:
    case LOSS_ASUMEXP:
        tau = Bnds.col(1) / (Bnds.col(1) - Bnds.col(0));
        break;
    case LOSS_rL1: // scale factor for relative errors
    case LOSS_rL2:
    case LOSS_RLOGCOSH:
    case LOSS_RSUMEXP:
        iscale = 1.0 / scale;
        break;
    default:
        break;
    }

    /* Calculate the errors, count the number
     * of missed targets, and create a temporary
     * copy of the vector of weights. */
    ee = y - A * w;
    oobo = countOOB(Bnds.col(0), Bnds.col(1), ee);
    cc = w;
    /* Calculate the currnet Loss */
    switch (lt) {
    case LOSS_L1: // L1-norm minimization
        ffnew = L1::ff(ee);
        s = sign(ee);
        break;
    case LOSS_aL1: // Asymmetric L1-norm
        s = sign(ee);
        lambda = tau - (ee < 0.0);
        ffnew = aL1::ff(ee, lambda);
        break;
    case LOSS_rL1: // L1-norm with relative errors
        ffnew = rL1::ff(ee, iscale);
        s = sign(ee);
        break;
    case LOSS_LB1: // L1-norm outside the boundaries and zero within
        ffnew = LB1::ff(Bnds.col(0), Bnds.col(1), ee);
        s = conv_to<colvec>::from(ee > Bnds.col(1)) - conv_to<colvec>::from(ee < Bnds.col(0));
        break;
    case LOSS_rB1: // relative L1-norm outside the boundaries and zero within
        ffnew = rB1::ff(Bnds.col(0), Bnds.col(1), ee);
        s = conv_to<colvec>::from((ee > Bnds.col(1)) - (ee < Bnds.col(0)));
        break;
    case LOSS_L2: // L2-norm minimization
        ffnew = L2::ff(ee);
        break;
    case LOSS_aL2: // Asymmetric L2-norm
        lambda = abs(tau - (ee < 0.0));
        ffnew = aL2::ff(ee, lambda);
        break;
    case LOSS_rL2: // L2-norm with relative errors
        ffnew = rL2::ff(ee, iscale);
        break;
    case LOSS_LB2: // L2-norm outside the boundaries and zero within
        ffnew = LB2::ff(Bnds.col(0), Bnds.col(1), ee);
        break;
    case LOSS_rB2: // relative L2-norm outside the boundaries and zero within
        ffnew = rB2::ff(Bnds.col(0), Bnds.col(1), ee);
        break;
    case LOSS_LOGCOSH:
        ffnew = Logcosh::ff(ee);
        break;
    case LOSS_ALOGCOSH:
        lambda = abs(tau - (ee < 0.0));
        ffnew = aLogcosh::ff(ee, lambda);
        break;
    case LOSS_RLOGCOSH:
        ffnew = rLogcosh::ff(ee, iscale);
        break;
    case LOSS_SUMEXP:
        ffnew = Sumexp::ff(ee);
        break;
    case LOSS_ASUMEXP:
        lambda = abs(tau - (ee < 0.0));
        ffnew = aSumexp::ff(ee, lambda);
        break;
    case LOSS_RSUMEXP:
        ffnew = rSumexp::ff(ee, iscale);
        break;
    default:
        break;
    }
    /* Computing the current penalty */
    switch (pn) {
    case PEN_L0NORM:
        ffnew += lambdaPen * l0norm::pen(w, ref);
        break;
    case PEN_LASSO:
        ffnew += lambdaPen * lasso::pen(w, ref);
        break;
    case PEN_RIDGE:
        ffnew += lambdaPen * ridge::pen(w, ref);
        break;
    case PEN_RACK:
        ffnew += lambdaPen * raking::pen(w, ref);
        break;
    case PEN_MINENT:
        ffnew += lambdaPen * min_entropy::pen(w, ref);
        break;
    case PEN_QUADRAT:
        ffnew += lambdaPen * quadrat::pen(w, ref);
        break;
    case PEN_QUADMOD:
        ffnew += lambdaPen * quadmod::pen(w, ref);
        break;
    case PEN_HELL:
        ffnew += lambdaPen * Hellinger::pen(w, ref);
        break;
    case PEN_MCP:
        ffnew += mcp::pen(w, ref, lambdaPen);
        break;
    case PEN_SCAD:
        ffnew += scad::pen(w, ref, lambdaPen);
        break;
    case PEN_RELASSO:
        ffnew += lambdaPen * relasso::pen(w, ref);
        break;
    case PEN_MODRELASSO:
        ffnew += lambdaPen* modrelasso::pen(w, ref);
        break;
    case PEN_REHUBER:
        ffnew += lambdaPen * rehuber::pen(w, ref);
        break;
    case PEN_MODREHUBER:    
        ffnew += lambdaPen * modrehuber::pen(w, ref);
        break;
    default:
        break;
    }

#if _DEBUG
                now = std::time(0);
                Rprintf("%s", std::ctime(&now));
#if _DEBUG != 2
                Rprintf("M:%d, F:%f, ", oobo, ffnew);
#else // IF _DEBUG == 2 THEN
                Rprintf("\033[1mM:\033[31m%d\033[39;0m, \033[1mF:\033[32m%f\033[39;0m\n", oobo, ffnew);
#endif // _DEBUG == 2
#endif // _DEBUG

    // Repeat the following until there are no more changes on "w"
    do {
        /* Calculate the gradient, and determine
         * the optimal ordered positions. */
        switch (lt) {
        case LOSS_L1: // L1-norm minimization
            s = sign(ee);
            grad = L1::ffGrd(A, s);
            break;
        case LOSS_aL1: // Asymmetric L1-norm
            lambda = tau - (ee < 0.0);
            grad = aL1::ffGrd(A, lambda);
            break;
        case LOSS_rL1: // L1-norm with relative errors
            s = sign(ee);
            grad = rL1::ffGrd(A, s, iscale);
            break;
        case LOSS_LB1: // L1-norm outside the boundaries and zero within
            s = conv_to<colvec>::from(ee > Bnds.col(1)) - conv_to<colvec>::from(ee < Bnds.col(0));
            grad = LB1::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
            break;
        case LOSS_rB1: // relative L1-norm outside the boundaries and zero within
            s = conv_to<colvec>::from(ee > Bnds.col(1)) - conv_to<colvec>::from(ee < Bnds.col(0));
            grad = rB1::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
            break;
        case LOSS_L2: // L2-norm minimization
            if (ac == 0) grad = L2::ffGrd(A, ee);
            break;
        case LOSS_aL2: // Asymmetric L2-norm
            lambda = abs(tau - (ee < 0.0));
            grad = aL2::ffGrd(A, ee, lambda);
            break;
        case LOSS_rL2: // L2-norm with relative errors
            if (ac == 0) grad = rL2::ffGrd(A, ee, iscale);
            break;
        case LOSS_LB2: // L2-norm outside the boundaries and zero within
            if (ac == 0) grad = LB2::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
            break;
        case LOSS_rB2: // relative L2-norm outside the boundaries and zero within
            if (ac == 0) grad = rB2::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
            break;
        case LOSS_LOGCOSH:
            grad = Logcosh::ffGrd(A, ee);
            break;
        case LOSS_ALOGCOSH:
            lambda = abs(tau - (ee < 0.0));
            grad = aLogcosh::ffGrd(A, ee, lambda);
            break;
        case LOSS_RLOGCOSH:
            grad = rLogcosh::ffGrd(A, ee, iscale);
            break;
        case LOSS_SUMEXP:
            grad = Sumexp::ffGrd(A, ee);
            break;
        case LOSS_ASUMEXP:
            lambda = abs(tau - (ee < 0.0));
            grad = aSumexp::ffGrd(A, ee, lambda);
            break;
        case LOSS_RSUMEXP:
            grad = rSumexp::ffGrd(A, ee, iscale);
            break;
        default:
            grad.zeros();
            break;
        }
        switch (pn) {
        case PEN_L0NORM:
            grad += lambdaPen * l0norm::penGrd(w, ref);
            break;
        case PEN_LASSO:
            grad += lambdaPen * lasso::penGrd(w, ref);
            break;
        case PEN_RIDGE:
            grad += lambdaPen * ridge::penGrd(w, ref);
            break;
        case PEN_RACK:
            grad += lambdaPen * raking::penGrd(w, ref);
            break;
        case PEN_MINENT:
            grad += lambdaPen * min_entropy::penGrd(w, ref);
            break;
        case PEN_QUADRAT:
            grad += lambdaPen * quadrat::penGrd(w, ref);
            break;
        case PEN_QUADMOD:
            grad += lambdaPen * quadmod::penGrd(w, ref);
            break;
        case PEN_HELL:
            grad += lambdaPen * Hellinger::penGrd(w, ref);
            break;
        case PEN_MCP:
            grad += mcp::penGrd(w, ref, lambdaPen);
            break;
        case PEN_SCAD:
            grad += scad::penGrd(w, ref, lambdaPen);
            break;
        case PEN_RELASSO:
            grad += lambdaPen * relasso::pen(w, ref);
            break;
        case PEN_MODRELASSO:
            grad += lambdaPen* modrelasso::pen(w, ref);
            break;
        case PEN_REHUBER:
            grad += lambdaPen * rehuber::pen(w, ref);
            break;
        case PEN_MODREHUBER:    
            grad += lambdaPen * modrehuber::pen(w, ref);
            break;
        default:
            break;
        }
        ord = stable_sort_index(abs(grad), "descend");

        /* Reset the counter of any change in vector w */
        ac = 0;
#if _DEBUG
#if _POSCHK
        pos = 0;
#endif // _POSCHK
#endif // _DEBUG
        for (j = 0; (size_t) j < w.size(); j++) {
            R_CheckUserInterrupt();
            /* Consider the weights with the most effective reduction,
             * change them according to the gradient, and update
             * the values of the errors (which are store on a temporary vector). */
            i = ord[j];
            if (grad[i] == 0.0) continue;
            if (grad[i] > 0.0) {
                if (cc[i] < (lower[i] + 1.0)) continue; //{Rprintf("%d - w_i < 2\n", j); continue;}
                cc[i]--;
                tmpE = ee + A.col(i);
            }
            else {
                if (cc[i] > (upper[i] - 1.0)) continue; //{Rprintf("%d - w_i > 5\n", j); continue;}
                cc[i]++;
                tmpE = ee - A.col(i);
            }
            /* Counting the number of missed targets (Out Of Bounds),
             * and the new value of the loss function to optimize. */
            oob = countOOB(Bnds.col(0), Bnds.col(1), tmpE);
            switch (lt) {
            case LOSS_L1:
                ffcc = L1::ff(tmpE);
                break;
            case LOSS_aL1:
                lambda = tau - (tmpE < 0.0);
                ffcc = aL1::ff(tmpE, lambda);
                break;
            case LOSS_rL1:
                ffcc = rL1::ff(tmpE, iscale);
                break;
            case LOSS_LB1:
                ffcc = LB1::ff(Bnds.col(0), Bnds.col(1), tmpE);
                break;
            case LOSS_rB1:
                ffcc = rB1::ff(Bnds.col(0), Bnds.col(1), tmpE);
                break;
            case LOSS_L2:
                ffcc = L2::ff(tmpE);
                break;
            case LOSS_aL2:
                lambda = abs(tau - (tmpE < 0.0));
                ffcc = aL2::ff(tmpE, lambda);
                break;
            case LOSS_rL2:
                ffcc = rL2::ff(tmpE, iscale);
                break;
            case LOSS_LB2:
                ffcc = LB2::ff(Bnds.col(0), Bnds.col(1), tmpE);
                break;
            case LOSS_rB2:
                ffcc = rB2::ff(Bnds.col(0), Bnds.col(1), tmpE);
                break;
            case LOSS_LOGCOSH:
                ffcc = Logcosh::ff(tmpE);
                break;
            case LOSS_ALOGCOSH:
                lambda = abs(tau - (tmpE < 0.0));
                ffcc = aLogcosh::ff(tmpE, lambda);
                break;
            case LOSS_RLOGCOSH:
                ffcc = rLogcosh::ff(tmpE, iscale);
                break;
            case LOSS_SUMEXP:
                ffcc = Sumexp::ff(tmpE);
                break;
            case LOSS_ASUMEXP:
                lambda = abs(tau - (tmpE < 0.0));
                ffcc = aSumexp::ff(tmpE, lambda);
                break;
            case LOSS_RSUMEXP:
                ffcc = rSumexp::ff(tmpE, iscale);
                break;
            default:
                break;
            }
            switch (pn) {
                case PEN_L0NORM:
                    ffcc += lambdaPen * l0norm::pen(w, ref);
                    break;
                case PEN_LASSO:
                    ffcc += lambdaPen * lasso::pen(w, ref);
                    break;
                case PEN_RIDGE:
                    ffcc += lambdaPen * ridge::pen(w, ref);
                    break;
                case PEN_RACK:
                    ffcc += lambdaPen * raking::pen(w, ref);
                    break;
                case PEN_MINENT:
                    ffcc += lambdaPen * min_entropy::pen(w, ref);
                    break;
                case PEN_QUADRAT:
                    ffcc += lambdaPen * quadrat::pen(w, ref);
                    break;
                case PEN_QUADMOD:
                    ffcc += lambdaPen * quadmod::pen(w, ref);
                    break;
                case PEN_HELL:
                    ffcc += lambdaPen * Hellinger::pen(w, ref);
                    break;
                case PEN_MCP:
                    ffcc += mcp::pen(w, ref, lambdaPen);
                    break;
                case PEN_SCAD:
                    ffcc += scad::pen(w, ref, lambdaPen);
                    break;
                case PEN_RELASSO:
                    ffcc += lambdaPen * relasso::pen(w, ref);
                    break;
                case PEN_MODRELASSO:
                    ffcc += lambdaPen* modrelasso::pen(w, ref);
                    break;
                case PEN_REHUBER:
                    ffcc += lambdaPen * rehuber::pen(w, ref);
                    break;
                case PEN_MODREHUBER:    
                    ffcc += lambdaPen * modrehuber::pen(w, ref);
                    break;
                default:
                    break;
            }
            /* If both the loss function and the number of missed
             * targets are reduced, then the old point is changed,
             * the gradient is updated and the algorithm starts
             * from the beginning until convergence (based on the
             * number of changes). */
            if ((ffcc < ffnew) && OFF_WHEN_BND(oob <= oobo, lt)) {// -1 instead of lt ???
#if _DEBUG
                now = std::time(0);
#if _POSCHK
                df = j - pos;
#endif
#if _DEBUG != 2
                Rprintf("M:%d, F:%f, ", oob, ffcc);
                Rprintf("S:%s # ", grad[i] > 0.0 ? "-" : "+");
#if _POSCHK
                Rprintf("[after %d in %dG of %dW] # ", df, j, i);
#endif
#else // IF _DEBUG == 2 THEN
                Rprintf("\033[1mM:\033[31m%d\033[39;0m, \033[1mF:\033[32m%f\033[39;0m, ", oob, ffcc);
                Rprintf("S:\033[34m%s\033[39;0m # ", grad[i] > 0.0 ? "-" : "+");
#if _POSCHK
                Rprintf("[after %d in %dG of %dW] # ", df, j, i);
#endif
#endif // _DEBUG == 2
                Rprintf("%s", std::ctime(&now));
#endif // _DEBUG
                ac++; // Count how many changes of the weigths are considered
                ffnew = ffcc;
                w[i] = cc[i];
                ee = tmpE;
                switch (lt) {
                case LOSS_L1:
                    j = L1::updategrd(A, s, ee, grad, ord, j);
                    s = sign(ee);
                    break;
                case LOSS_aL1:
                    lambda = abs(tau - (ee < 0.0));
                    grad = aL1::ffGrd(A, lambda);
                    break;
                case LOSS_rL1:
                    j = rL1::updategrd(A, iscale, s, ee, grad, ord, j);
                    s = sign(ee);
                    break;
                case LOSS_LB1:
                    j = LB1::updategrd(A, Bnds, s, ee, grad, ord, j);
                    s = conv_to<colvec>::from(ee > Bnds.col(1)) - conv_to<colvec>::from(ee < Bnds.col(0));
                    break;
                case LOSS_rB1:
                    j = rB1::updategrd(A, Bnds, s, ee, grad, ord, j);
                    s = conv_to<colvec>::from(ee > Bnds.col(1)) - conv_to<colvec>::from(ee < Bnds.col(0));
                    break;
                case LOSS_L2:
                    grad = L2::ffGrd(A, ee);
                    break;
                case LOSS_aL2:
                    lambda = abs(tau - (ee < 0.0));
                    grad = aL2::ffGrd(A, ee, lambda);
                    break;
                case LOSS_rL2:
                    grad = rL2::ffGrd(A, ee, iscale);
                    break;
                case LOSS_LB2:
                    grad = LB2::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
                    break;
                case LOSS_rB2:
                    grad = rB2::ffGrd(A, ee, Bnds.col(0), Bnds.col(1));
                    break;
                case LOSS_LOGCOSH:
                    grad = Logcosh::ffGrd(A, ee);
                    break;
                case LOSS_ALOGCOSH:
                    lambda = abs(tau - (ee < 0.0));
                    grad = aLogcosh::ffGrd(A, ee, lambda);
                    break;
                case LOSS_RLOGCOSH:
                    grad = rLogcosh::ffGrd(A, ee, iscale);
                    break;
                case LOSS_SUMEXP:
                    grad = Sumexp::ffGrd(A, ee);
                    break;
                case LOSS_ASUMEXP:
                    lambda = abs(tau - (ee < 0.0));
                    grad = aSumexp::ffGrd(A, ee, lambda);
                    break;
                case LOSS_RSUMEXP:
                    grad = rSumexp::ffGrd(A, ee, iscale);
                    break;
                default:
                    break;
                }
                switch (pn) {
                case PEN_L0NORM:
                    grad += lambdaPen * l0norm::penGrd(w, ref);
                    break;
                case PEN_LASSO:
                    grad += lambdaPen * lasso::penGrd(w, ref);
                    break;
                case PEN_RIDGE:
                    grad += lambdaPen * ridge::penGrd(w, ref);
                    break;
                case PEN_RACK:
                    grad += lambdaPen * raking::penGrd(w, ref);
                    break;
                case PEN_MINENT:
                    grad += lambdaPen * min_entropy::penGrd(w, ref);
                    break;
                case PEN_QUADRAT:
                    grad += lambdaPen * quadrat::penGrd(w, ref);
                    break;
                case PEN_QUADMOD:
                    grad += lambdaPen * quadmod::penGrd(w, ref);
                    break;
                case PEN_HELL:
                    grad += lambdaPen * Hellinger::penGrd(w, ref);
                    break;
                case PEN_MCP:
                    grad += mcp::penGrd(w, ref, lambdaPen);
                    break;
                case PEN_SCAD:
                    grad += scad::penGrd(w, ref, lambdaPen);
                    break;
                case PEN_RELASSO:
                    grad += lambdaPen * relasso::pen(w, ref);
                    break;
                case PEN_MODRELASSO:
                    grad += lambdaPen* modrelasso::pen(w, ref);
                    break;
                case PEN_REHUBER:
                    grad += lambdaPen * rehuber::pen(w, ref);
                    break;
                case PEN_MODREHUBER:    
                    grad += lambdaPen * modrehuber::pen(w, ref);
                    break;
                default:
                    break;
                }
                if (!IS_L1BASED(lt) || IS_PENALIZED(pn)) { // if the objective function is based on L2-norm (or L0-norm)
                    nord = stable_sort_index(abs(grad), "descend"); //sort the gradient
                    if (any(conv_to<vec>::from(nord != ord))) { // and calculate the position according to the changes
                        j = -1;
                        ord = nord;
                    }
                }
                if (oob < oobo) oobo = oob;
#if _DEBUG
#if _POSCHK
                if (j < 0) {
                    pos ^= pos;
                }
                else {
                    pos = j;
                }
#endif // _POSCHK
#endif // _DEBUG
            }
            else {
                cc[i] = w[i];
            }
        }
    } while(ac);
#if _DEBUG
    now = std::time(0);
    Rprintf("%s", std::ctime(&now));
#endif
    return w;
}

#endif // INTCALIB_HPP
