% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R, R/toad.R
\name{toad-model}
\alias{toad-model}
\alias{toadSim}
\alias{toadStat}
\alias{toad}
\title{Marchand et al.'s toad movement model}
\usage{
toadSim(theta, data)

toadStat(
  data,
  lags = c(1, 2, 4, 8),
  q = c(0.01, seq(0.05, 0.95, by = 0.05), 0.99),
  dret = 10
)
}
\arguments{
\item{theta}{a vector of length 3 containing the model parameters}

\item{data}{a (number of days)x(number of toads) numerical matrix
containing the locations of a set of toads. The simulated
dataset will replicates the size, the NA pattern and
the initial positions (first row) of this argument.}

\item{lags}{an integer vector giving the lags  used to compute the statistic}

\item{q}{a numeric vector specifying the desired quantiles}

\item{dret}{the distance used to classify the displacements as "returns" or
"non-returns".}
}
\value{
\code{toadSim} return a numerical matrix of the same size of \code{data}

\code{toadStat} returns a numeric vector of length
\verb{length(lags)x(1+length(q))}. The summary statistic is computed
from absolute displacements at time lags
classified as "returns" or "non-returns"
depending on whether they are smaller than \code{dret}.
For each lag in \code{lags}, the statistic comprises the "return" frequency
and the median and adjacent quantile differences of the logarithms of
the "non-return" distances (the quantiles are those specified in \code{q}).
}
\description{
Function \code{toadSim} (written in C++) simulates a realization from
the "model 1" suggested by Marchand et al. (2017);
function \code{toadStat} computes a possible summary statistics.
}
\details{
The model describes the nocturnal movements (along a single spatial
dimension) of Fowler's toads.
It assumes that toads leave their refuges at night to forage and
hide within sand dunes during the day. After the t-th nocturnal
foraging phase, a toad is located at a displacement of \eqn{\Delta_t} meters
from its previous refuge site. The
displacements \eqn{\Delta_t} are assumed to be independent and identically
distributed realizations of a symmetric, zero-centered \eqn{\alpha}-stable
random variable with stability parameter \eqn{\alpha} and scale parameter
\eqn{\gamma}. The toad then either returns to one of the previously visited
sites (with probability \eqn{\pi}) or remains at its current location (with
probability \eqn{1-\pi}). In the former case, the refuge site is selected at
random. The model parameter vector is
\eqn{\theta=(\alpha, \gamma, \pi)'}.
}
\examples{
\donttest{
# It takes some time to run
set.seed(20251025L)
tobs <- toadStat(toads)
tsim <- function(theta) toadStat(toadSim(theta, toads))
m <- ifit(tobs, tsim, l = c(0.01, 0, 0), u = c(2, 100, 1), trace = 1000)
m
confint(m)
numsimIFIT(m)
estfunIFIT(m)
diagIFIT(m)
}
}
\references{
Philippe Marchand, Morgan Boenke, and David M. Green (2017)
‘A Stochastic Movement Model Reproduces Patterns of Site Fidelity and
Long-Distance Dispersal in a Population of Fowler’s Toads (Anaxyrus
Fowleri)’,  Ecological Modelling, 360,  pp 63–69,
\doi{10.1016/j.ecolmodel.2017.06.025}.
}
