% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/realizedMeasures.R
\name{rThresholdCov}
\alias{rThresholdCov}
\title{Threshold Covariance}
\usage{
rThresholdCov(
  rData,
  cor = FALSE,
  alignBy = NULL,
  alignPeriod = NULL,
  makeReturns = FALSE,
  ...
)
}
\arguments{
\item{rData}{an \code{xts} or \code{data.table} object containing returns or prices, possibly for multiple assets over multiple days.}

\item{cor}{boolean, in case it is \code{TRUE}, and the input data is multivariate, the correlation is returned instead of the covariance matrix. \code{FALSE} by default.}

\item{alignBy}{character, indicating the time scale in which \code{alignPeriod} is expressed. 
Possible values are: \code{"ticks"}, \code{"secs"}, \code{"seconds"}, \code{"mins"}, \code{"minutes"}, \code{"hours"}}

\item{alignPeriod}{positive numeric, indicating the number of periods to aggregate over. For example, to aggregate
based on a 5-minute frequency, set \code{alignPeriod = 5} and \code{alignBy = "minutes"}.}

\item{makeReturns}{boolean, should be \code{TRUE} when \code{rData} contains prices instead of returns. 
\code{FALSE} by default.}

\item{...}{used internally, do not change.}
}
\value{
in case the input is and contains data from one day, an \eqn{N \times N} matrix is returned. 
If the data is a univariate \code{xts} object with multiple days, an \code{xts} is returned.
If the data is multivariate and contains multiple days (\code{xts} or \code{data.table}), 
the function returns a list containing \eqn{N \times N} matrices. 
Each item in the list has a name which corresponds to the date for the matrix.
}
\description{
Calculate the threshold covariance matrix proposed in Gobbi and Mancini (2009).
Unlike the \code{\link{rOWCov}}, the rThresholdCov uses univariate jump detection rules to truncate the effect of jumps on the covariance
estimate. As such, it remains feasible in high dimensions, but it is less robust to small cojumps.

Let \eqn{r_{t,i}} be an intraday \eqn{N x 1} return vector of \eqn{N} assets where \eqn{i=1,...,M} and
\eqn{M} being the number of intraday returns.

Then, the \eqn{k,q}-th element of the threshold covariance matrix is defined as

\deqn{
\mbox{thresholdcov}[k,q]_{t} = \sum_{i=1}^{M} r_{(k)t,i} 1_{\{r_{(k)t,i}^2 \leq TR_{M}\}}  \ \ r_{(q)t,i} 1_{\{r_{(q)t,i}^2 \leq TR_{M}\}},
}
with the threshold value \eqn{TR_{M}} set to \eqn{3 \Delta^{0.49}} times the daily realized bi-power variation of asset \eqn{k},
as suggested in Jacod and Todorov (2009).
}
\examples{
# Realized threshold  Variance/Covariance:
# Multivariate:
\dontrun{
library("xts")
set.seed(123)
start <- strptime("1970-01-01", format = "\%Y-\%m-\%d", tz = "UTC")
timestamps <- start + seq(34200, 57600, length.out = 23401)

dat <- cbind(rnorm(23401) * sqrt(1/23401), rnorm(23401) * sqrt(1/23401))

dat <- exp(cumsum(xts(dat, timestamps)))
rcThreshold <- rThresholdCov(dat, alignBy = "minutes", alignPeriod = 1, makeReturns = TRUE)
rcThreshold
}
}
\references{
Barndorff-Nielsen, O. and Shephard, N. (2004). Measuring the impact of jumps in multivariate price processes using bipower covariation. Discussion paper, Nuffield College, Oxford University.

Jacod, J. and Todorov, V. (2009). Testing for common arrival of jumps in discretely-observed multidimensional processes. \emph{Annals of Statistics}, 37, 1792-1838.

Mancini, C. and Gobbi, F. (2012). Identifying the Brownian covariation from the co-jumps given discrete observations. \emph{Econometric Theory}, 28, 249-273.
}
\seealso{
\code{\link{ICov}} for a list of implemented estimators of the integrated covariance.
}
\author{
Jonathan Cornelissen, Kris Boudt, and Emil Sjoerup.
}
\keyword{volatility}
