% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/4_1_calibrate.R
\name{calibrate}
\alias{calibrate}
\title{Calibrate high-dimensional Cox models}
\usage{
calibrate(
  x,
  time,
  event,
  model.type = c("lasso", "alasso", "flasso", "enet", "aenet", "mcp", "mnet", "scad",
    "snet"),
  alpha,
  lambda,
  pen.factor = NULL,
  gamma,
  lambda1,
  lambda2,
  method = c("fitting", "bootstrap", "cv", "repeated.cv"),
  boot.times = NULL,
  nfolds = NULL,
  rep.times = NULL,
  pred.at,
  ngroup = 5,
  seed = 1001,
  trace = TRUE
)
}
\arguments{
\item{x}{Matrix of training data used for fitting the model;
on which to run the calibration.}

\item{time}{Survival time.
Must be of the same length with the number of rows as \code{x}.}

\item{event}{Status indicator, normally 0 = alive, 1 = dead.
Must be of the same length with the number of rows as \code{x}.}

\item{model.type}{Model type to calibrate. Could be one of \code{"lasso"},
\code{"alasso"}, \code{"flasso"}, \code{"enet"}, \code{"aenet"},
\code{"mcp"}, \code{"mnet"}, \code{"scad"}, or \code{"snet"}.}

\item{alpha}{Value of the elastic-net mixing parameter alpha for
\code{enet}, \code{aenet}, \code{mnet}, and \code{snet} models.
For \code{lasso}, \code{alasso}, \code{mcp}, and \code{scad} models,
please set \code{alpha = 1}.
\code{alpha=1}: lasso (l1) penalty; \code{alpha=0}: ridge (l2) penalty.
Note that for \code{mnet} and \code{snet} models,
\code{alpha} can be set to very close to 0 but not 0 exactly.}

\item{lambda}{Value of the penalty parameter lambda to use in the
model fits on the resampled data. From the Cox model you have built.}

\item{pen.factor}{Penalty factors to apply to each coefficient.
From the built \emph{adaptive lasso} or \emph{adaptive elastic-net} model.}

\item{gamma}{Value of the model parameter gamma for
MCP/SCAD/Mnet/Snet models.}

\item{lambda1}{Value of the penalty parameter lambda1 for fused lasso model.}

\item{lambda2}{Value of the penalty parameter lambda2 for fused lasso model.}

\item{method}{Calibration method.
Options including \code{"fitting"}, \code{"bootstrap"}, \code{"cv"},
and \code{"repeated.cv"}.}

\item{boot.times}{Number of repetitions for bootstrap.}

\item{nfolds}{Number of folds for cross-validation and
repeated cross-validation.}

\item{rep.times}{Number of repeated times for repeated cross-validation.}

\item{pred.at}{Time point at which calibration should take place.}

\item{ngroup}{Number of groups to be formed for calibration.}

\item{seed}{A random seed for resampling.}

\item{trace}{Logical. Output the calibration progress or not.
Default is \code{TRUE}.}
}
\description{
Calibrate high-dimensional Cox models
}
\examples{
data("smart")
x <- as.matrix(smart[, -c(1, 2)])
time <- smart$TEVENT
event <- smart$EVENT
y <- survival::Surv(time, event)

# Fit Cox model with lasso penalty
fit <- fit_lasso(x, y, nfolds = 5, rule = "lambda.1se", seed = 1001)

# Model calibration by fitting the original data directly
cal.fitting <- calibrate(
  x, time, event,
  model.type = "lasso",
  alpha = 1, lambda = fit$lambda,
  method = "fitting",
  pred.at = 365 * 9, ngroup = 5,
  seed = 1010
)

# Model calibration by 5-fold cross-validation
cal.cv <- calibrate(
  x, time, event,
  model.type = "lasso",
  alpha = 1, lambda = fit$lambda,
  method = "cv", nfolds = 5,
  pred.at = 365 * 9, ngroup = 5,
  seed = 1010
)

print(cal.fitting)
summary(cal.fitting)
plot(cal.fitting)

print(cal.cv)
summary(cal.cv)
plot(cal.cv)

# # Test fused lasso, SCAD, and Mnet models
# data(smart)
# x = as.matrix(smart[, -c(1, 2)])[1:500, ]
# time = smart$TEVENT[1:500]
# event = smart$EVENT[1:500]
# y = survival::Surv(time, event)
#
# set.seed(1010)
# cal.fitting = calibrate(
#   x, time, event, model.type = "flasso",
#   lambda1 = 5, lambda2 = 2,
#   method = "fitting",
#   pred.at = 365 * 9, ngroup = 5,
#   seed = 1010)
#
# cal.boot = calibrate(
#   x, time, event, model.type = "scad",
#   gamma = 3.7, alpha = 1, lambda = 0.03,
#   method = "bootstrap", boot.times = 10,
#   pred.at = 365 * 9, ngroup = 5,
#   seed = 1010)
#
# cal.cv = calibrate(
#   x, time, event, model.type = "mnet",
#   gamma = 3, alpha = 0.3, lambda = 0.03,
#   method = "cv", nfolds = 5,
#   pred.at = 365 * 9, ngroup = 5,
#   seed = 1010)
#
# cal.repcv = calibrate(
#   x, time, event, model.type = "flasso",
#   lambda1 = 5, lambda2 = 2,
#   method = "repeated.cv", nfolds = 5, rep.times = 3,
#   pred.at = 365 * 9, ngroup = 5,
#   seed = 1010)
#
# print(cal.fitting)
# summary(cal.fitting)
# plot(cal.fitting)
#
# print(cal.boot)
# summary(cal.boot)
# plot(cal.boot)
#
# print(cal.cv)
# summary(cal.cv)
# plot(cal.cv)
#
# print(cal.repcv)
# summary(cal.repcv)
# plot(cal.repcv)
}
