% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gremlin.R
\docType{package}
\name{gremlin-package}
\alias{gremlin-package}
\title{Mixed-Effects REML Incorporating Generalized Inverses}
\description{
Fit linear mixed-effects models using restricted (or residual) maximum
likelihood (REML) and with generalized inverse matrices to specify covariance
structures for random effects. In particular, the package is suited to fit
quantitative genetic mixed models, often referred to as 'animal models'
(Henderson 1973). Implements the average information algorithm (Johnson &
Thompson 1995; Gilmour et al. 1995; Meyer & Smith 1996) as the main tool to
maximize the restricted log-likelihood, but with other algorithms available.
}
\details{
The average information algorithm combined with sparse matrix techniques can
potentially make model fitting very efficient.
}
\examples{
\dontrun{
  # Following the example from Mrode 2005, chapter 11.
  library(nadiv)  #<-- to construct inverse of the numerator relatedness matrix
  pedMrode11 <- prepPed(Mrode11[, 1:3])
  Ainv <- makeAinv(pedMrode11)$Ainv

  gr11lmm <- gremlin(WWG11 ~ sex - 1,
random = ~ calf,
data = Mrode11,
ginverse = list(calf = Ainv),
Gstart = matrix(0.2), Rstart = matrix(0.4),  #<-- specify starting values
maxit = 15,    #<-- maximum iterations
     v = 2, vit = 1,  #<-- moderate screen output (`v`) every iteration (`vit`)
     algit = "AI")  #<-- only use Average Information algorithm iterations
  summary(gr11lmm)

  # Compare the model to a Linear Model with no random effects
  ## Use `update()` to update the model
  gr11lm <- update(gr11lmm, random = ~ 1)  #<-- `~1`=drop all random effects
  summary(gr11lm)

  # Do analysis of variance between the two models
  ## See AIC or evaluate likelihood ratio against a Chi-squared distribution
  anova(gr11lm, gr11lmm)
}
}
\references{
Henderson, C.R. 1973. Sire evaluation and genetic trends. Journal of Animal
  Science 1973:10-41. 
  
  Johnson, D.L. and R. Thompson. 1995. Restricted maximum likelihood
  estimation of variance components for univariate animal models using sparse
  matrix techniques and average information. Journal of Dairy Science
  78:449-456.

  Gilmour, A.R., R. Thompson, and B.R. Cullis. 1995. Average information
  REML: An efficient algorithm for variance parameter estimation in linear
  mixed models. Biometrics 51:1440-1450.

  Meyer, K. and S.P. Smith. 1996. Restricted maximum likelihood estimation for
  animal models using derivatives of the likelihood. Genetics, Selection, and
  Evolution 28:23-49.

  Mrode, R.A. 2005. Linear Models for the Prediction of Animal Breeding
  Values, 2nd ed. CABI Publishing, Cambridge, MA, USA.
}
\seealso{
Useful links:
\itemize{
  \item \url{https://github.com/matthewwolak/gremlin}
  \item Report bugs at \url{https://github.com/matthewwolak/gremlin/issues}
}

}
\author{
\strong{Maintainer}: Matthew Wolak \email{matthewwolak@gmail.com} (\href{https://orcid.org/0000-0002-7962-0071}{ORCID})

}
