% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{t_running_sd3}
\alias{t_running_sd3}
\alias{t_running_skew4}
\alias{t_running_kurt5}
\alias{t_running_sd}
\alias{t_running_skew}
\alias{t_running_kurt}
\alias{t_running_cent_moments}
\alias{t_running_std_moments}
\alias{t_running_cumulants}
\title{Compute first K moments over a sliding time-based window}
\usage{
t_running_sd3(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_skew4(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_kurt5(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_sd(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_skew(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_kurt(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_cent_moments(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  max_order = 5L,
  na_rm = FALSE,
  max_order_only = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_std_moments(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

t_running_cumulants(
  v,
  time = NULL,
  time_deltas = NULL,
  window = NULL,
  wts = NULL,
  lb_time = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  variable_win = FALSE,
  wts_as_delta = TRUE,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)
}
\arguments{
\item{v}{a vector of data.}

\item{time}{an optional vector of the timestamps of \code{v}. If given, must be
the same length as \code{v}. If not given, we try to infer it by summing the
\code{time_deltas}.}

\item{time_deltas}{an optional vector of the deltas of timestamps. If given, must be
the same length as \code{v}. If not given, and \code{wts} are given and \code{wts_as_delta} is true,
we take the \code{wts} as the time deltas.  The deltas must be positive. We sum them to arrive
at the times.}

\item{window}{the window size, in time units. if given as finite integer or double, passed through.
If \code{NULL}, \code{NA_integer_}, \code{NA_real_} or \code{Inf} are given, 
 and \code{variable_win} is true, then we infer the window from the lookback times: the
 first window is infinite, but the remaining is the deltas between lookback times.
 If \code{variable_win} is false, then these undefined values are equivalent to an
 infinite window.
 If negative, an error will be thrown.}

\item{wts}{an optional vector of weights. Weights are \sQuote{replication}
weights, meaning a value of 2 is shorthand for having two observations
with the corresponding \code{v} value. If \code{NULL}, corresponds to
equal unit weights, the default. Note that weights are typically only meaningfully defined
up to a multiplicative constant, meaning the units of weights are
immaterial, with the exception that methods which check for minimum df will,
in the weighted case, check against the sum of weights. For this reason,
weights less than 1 could cause \code{NA} to be returned unexpectedly due
to the minimum condition. When weights are \code{NA}, the same rules for checking \code{v}
are applied. That is, the observation will not contribute to the moment
if the weight is \code{NA} when \code{na_rm} is true. When there is no
checking, an \code{NA} value will cause the output to be \code{NA}.}

\item{lb_time}{a vector of the times from which lookback will be performed. The output should
be the same size as this vector. If not given, defaults to \code{time}.}

\item{na_rm}{whether to remove NA, false by default.}

\item{min_df}{the minimum df to return a value, otherwise \code{NaN} is returned.
This can be used to prevent moments from being computed on too few observations.
Defaults to zero, meaning no restriction.}

\item{used_df}{the number of degrees of freedom consumed, used in the denominator
of the centered moments computation. These are subtracted from the number of
observations.}

\item{restart_period}{the recompute period. because subtraction of elements can cause
loss of precision, the computation of moments is restarted periodically based on 
this parameter. Larger values mean fewer restarts and faster, though less accurate
results.}

\item{variable_win}{if true, and the \code{window} is not a concrete number,
the computation window becomes the time between lookback times.}

\item{wts_as_delta}{if true and the \code{time} and \code{time_deltas} are not
given, but \code{wts} are given, we take \code{wts} as the \code{time_deltas}.}

\item{check_wts}{a boolean for whether the code shall check for negative
weights, and throw an error when they are found. Default false for speed.}

\item{normalize_wts}{a boolean for whether the weights should be
renormalized to have a mean value of 1. This mean is computed over elements
which contribute to the moments, so if \code{na_rm} is set, that means non-NA
elements of \code{wts} that correspond to non-NA elements of the data
vector.}

\item{check_negative_moments}{a boolean flag. Normal computation of running
moments can result in negative estimates of even order moments due to loss of
numerical precision. With this flag active, the computation checks for negative
even order moments and restarts the computation when one is detected. This
should eliminate the possibility of negative even order moments. The
downside is the speed hit of checking on every output step. Note also the
code checks for negative moments of every even order tracked, even if they
are not output; that is if the kurtosis, say, is being computed, and a
negative variance is detected, then the computation is restarted.
Defaults to \code{TRUE} to avoid negative even moments. Set to \code{FALSE}
only if you know what you are doing.}

\item{max_order}{the maximum order of the centered moment to be computed.}

\item{max_order_only}{for \code{running_cent_moments}, if this flag is set, only compute
the maximum order centered moment, and return in a vector.}
}
\value{
Typically a matrix, where the first columns are the kth, k-1th through 2nd standardized, 
centered moments, then a column of the mean, then a column of the number of (non-nan) elements in the input,
with the following exceptions:
\describe{
\item{t_running_cent_moments}{Computes arbitrary order centered moments. When \code{max_order_only} is set,
only a column of the maximum order centered moment is returned.}
\item{t_running_std_moments}{Computes arbitrary order standardized moments, then the standard deviation, the mean,
and the count. There is not yet an option for \code{max_order_only}, but probably should be.}
\item{t_running_cumulants}{Computes arbitrary order cumulants, and returns the kth, k-1th, through the second 
(which is the variance) cumulant, then the mean, and the count.}
}
}
\description{
Compute the (standardized) 2nd through kth moments, the mean, and the number of elements over
an infinite or finite sliding time based window, returning a matrix.
}
\details{
Computes the number of elements, the mean, and the 2nd through kth
centered (and typically standardized) moments, for \eqn{k=2,3,4}{k=2,3,4}. These
are computed via the numerically robust one-pass method of Bennett \emph{et. al.}

Given the length \eqn{n} vector \eqn{x}, we output matrix \eqn{M} where
\eqn{M_{i,j}}{M_i,j} is the \eqn{order - j + 1} moment (\emph{i.e.}
excess kurtosis, skewness, standard deviation, mean or number of elements)
of some elements \eqn{x_i} defined by the sliding time window.
Barring \code{NA} or \code{NaN}, this is over a window of time width \code{window}.
}
\note{
the kurtosis is \emph{excess kurtosis}, with a 3 subtracted, and should be nearly zero
for Gaussian input.

The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.

Note that when weights are given, they are treated as replication weights.
This can have subtle effects on computations which require minimum
degrees of freedom, since the sum of weights will be compared to
that minimum, not the number of data points. Weight values
(much) less than 1 can cause computations to return \code{NA}
somewhat unexpectedly due to this condition, while values greater
than one might cause the computation to spuriously return a value
with little precision.

As this code may add and remove observations, numerical imprecision
may result in negative estimates of squared quantities, like the
second or fourth moments.  By default we check for this condition
in running computations. It may also be mitigated somewhat by setting 
a smaller \code{restart_period}. Post an issue if you experience this bug.
}
\section{Time Windowing }{


This function supports time (or other counter) based running computation. 
Here the input are the data \eqn{x_i}, and optional weights vectors, \eqn{w_i}, defaulting to 1,
and a vector of time indices, \eqn{t_i} of the same length as \eqn{x}. The
times must be non-decreasing:
\deqn{t_1 \le t_2 \le \ldots}{t_1 <= t_2 <= ...}
It is assumed that \eqn{t_0 = -\infty}.
The window, \eqn{W} is now a time-based window. 
An optional set of \emph{lookback times} are also given, \eqn{b_j}, which
may have different length than the \eqn{x} and \eqn{w}. 
The output will correspond to the lookback times, and should be the same
length. The \eqn{j}th output is computed over indices \eqn{i} such that
\deqn{b_j - W < t_i \le b_j.}

For comparison functions (like Z-score, rescaling, centering), which compare
values of \eqn{x_i} to local moments, the lookbacks may not be given, but
a lookahead \eqn{L} is admitted. In this case, the \eqn{j}th output is computed over
indices \eqn{i} such that
\deqn{t_j - W + L < t_i \le t_j + L.}

If the times are not given, \sQuote{deltas} may be given instead. If
\eqn{\delta_i} are the deltas, then we compute the times as
\deqn{t_i = \sum_{1 \le j \le i} \delta_j.}
The deltas must be the same length as \eqn{x}.  
If times and deltas are not given, but weights are given and the \sQuote{weights as deltas}
flag is set true, then the weights are used as the deltas.

Some times it makes sense to have the computational window be the space
between lookback times. That is, the \eqn{j}th output is to be
computed over indices \eqn{i} such that
\deqn{b_{j-1} - W < t_i \le b_j.}
This can be achieved by setting the \sQuote{variable window} flag true
and setting the window to null. This will not make much sense if
the lookback times are equal to the times, since each moment computation
is over a set of a single index, and most moments are underdefined.
}

\examples{
x <- rnorm(1e5)
xs3 <- t_running_sd3(x,time=seq_along(x),window=10)
xs4 <- t_running_skew4(x,time=seq_along(x),window=10)
# but what if you only cared about some middle values?
xs4 <- t_running_skew4(x,time=seq_along(x),lb_time=(length(x) / 2) + 0:10,window=20)

}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{https://web.archive.org/web/20140423031833/http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\doi{10.1109/CLUSTR.2009.5289161}

Cook, J. D. "Accurately computing running variance."
\url{https://www.johndcook.com/standard_deviation/}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{https://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation/}
}
\seealso{
\code{\link{running_sd3}}.
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
