% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fregre.gsam.R
\name{fregre.gsam}
\alias{fregre.gsam}
\title{Fitting Functional Generalized Spectral Additive Models}
\usage{
fregre.gsam(
  formula,
  family = gaussian(),
  data = list(),
  weights = NULL,
  basis.x = NULL,
  basis.b = NULL,
  ...
)
}
\arguments{
\item{formula}{an object of class \code{formula} (or one that can be coerced
to that class): a symbolic description of the model to be fitted. The
details of model specification are given under \code{Details}.}

\item{family}{a description of the error distribution and link function to
be used in the model. This can be a character string naming a family
function, a family function or the result of a call to a family function.
(See \code{\link{family}} for details of family functions.)}

\item{data}{List that containing the variables in the model.}

\item{weights}{weights}

\item{basis.x}{List of basis for functional explanatory data estimation.}

\item{basis.b}{List of basis for functional beta parameter estimation.}

\item{\dots}{Further arguments passed to or from other methods.}
}
\value{
Return \code{gam} object plus:
\itemize{
\item \code{basis.x}: Basis used for \code{fdata} or \code{fd} covariates. 
\item \code{basis.b}: Basis used for beta parameter estimation. 
\item \code{data}: List containing the variables in the model. 
\item \code{formula}: Formula used in the model. 
\item \code{y.pred}: Predicted response by cross-validation.
}
}
\description{
Computes a functional GAM model between a functional covariate
\eqn{(X^1(t_1), \dots, X^{q}(t_q))}{(X(t_1), ..., X(t_q))} and a non-functional
covariate \eqn{(Z^1, ..., Z^p)}{(Z1, ..., Zp)} with a scalar response \eqn{Y}.

This function extends functional generalized linear regression models (\code{\link{fregre.glm}}) 
where \eqn{E[Y|X,Z]} is related to the linear predictor \eqn{\eta} via a link function 
\eqn{g(\cdot)}{g(.)} with integrated smoothness estimation by the smooth
functions \eqn{f(\cdot)}{f(.)}.

\deqn{E[Y|X,Z]=\eta=g^{-1}\left(\alpha+\sum_{i=1}^{p}f_{i}(Z^{i})+\sum_{k=1}^{q}\sum_{j=1}^{k_q} f_{j}^{k}(\xi_j^k)\right)}{E[Y|X,Z]=\eta=g^{-1}(\alpha+\sum_i f_i(Z_{i})+\sum_k^q \sum_{j=1}^{k_q} f_j^k(\xi_j^k))}

where \eqn{\xi_j^k}{\xi_j^k} is the coefficient of the basis function expansion of
\eqn{X^k}; in PCA analysis, \eqn{\xi_j^k}{\xi_j^k} is the score of the
\eqn{j}-functional PC of \eqn{X^k}.
}
\details{
The smooth functions \eqn{f(\cdot)}{f(.)} can be added to the right-hand
side of the formula to specify that the linear predictor depends on smooth
functions of predictors using smooth terms \code{\link[mgcv]{s}} and
\code{\link[mgcv]{te}} as in \code{\link[mgcv]{gam}} (or linear functionals of these as
\eqn{Z \beta} and \eqn{\langle X(t), \beta \rangle}{< X(t),\beta(t) >} in
\code{\link{fregre.glm}}).

The first item in the \code{data} list is called \emph{"df"} and is a data
frame with the response and non-functional explanatory variables, as in
\code{\link[mgcv]{gam}}.\cr

Functional covariates of class \code{fdata} or \code{fd} are introduced in
the following items of the \code{data} list.\cr \code{basis.x} is a list of
basis functions for representing each functional covariate. The basis object can be
created using functions such as \code{\link{create.pc.basis}}, \code{\link[fda]{pca.fd}},
\code{\link{create.fdata.basis}}, or \code{\link[fda]{create.basis}}.\cr 
\code{basis.b} is a list of basis functions for representing each functional beta parameter. 
If \code{basis.x} is a list of functional principal components basis functions 
(see \code{\link{create.pc.basis}} or \code{\link[fda]{pca.fd}}), the argument \code{basis.b} is ignored.
}
\note{
If the formula only contains a non functional explanatory variables
(multivariate covariates), the function compute a standard \code{\link{glm}}
procedure.
}
\examples{
\dontrun{
data(tecator)
x <- tecator$absorp.fdata
x.d1 <- fdata.deriv(x)
tt <- x[["argvals"]]
dataf <- as.data.frame(tecator$y)
nbasis.x <- 11
nbasis.b <- 5
basis1 <- create.bspline.basis(rangeval=range(tt),nbasis=nbasis.x)
basis2 <- create.bspline.basis(rangeval=range(tt),nbasis=nbasis.b)
f <- Fat ~ s(Protein) + s(x)
basis.x <- list("x"=basis1,"x.d1"=basis1)
basis.b <- list("x"=basis2,"x.d1"=basis2)
ldat <- ldata("df" = dataf, "x" = x , "x.d1" = x.d1)
res <- fregre.gsam(Fat ~ Water + s(Protein) + x + s(x.d1), ldat,
                   family = gaussian(),  basis.x = basis.x,
                   basis.b = basis.b)
summary(res)
pred <- predict(res,ldat)
plot(pred-res$fitted)
pred2 <- predict.gam(res,res$XX)
plot(pred2-res$fitted)
plot(pred2-pred)
res2 <- fregre.gsam(Fat ~ te(Protein, k = 3) + x, data =  ldat,
                     family=gaussian())
summary(res2)

##  dropind basis pc
basis.pc0 <- create.pc.basis(x,c(2,4,7))
basis.pc1 <- create.pc.basis(x.d1,c(1:3))
basis.x <- list("x"=basis.pc0,"x.d1"=basis.pc1)
ldata <- ldata("df"=dataf,"x"=x,"x.d1"=x.d1)  
res.pc <- fregre.gsam(f,data=ldata,family=gaussian(),
          basis.x=basis.x,basis.b=basis.b)
summary(res.pc)
 
##  Binomial family
ldat$df$FatCat <- factor(ifelse(tecator$y$Fat > 20, 1, 0))
res.bin <- fregre.gsam(FatCat ~ Protein + s(x),ldat,family=binomial())
summary(res.bin)
table(ldat$df$FatCat, ifelse(res.bin$fitted.values < 0.5,0,1))
}
}
\references{
Muller HG and Stadtmuller U. (2005). \emph{Generalized
functional linear models.} Ann. Statist.33 774-805.

Wood (2001) \emph{mgcv:GAMs and Generalized Ridge Regression for R}. R News
1(2):20-25.

Ramsay, James O., and Silverman, Bernard W. (2006), \emph{ Functional Data
Analysis}, 2nd ed., Springer, New York.

Venables, W. N. and Ripley, B. D. (2002) \emph{Modern Applied Statistics
with S}, New York: Springer.
}
\seealso{
See Also as: \code{\link{predict.fregre.gsam}} and
\link[mgcv]{summary.gam}.\cr Alternative methods: \code{\link{fregre.glm}}
and \code{\link{fregre.gkam}}.
}
\author{
Manuel Febrero-Bande, Manuel Oviedo de la Fuente
\email{manuel.oviedo@udc.es}
}
\keyword{regression}
