% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ff_function.R
\name{fasano.franceschini.test}
\alias{fasano.franceschini.test}
\title{Fasano-Franceschini Test}
\usage{
fasano.franceschini.test(
  S1,
  S2,
  nPermute = 100,
  threads = 1,
  seed = NULL,
  verbose = TRUE,
  method = c("r", "b")
)
}
\arguments{
\item{S1}{\code{matrix} or \code{data.frame}.}

\item{S2}{\code{matrix} or \code{data.frame}.}

\item{nPermute}{A nonnegative \code{integer} setting the number of permutations
to use for performing the permutation test. Default is \code{100}. If set to
\code{0}, only the test statistic is computed.}

\item{threads}{A positive \code{integer} or \code{"auto"} setting the number
of threads to use during the permutation test. If set to \code{"auto"}, the
number of threads is determined by \code{RcppParallel::defaultNumThreads()}.
Default is \code{1}.}

\item{seed}{An optional integer to seed the PRNG used for the permutation test.
A seed must be passed to reproducibly compute p-values.}

\item{verbose}{A \code{boolean} indicating whether to display a progress bar.
Default is \code{TRUE}. Only available when \code{threads = 1}.}

\item{method}{An optional \code{character} indicating which method to use to
compute the test statistic. The two methods are \code{'r'} (range tree) and
\code{'b'} (brute force). Both methods return the same results but may vary in
computation speed. If this argument is not passed, the sample sizes and dimension
of the data are used to infer which method is likely faster. See the Details
section for more information.}
}
\value{
A list of class \code{htest} containing the following components:
  \item{statistic}{The value of the test statistic.}
  \item{p.value}{The permutation test p-value.}
  \item{method}{The name of the test.}
  \item{data.name}{The names of the original data objects.}
}
\description{
Performs a two-sample multivariate Kolmogorov-Smirnov test as described
by Fasano and Franceschini (1987). This test evaluates the null hypothesis
that two i.i.d. random samples were drawn from the same underlying
probability distribution. The data can be of any dimension and of any
type (continuous, discrete, or mixed).
}
\details{
The test statistic can be computed using two different methods.
Both methods return identical results, but have different time complexities:
\itemize{
  \item Range tree method: This method has a time complexity of
  \emph{O(N*log(N)^(d-1))}, where \emph{N} is the size of the larger sample
  and \emph{d} is the dimension of the data.
  \item Brute force method: This method has a time complexity of \emph{O(N^2)}.
}
The range tree method tends to be faster for low dimensional data or large
sample sizes, while the brute force method tends to be faster for high
dimensional data or small sample sizes. When \code{method} is not passed,
the sample sizes and dimension of the data are used to infer which method will
likely be faster. However, as the geometry of the samples can influence
computation time, the method inferred to be faster may not actually be faster. To
perform more comprehensive benchmarking for a specific dataset, \code{nPermute}
can be set equal to \code{0}, which bypasses the permutation test and only
computes the test statistic.
}
\examples{
set.seed(0)

# create 2-D samples
S1 <- data.frame(x = rnorm(n = 20, mean = 0, sd = 1),
                 y = rnorm(n = 20, mean = 1, sd = 2))
S2 <- data.frame(x = rnorm(n = 40, mean = 0, sd = 1),
                 y = rnorm(n = 40, mean = 1, sd = 2))

# perform test
fasano.franceschini.test(S1, S2)

# perform test with more permutations
fasano.franceschini.test(S1, S2, nPermute = 150)

# set seed for reproducible p-value
fasano.franceschini.test(S1, S2, seed = 0)$p.value
fasano.franceschini.test(S1, S2, seed = 0)$p.value

# perform test using range tree method
fasano.franceschini.test(S1, S2, method = 'r')

# perform test using brute force method
fasano.franceschini.test(S1, S2, method = 'b')

# perform test using multiple threads to speed up p-value computation
\dontrun{
fasano.franceschini.test(S1, S2, threads = 2)
}

}
\references{
{
\itemize{
  \item{Fasano, G. & Franceschini, A. (1987). A multidimensional version of the
  Kolmogorov-Smirnov test. \emph{Monthly Notices of the Royal Astronomical Society},
  225:155-170. \doi{10.1093/mnras/225.1.155}.}
} }
}
