#' Print a FACT Tree Object
#'
#' Provides a concise textual representation of the FACT tree structure.
#'
#' @param x An object of class `FACT`, typically the result of a call to `FACT()`.
#' @param ... Additional arguments (not used).
#'
#' @return Invisibly returns the original `FACT` object.
#' @export
#' @method print FACT
print.FACT <- function(x, ...) {
  cat("Factor-Augmented Clustering Tree\n\n")
  cat("Node structure:\n")
  print_node <- function(node_id, depth) {
    node_info <- x$frame[x$frame$node_id == node_id, ]
    prefix <- paste(rep("  ", depth), collapse = "")
    if (node_info$is_leaf) {
      cat(sprintf("%s* Node %d (n=%d, Leaf)\n", prefix, node_info$node_id, node_info$n_obs))
    } else {
      split_text <- sprintf("%s < %.3f", node_info$split_var, node_info$split_val)
      stat_text <- sprintf("stat=%.3f", node_info$statistic)
      if (!is.na(node_info$p_value)) {
        # CHANGED: Use base::format.pval
        stat_text <- paste0(stat_text, ", p=", base::format.pval(node_info$p_value, digits = 2))
      }
      cat(sprintf("%s- Node %d (n=%d): %s (%s)\n", prefix, node_info$node_id, node_info$n_obs, split_text, stat_text))
      print_node(node_info$left_child, depth + 1)
      print_node(node_info$right_child, depth + 1)
    }
  }
  print_node(node_id = 1, depth = 0)
  invisible(x)
}


#' Plot a FACT Tree Object
#'
#' Generates a visual plot of the FACT tree structure.
#'
#' @param x An object of class `FACT`, typically the result of a call to `FACT()`.
#' @param ... Additional arguments (not used).
#'
#' @return No return value, called for side effects (plotting).
#' @export
#' @method plot FACT
plot.FACT <- function(x, ...) {
  frame <- x$frame
  if (nrow(frame) == 1 && frame$is_leaf[1]) {
    # CHANGED: Use graphics::
    graphics::plot(1, 1, type = "n", axes = FALSE, xlab = "", ylab = "")
    graphics::text(1, 1, paste("Node 1 (Leaf)\nn =", frame$n_obs[1]), cex = 1.2)
    return()
  }
  # ... (code to calculate positions is fine) ...
  # --- 1. Calculate node positions ---
  leaf_count <- integer(nrow(frame))
  node_depth <- integer(nrow(frame))
  id_map <- match(frame$node_id, frame$node_id)

  traverse_pos <- function(node_id, depth) {
    node_idx <- id_map[node_id]
    node_depth[node_idx] <<- depth
    node_info <- frame[node_idx, ]
    if (node_info$is_leaf) {
      leaf_count[node_idx] <<- 1
      return(1)
    } else {
      left_leaves <- traverse_pos(node_info$left_child, depth + 1)
      right_leaves <- traverse_pos(node_info$right_child, depth + 1)
      total_leaves <- left_leaves + right_leaves
      leaf_count[node_idx] <<- total_leaves
      return(total_leaves)
    }
  }
  total_leaves <- traverse_pos(1, 0)
  # --- 2. Calculate x, y coordinates for each node ---
  max_depth <- max(node_depth)
  y_coords <- if (max_depth == 0) 0.5 else 0.95 - (node_depth / max_depth) * 0.9

  x_coords <- numeric(nrow(frame))
  assign_x <- function(node_id, x_start) {
    node_idx <- id_map[node_id]
    node_info <- frame[node_idx, ]
    if (node_info$is_leaf) {
      x_coords[node_idx] <<- x_start + 0.5
      return()
    }
    left_idx <- id_map[node_info$left_child]
    left_prop <- leaf_count[left_idx] / leaf_count[node_idx]
    x_coords[node_idx] <<- x_start + left_prop * leaf_count[node_idx]
    assign_x(node_info$left_child, x_start)
    assign_x(node_info$right_child, x_start + leaf_count[left_idx])
  }
  assign_x(1, 0)
  x_coords <- x_coords / total_leaves

  # --- 3. Plotting ---
  # CHANGED: Use graphics:: for all plotting functions
  graphics::plot(NA, xlim = c(0, 1), ylim = c(0, 1), type = "n", axes = FALSE, xlab = "", ylab = "")
  # Draw segments
  for (i in 1:nrow(frame)) {
    if (!frame$is_leaf[i]) {
      parent_x <- x_coords[i]
      parent_y <- y_coords[i]
      left_child_idx <- id_map[frame$left_child[i]]
      right_child_idx <- id_map[frame$right_child[i]]
      child_x_L <- x_coords[left_child_idx]
      child_y_L <- y_coords[left_child_idx]
      child_x_R <- x_coords[right_child_idx]
      child_y_R <- y_coords[right_child_idx]
      graphics::segments(parent_x, parent_y, child_x_L, child_y_L)
      graphics::segments(parent_x, parent_y, child_x_R, child_y_R)
    }
  }
  # Draw nodes and text
  for (i in 1:nrow(frame)) {
    x_pos <- x_coords[i]
    y_pos <- y_coords[i]
    box_text <- ""
    if (frame$is_leaf[i]) {
      box_text <- sprintf("Leaf %d\nn=%d", frame$node_id[i], frame$n_obs[i])
    } else {
      box_text <- sprintf("%s < %.2f\nn=%d", frame$split_var[i], frame$split_val[i], frame$n_obs[i])
    }
    box_width <- graphics::strwidth(box_text, cex = 0.8) * 1.2
    box_height <- graphics::strheight(box_text, cex = 0.8) * 2.5
    graphics::rect(x_pos - box_width / 2, y_pos - box_height / 2, x_pos + box_width / 2, y_pos + box_height / 2, col = "white", border = "black")
    graphics::text(x_pos, y_pos, box_text, cex = 0.8)
  }
}
