% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dfcrm_selector.R
\name{get_dfcrm}
\alias{get_dfcrm}
\title{Get an object to fit the CRM model using the dfcrm package.}
\usage{
get_dfcrm(parent_selector_factory = NULL, skeleton, target, tite = FALSE, ...)
}
\arguments{
\item{parent_selector_factory}{optional object of type
\code{\link{selector_factory}} that is in charge of dose selection before
this class gets involved. Leave as NULL to just use CRM from the start.}

\item{skeleton}{Dose-toxicity skeleton, a non-decreasing vector of
probabilities.}

\item{target}{We seek a dose with this probability of toxicity.}

\item{tite}{FALSE to use regular CRM; TRUE to use TITE-CRM. See Description.}

\item{...}{Extra args are passed to \code{\link[dfcrm]{crm}}.}
}
\value{
an object of type \code{\link{selector_factory}} that can fit the
CRM model to outcomes.
}
\description{
This function returns an object that can be used to fit a CRM model using
methods provided by the dfcrm package.

Dose selectors are designed to be daisy-chained together to achieve different
behaviours. This class is a **resumptive** selector, meaning it carries on
when the previous dose selector, where present, has elected not to continue.
For example, this allows instances of this class to be preceded by a selector
that follows a fixed path in an initial escalation plan, such as that
provided by \code{\link{follow_path}}. In this example, when the observed
trial outcomes deviate from that initial plan, the selector following the
fixed path elects not to continue and responsibility passes to this class.
See Examples.

The time-to-event variant, TITE-CRM, is used via the
\code{dfcrm::titecrm} function when you specify \code{tite = TRUE}. This
weights the observations to allow dose-selections based on partially observed
outcomes.
}
\examples{
skeleton <- c(0.05, 0.1, 0.25, 0.4, 0.6)
target <- 0.25
model1 <- get_dfcrm(skeleton = skeleton, target = target)

# By default, dfcrm fits the empiric model:
outcomes <- '1NNN 2NTN'
model1 \%>\% fit(outcomes) \%>\% recommended_dose()

# But we can provide extra args to get_dfcrm that are than passed onwards to
# the call to dfcrm::crm to override the defaults. For example, if we want
# the one-parameter logistic model:
model2 <- get_dfcrm(skeleton = skeleton, target = target, model = 'logistic')
model2 \%>\% fit(outcomes) \%>\% recommended_dose()
# dfcrm does not offer a two-parameter logistic model but other classes do.

# We can use an initial dose-escalation plan, a pre-specified path that
# should be followed until trial outcomes deviate, at which point the CRM
# model takes over. For instance, if we want to use two patients at each of
# the first three doses in the absence of toxicity, irrespective the model's
# advice, we would run:
model1 <- follow_path('1NN 2NN 3NN') \%>\%
  get_dfcrm(skeleton = skeleton, target = target)

# If outcomes match the desired path, the path is followed further:
model1 \%>\% fit('1NN 2N') \%>\% recommended_dose()

# But when the outcomes diverge:
model1 \%>\% fit('1NN 2T') \%>\% recommended_dose()

# Or the pre-specified path comes to an end:
model1 \%>\% fit('1NN 2NN 3NN') \%>\% recommended_dose()
# The CRM model takes over.

}
\references{
Cheung, K. 2019. dfcrm: Dose-Finding by the Continual Reassessment Method.
R package version 0.2-2.1. https://CRAN.R-project.org/package=dfcrm

Cheung, K. 2011. Dose Finding by the Continual Reassessment Method.
Chapman and Hall/CRC. ISBN 9781420091519

O’Quigley J, Pepe M, Fisher L. Continual reassessment method: a practical
design for phase 1 clinical trials in cancer. Biometrics. 1990;46(1):33-48.
doi:10.2307/2531628
}
