% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/anovaPB.R
\name{anovaPB}
\alias{anovaPB}
\title{Parametric bootstrap testing to compare two models by analysis of variance (deviance)}
\usage{
anovaPB(
  objectNull,
  object,
  n.sim = 999,
  colRef = switch(class(object)[1], lm = 5, lmerMod = 6, glmmTMB = 6, 4),
  rowRef = 2,
  ncpus = NULL,
  ...
)
}
\arguments{
\item{objectNull}{is the fitted model under the null hypothesis. This can be
\emph{any} object that responds to \code{simulate}, \code{update} and \code{anova}.}

\item{object}{is the fitted model under the alternative hypothesis. This can be
\emph{any} object that responds to \code{update}, \code{anova} and \code{model.frame}.
It must be larger than \code{objectNull} and its model frame should contain
all the terms required to fit \code{objectNull}.}

\item{n.sim}{the number of simulated datasets to generate for parametric
bootstrap testing. Defaults to 999.}

\item{colRef}{the column number where the test statistic of interest can be found
in the standard \code{anova} output when calling \code{anova(objectNull,object)}.
Default choices have been set for common models (\code{colRef=5} for \code{lm} objects,
\code{colRef=6} for \code{lmer} and \code{glmmTMB} objects and \code{colRef=4} otherwise, which is correct 
for \code{glm} and \code{gam} objects).}

\item{rowRef}{the row number where the test statistic of interest can be found
in the standard \code{anova} output when calling \code{anova(objectNull,object)}.
Defaults to \code{2}, because it is on the second row for most common models.}

\item{ncpus}{the number of CPUs to use. Default (NULL) uses two less than the total 
available.}

\item{...}{further arguments sent through to \code{anova}.}
}
\value{
A matrix which has the appearance and behaviour of an object returned by 
\code{anova(objectNull,object)}, except that the \emph{P}-value has been computed
by parametric bootstrap, and the matrix now inherits class \code{anovaPB}.
}
\description{
Compute analysis of variance (or deviance) tables for two fitted model objects,
with the \emph{P}-value estimated using a parametric bootstrap -- by repeatedly
simulating new responses from the fitted model under the null hypothesis.
}
\details{
The \code{anova} function typically relies on classical asymptotic results
which sometimes don't apply well, e.g. when using penalised likelihood to fit a
generalised additive model, or when testing whether a random effect should be
included in a mixed model. In such cases we can get a more accurate test by
using simulation to estimate the \emph{P}-value -- repeatedly simulating 
new sets of simulated responses, refitting the null and alternative models, and 
recomputing the test statistic. This process allows us to estimate the 
distribution of the test statistic when the null hypothesis is true, hence 
draw a conclusion about whether the observed test statistic is large relative
to what would be expected under the null hypothesis. The process is often
referred to as a \emph{parametric bootstrap} (Davison & Hinkley 1997), which
the \emph{PB} in the function name (\code{anovaPB}) is referring to.

This function will take any pair of fitted models, a null (\code{objectNull})
and an alternative (\code{object}), and use simulation to estimate the 
\emph{P}-value of the test of whether there is evidence against the model in
\code{objectNull} in favour of the model in \code{object}. This function works
by repeatedly performing an \code{anova} to compare \code{objectNull}  to
\code{object}, where the responses in the \code{model.frame} have been replaced
with values simulated by applying \code{simulate} to \code{objectNull}. Hence
for this function to work, the objects being compared need to respond 
to the \code{anova}, \code{simulate} and \code{model.frame} functions.

This function needs to be able to find the test statistic in the \code{anova}
output, but it is stored in different places for different types of objects.
It is assumed that \code{anova(objectNull,object)} returns a matrix and that the
test statistic is stored in the \code{(rowRef,colRef)}th element, where both
\code{rowRef} and \code{colRef} have been chosen to be correct for common
model types (\code{glm}, \code{gam}, \code{lmer}).
}
\examples{
# fit a Poisson regression to random data:
y = rpois(50,lambda=1)
x = 1:50
rpois_glm = glm(y~x, family=poisson())
rpois_int = glm(y~1, family=poisson())
anovaPB(rpois_int, rpois_glm, n.sim=99, ncpus=1)
# this approach would run faster on some larger problems (but maybe not this one!):
\dontrun{anovaPB(rpois_int, rpois_glm, n.sim=99, ncpus=4)}

}
\references{
Davison A.C. & Hinkley D. V. (1997) Bootstrap methods and their application, Cambridge University Press.
Warton DI (2022) Eco-Stats - Data Analysis in Ecology, from \emph{t}-tests to multivariate abundances. Springer, ISBN 978-3-030-88442-0
}
\seealso{
\code{\link[stats]{anova}}
}
\author{
David Warton <david.warton@unsw.edu.au>
}
