% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulateMTLNR.R
\name{simulateMTLNR}
\alias{simulateMTLNR}
\title{Simulation of confidence ratings and RTs in the
Multiple-threshold Log-normal Race Model}
\usage{
simulateMTLNR(paramDf, n = 10000, gamma = FALSE, agg_simus = FALSE,
  stimulus = c(1, 2), seed = NULL)
}
\arguments{
\item{paramDf}{a list or data frame with one row. Column names should
match the following names (see \link{dMTLNR}):
For different stimulus quality/mean
drift rates, names should be \code{v1}, \code{v2}, \code{v3},.... (corresponding to the mean
parameter for the accumulation rate for the stimulus-corresponding accumulator,
therefore \code{mu_v1} and \code{mu_v2} are not used in this context but
replaced by the parameter \code{v}); \code{mu_d1} and \code{mu_d2} correspond to the mean
parameters for boundary distance of the two accumulators;
\code{s1} and \code{s2} correspond to the variance parameters of the first and
second boundary hitting time;
\code{rho} corresponds to the correlation of boundary hitting times.
Note that \code{s_v1},\code{s_v2},\code{rho_v},\code{s_d1},\code{s_d2}, and \code{rho_d} are not used in this
context, although the accumulation rate-related parameters can be used to replace
the above-mentioned variance parameters.
Additionally, the confidence thresholds should be given by names with
\code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},.... (see Details for the correspondence to the data)}

\item{n}{integer. The number of samples (per condition and stimulus direction) generated.
Total number of samples is \code{n*nConditions*length(stimulus)}.}

\item{gamma}{logical. If TRUE, the gamma correlation between confidence ratings, rt
and accuracy is computed.}

\item{agg_simus}{logical. Simulation is done on a trial basis with RTs outcome.
If TRUE, the simulations will be aggregated over RTs to return only the distribution
of response and confidence ratings. Default: FALSE.}

\item{stimulus}{numeric vector. Either 1, 2 or c(1,2) (default). Together with
condition represents the experimental situation. In a binary decision task the presented
stimulus belongs to one of two categories. In the default setting trials with
both categories presented are simulated but one can choose to simulate only trials with
the stimulus coming from one category (1 for the category that is associated with positive
drift in the decision process where 1 responses are considered correct and 2
correspondingly for negative drifts and 2 correct decisions).}

\item{seed}{numerical. Seeding for non-random data generation.}
}
\value{
Depending on \code{gamma} and \code{agg_simus}.

If \code{gamma} is \code{FALSE}, returns a \code{data.frame} with columns: \code{condition},
\code{stimulus}, \code{response}, \code{correct}, \code{rt}, \code{conf} (the continuous confidence
measure) and \code{rating} (the discrete confidence rating), and \code{dec} and \code{vis}
(only if \code{process_results=TRUE}) for the final states of accumulators in the
simulation or
(if \code{agg_simus=TRUE}): \code{condition}, \code{stimulus},\code{response}, \code{correct},
\code{rating} and \code{p} (for the probability of a response and rating, given
the condition and stimulus).

If \code{gamma} is \code{TRUE}, returns a \code{list} with elements:
\code{simus} (the simulated data frame) and \code{gamma}, which is again a \code{list} with elements
\code{condition}, \code{rt} and \code{correct}, each a \code{tibble} with two columns (see details for more
information).
}
\description{
Simulates the decision responses and reaction times together with a
discrete confidence judgment in the MTLNR (Reynolds et al., 2020),
given specific parameter constellations.
See \code{\link{dMTLNR}} for more information about parameters.
Also computes the Gamma rank correlation between the confidence ratings and condition
(task difficulty), reaction times and accuracy in the simulated output.
Basically, this function is a wrapper for \code{\link{rMTLNR}}
for application in confidence experiments with manipulation of specific parameters.
}
\details{
Simulation is done by simulating normally distributed logarithms of
boundary crossing times for both accumulators based on the MTLNR model.
The smaller time determines decision time and response (i.e. the winning
accumulator). The confidence variable is computed based on the log-ratio of the
loosing boundary crossing time over the winning boundary crossing time.

The confidence values are then binned according to the given thresholds.
The output of the fitting function \code{\link{fitRTConf}} with the respective model
fits the argument \code{paramDf} for simulation.
The Gamma coefficients are computed separately for correct/incorrect responses for the
correlation of confidence ratings with condition and rt and separately for conditions
for the correlation of accuracy and confidence. The
resulting data frames in the output thus have two columns. One for the grouping variable
and one for the Gamma coefficient.
}
\note{
Different parameters for different conditions are only allowed for drift rate,
\code{v}.
All other parameters are used for all conditions.
}
\examples{
# 1. Define some parameter set in a data.frame
paramDf <- data.frame(v1=0.5, v2=1.0, t0=0.1, st0=0,
                      mu_d1=1, mu_d2=1,
                      s1=0.5, s2=0.5, rho=0.2,
                      theta1=0.8, theta2=1.5)

# 2. Simulate trials for both stimulus categories and all conditions (2)
simus <- simulateMTLNR(paramDf)
head(simus)
\donttest{
  library(ggplot2)
  simus <- simus[simus$response != 0, ]
  simus$rating <- factor(simus$rating, labels = c("unsure", "medium", "sure"))
  ggplot(simus, aes(x = rt, group = interaction(correct, rating),
                    color = as.factor(correct), linetype = rating)) +
    geom_density(linewidth = 1.2, na.rm=TRUE) + xlim(c(0, 5)) +
    facet_grid(rows = vars(condition), labeller = "label_both")
}

# automatically aggregate simulation distribution
# to get only accuracy x confidence rating distribution for
# all conditions
agg_simus <- simulateMTLNR(paramDf, agg_simus = TRUE)
head(agg_simus)
\donttest{
  agg_simus$rating <- factor(agg_simus$rating, labels = c("unsure", "medium", "sure"))
  library(ggplot2)
  ggplot(agg_simus, aes(x = rating, group = correct, fill = as.factor(correct), y = p)) +
    geom_bar(stat = "identity", position = "dodge") +
    facet_grid(cols = vars(condition), labeller = "label_both")
}
\donttest{
  # Compute Gamma correlation coefficients between
  # confidence and other behavioral measures
  # output will be a list
  simu_list <- simulateMTLNR(paramDf, n = 400, gamma = TRUE)
  simu_list
}

# Example with asymmetric confidence thresholds
paramDf_asym <- data.frame(v1=0.5, v2=1.0, t0=0.1, st0=0,
                          mu_d1=1, mu_d2=1,
                          s1=0.5, s2=0.5, rho=0.2,
                          thetaLower1=0.5, thetaLower2=1.2,
                          thetaUpper1=0.7, thetaUpper2=1.8)

simus_asym <- simulateMTLNR(paramDf_asym, n = 1000)
head(simus_asym)

# Example with multiple conditions
paramDf_multi <- data.frame(v1=0.3, v2=0.6, v3=1.2, t0=0.1, st0=0,
                           mu_d1=1, mu_d2=1,
                           s1=0.5, s2=0.5, rho=0.2,
                           theta1=0.8, theta2=1.5)

simus_multi <- simulateMTLNR(paramDf_multi, n = 1000)
table(simus_multi$condition, simus_multi$correct)

}
\references{
Reynolds, A., Kvam, P. D., Osth, A. F., & Heathcote, A. (2020). Correlated racing evidence accumulator models. \emph{Journal of Mathematical Psychology, 96}, 102331. doi: doi: 10.1016/j.jmp.2020.102331
}
\author{
Sebastian Hellmann.
}
