% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predictratingdist_MTLNR.R
\name{predictMTLNR}
\alias{predictMTLNR}
\alias{predictMTLNR_Conf}
\alias{predictMTLNR_RT}
\title{Prediction of Confidence Rating and Reaction Time Distribution in the
multiple-threshold log-normal noise race model}
\usage{
predictMTLNR_Conf(paramDf, maxrt = Inf, subdivisions = 100L,
  stop.on.error = FALSE, .progress = TRUE)

predictMTLNR_RT(paramDf, maxrt = 9, subdivisions = 100L, minrt = NULL,
  scaled = FALSE, DistConf = NULL, .progress = TRUE)
}
\arguments{
\item{paramDf}{a list or data frame with one row. Column names should
match the following names (see \link{dMTLNR}):
For different stimulus quality/mean
drift rates, names should be \code{v1}, \code{v2}, \code{v3},.... (corresponding to the mean
parameter for the accumulation rate for the stimulus-corresponding accumulator,
therefore \code{mu_v1} and \code{mu_v2} are not used in this context but
replaced by the parameter \code{v}); \code{mu_d1} and \code{mu_d2} correspond to the mean
parameters for boundary distance of the two accumulators;
\code{s1} and \code{s2} correspond to the variance parameters of the first and
second boundary hitting time;
\code{rho} corresponds to the correlation of boundary hitting times.
Note that \code{s_v1},\code{s_v2},\code{rho_v},\code{s_d1},\code{s_d2}, and \code{rho_d} are not used in this
context, although the accumulation rate-related parameters can be used to replace
the above-mentioned variance parameters.
Additionally, the confidence thresholds should be given by names with
\code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},.... (see Details for the correspondence to the data)}

\item{maxrt}{numeric. The maximum RT for the
integration/density computation. Default: 15 (for \code{predictMTLNR_Conf}
(integration)), 9 (for \code{predictMTLNR_RT}).}

\item{subdivisions}{\code{integer} (default: 100).
For \code{predictMTLNR_Conf} it is used as argument for the inner integral routine.
For \code{predictMTLNR_RT} it is the number of points for which the density is computed.}

\item{stop.on.error}{logical. Argument directly passed on to integrate. Default is FALSE,
since the densities invoked may lead to slow convergence of the integrals (which are still
quite accurate) which causes R to throw an error.}

\item{.progress}{logical. If TRUE (default) a progress bar is drawn to the console.}

\item{minrt}{numeric or NULL(default). The minimum rt for the density computation.}

\item{scaled}{logical. For \code{predictMTLNR_RT}. Whether the computed density
should be scaled to integrate to one (additional column \code{densscaled}). Otherwise the output
contains only the defective density (i.e. its integral is equal to the probability of a
response and not 1). If \code{TRUE}, the argument \code{DistConf} should be given, if available.
Default: \code{FALSE}.}

\item{DistConf}{\code{NULL} or \code{data.frame}. A \code{data.frame} or \code{matrix} with column
names, giving the distribution of response and rating choices for
different conditions and stimulus categories in the form of the output of
\code{predictMTLNR_Conf}. It is only necessary, if \code{scaled=TRUE}, because these
probabilities are used for scaling. If \code{scaled=TRUE} and \code{DistConf=NULL}, it will be computed
with the function \code{predictMTLNR_Conf}, which takes some time and the function will
throw a message. Default: \code{NULL}}
}
\value{
\code{predictMTLNR_Conf} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{p}, \code{info}, \code{err}. \code{p} is the predicted probability of a response
and \code{rating}, given the stimulus category and condition. \code{info} and \code{err} refer to the
respective outputs of the integration routine used for the computation.
\code{predictMTLNR_RT} returns a \code{data.frame}/\code{tibble} with columns: \code{condition}, \code{stimulus},
\code{response}, \code{rating}, \code{correct}, \code{rt} and \code{dens} (and \code{densscaled}, if \code{scaled=TRUE}).
}
\description{
\code{predictMTLNR_Conf} predicts the categorical response distribution of
decision and confidence ratings, \code{predictMTLNR_RT} computes the
RT distribution (density) in the multiple-threshold log-normal noise
race model (Reynolds et al., 2020), given specific parameter
constellations. See \code{\link{dMTLNR}} for more information about the models
and parameters.
}
\details{
The function \code{predictMTLNR_Conf} consists merely of an integration of
the response time density, \code{\link{dMTLNR}}, over the
response time in a reasonable interval (0 to \code{maxrt}). The function
\code{predictMTLNR_RT} wraps these density
functions to a parameter set input and a data.frame output.
For the argument \code{paramDf}, the output of the fitting function \code{\link{fitRTConf}}
with the respective model may be used.

The names of the accumulation rate parameters differ from those used in
\code{\link{dMTLNR}} because the accumulation rates for the two options depend
on stimulus and condition. Here, the mean parameter for the accumulation rate
of the correct accumulator is \code{v} (\code{v1}, \code{v2},... respectively) in \code{paramDf}
and the other one has a mean parameter of 0.
}
\note{
Different parameters for different conditions are only allowed for drift rate,
\code{v}. All other parameters are used for all
conditions.
}
\examples{
# 1. Define some parameter set in a data.frame
paramDf <- data.frame(v1=0.5, v2=1.0, t0=0.1, st0=0,
                      mu_d1=1, mu_d2=1,
                      s1=0.5, s2=0.5, rho=0.2,
                      theta1=0.8, theta2=1.5)

# 2. Predict discrete Choice x Confidence distribution:
preds_Conf <- predictMTLNR_Conf(paramDf, maxrt=7, subdivisions=50)
head(preds_Conf)

# 3. Compute RT density
preds_RT <- predictMTLNR_RT(paramDf, maxrt=7, subdivisions=50)
# same output with scaled density column:
preds_RT <- predictMTLNR_RT(paramDf, maxrt=7, subdivisions=50,
                         scaled=TRUE, DistConf = preds_Conf)
head(preds_RT)
\donttest{
  # produces a warning, if scaled=TRUE and DistConf missing
  preds_RT <- predictMTLNR_RT(paramDf, maxrt=7, subdivisions=50,
                           scaled=TRUE)
}

\donttest{
  # Example of visualization
  library(ggplot2)
  preds_Conf$rating <- factor(preds_Conf$rating, labels=c("unsure", "medium", "sure"))
  preds_RT$rating <- factor(preds_RT$rating, labels=c("unsure", "medium", "sure"))
  ggplot(preds_Conf, aes(x=interaction(rating, response), y=p))+
    geom_bar(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")
  ggplot(preds_RT, aes(x=rt, color=interaction(rating, response), y=dens))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(stimulus), rows=vars(condition), labeller = "label_both")+
    theme(legend.position = "bottom")
  ggplot(aggregate(densscaled~rt+correct+rating+condition, preds_RT, mean),
         aes(x=rt, color=rating, y=densscaled))+
    geom_line(stat="identity")+
    facet_grid(cols=vars(condition), rows=vars(correct), labeller = "label_both")+
    theme(legend.position = "bottom")
}
\donttest{
  # Use PDFtoQuantiles to get predicted RT quantiles
  # (produces warning because of few rt steps (--> inaccurate calculations))
  PDFtoQuantiles(preds_RT, scaled = FALSE)
}

# Example with asymmetric confidence thresholds
paramDf_asym <- data.frame(v1=0.5, v2=1.0, t0=0.1, st0=0,
                          mu_d1=1, mu_d2=1,
                          s1=0.5, s2=0.5, rho=0.2,
                          thetaLower1=0.5, thetaLower2=1.2,
                          thetaUpper1=0.7, thetaUpper2=1.8)

preds_Conf_asym <- predictMTLNR_Conf(paramDf_asym, maxrt=7, subdivisions=50)
head(preds_Conf_asym)

# Example with multiple conditions
paramDf_multi <- data.frame(v1=0.3, v2=0.6, v3=1.2, t0=0.1, st0=0,
                           mu_d1=1, mu_d2=1,
                           s1=0.5, s2=0.5, rho=0.2,
                           theta1=0.8, theta2=1.5)

preds_Conf_multi <- predictMTLNR_Conf(paramDf_multi, maxrt=7, subdivisions=50)
head(preds_Conf_multi)

}
\references{
Reynolds, A., Kvam, P. D., Osth, A. F., & Heathcote, A. (2020). Correlated racing evidence accumulator models. \emph{Journal of Mathematical Psychology, 96}, 102331. doi: doi: 10.1016/j.jmp.2020.102331
}
\author{
Sebastian Hellmann.
}
