% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert.R
\name{convert}
\alias{convert}
\title{Convert a matrix of semi-processed DICOM images to mass and volume of material classes}
\usage{
convert(mat.list, upperLim = 3045, lowerLim = -1025, 
pixelA, thickness = 0.625, # all in mm 
means     = c(-850.3233, 63.912, 271.7827, 1345.0696),
sds       = c(77.6953, 14.1728, 39.2814, 45.4129),
densities = c(0.0012, 1, 1.23, 2.2))
}
\arguments{
\item{mat.list}{list of DICOM images for a sediment core (values in Hounsfield Units)}

\item{upperLim}{upper bound cutoff for pixels (Hounsfield Units)}

\item{lowerLim}{lower bound cutoff for pixels (Hounsfield Units)}

\item{pixelA}{pixel area (mm2)}

\item{thickness}{slice thickness for computed tomography image series (mm)}

\item{means}{mean values (units = Hounsfield Units) for calibration rods used.}

\item{sds}{standard deviations (units = Hounsfield Units) for calibration rods used. Must be in the same order as \code{means}.}

\item{densities}{numeric vector of known cal rod densities. Must be in the same order as \code{means} and \code{sds}.}
}
\value{
value \code{convert} returns a dataframe with one row per CT slice. Values returned are the average Hounsfield Unit value, the area (cm2), volume (cm3), and mass (grams) of 7 material classes: gas, peat, roots and rhizomes, particulates, sand, water, and rock/shell. If <code>rootData = TRUE</code>, data for specified root size classes are also returned. See <code>getRoots</code> for more detail on those values.
}
\description{
Converts raw CT units to material classes for each CT slice. This version accommodates calibration curves with >4 calibrants, and uses density thresholds converted to Hounsfield Units using the calibration curve (rather than direct calibration rod values) to partition sediment components.
}
\details{
Calculates average Hounsfield units, cross-sectional areas (cm2), volumes (cm3), and masses (g) of material classes for each CT slice. This function assumes that core walls and all non-sediment material have been removed from the raw DICOM imagery. This function converts data from raw x-ray attenuation values to Hounsfield Units, and then uses user-defined calibration rod inputs to categorize sediment components: air, roots and rhizomes, peat, water, particulates, sand, and rock/shell. The input style for calibration rods ensures sediment components are partitioned following the density divisions in Davey et al. 2011. Calibration rods and are used to develop the calibration curve. Separately, the densities used for partitioning in Davey et al. 2011 (0.0012, 1, 1.23, 2.2 g/cm3) are converted to Hounsfield Units and used for partitioning sediment components. The standard deviation for the calibration rod nearest to the target value is used for the standard deviation for the division between two sediment components.
}
\examples{
ct.slope <- unique(extractHeader(core_426$hdr, "RescaleSlope"))
ct.int   <- unique(extractHeader(core_426$hdr, "RescaleIntercept")) 
# convert raw units to Hounsfield units
HU_426 <- lapply(core_426$img, function(x) x*ct.slope + ct.int)

materials <- convert(HU_426, pixelA = 0.0596)

\dontrun{
# plot using "ggplot" package after transforming with "reshape2" package
mass.long <- reshape2::melt(materials, id.vars = c("depth"), 
   measure.vars = grep(".g", names(materials)))
ggplot2::ggplot(data = mass.long, ggplot2::aes(y = -depth, x = value, 
   color = variable)) + ggplot2::geom_point() + ggplot2::theme_classic() + 
   ggplot2::xlab("mass per section (g)")
}

}
\seealso{
\code{\link{getRoots}} operates similarly.
}
