% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/prop_ci_mn.R
\name{ci_prop_diff_mn}
\alias{ci_prop_diff_mn}
\title{Miettinen-Nurminen Confidence Interval for Difference in Proportions}
\usage{
ci_prop_diff_mn(x, by, conf.level = 0.95, delta = NULL, data = NULL)
}
\arguments{
\item{x}{(\code{binary}/\code{numeric}/\code{logical})\cr
vector of a binary values, i.e. a logical vector, or numeric with values \code{c(0, 1)}}

\item{by}{(\code{string}) \cr A character or factor vector with exactly two unique levels
identifying the two groups to compare. Can also be a column name if a data
frame provided in the \code{data} argument.}

\item{conf.level}{(\verb{scalar numeric}) \cr a scalar in (0,1) indicating the confidence level. Default is 0.95}

\item{delta}{(\code{numeric}) \cr Optionally a single number or a vector of
numbers between -1 and 1 (not inclusive) to set the difference between two
groups under the null hypothesis. If provided, the function returns the
test statistic and p-value under the \code{delta}
hypothesis.}

\item{data}{(\code{data.frame}) \cr Optional data frame containing the variables specified in \code{x} and \code{by}.}
}
\value{
An object containing the following components:

\item{estimate}{The point estimate of the difference in proportions (p_x - p_y)}
\item{conf.low}{Lower bound of the confidence interval}
\item{conf.high}{Upper bound of the confidence interval}
\item{conf.level}{The confidence level used}
\item{delta}{delta value(s) used}
\item{statistic}{Z-Statistic under the null hypothesis based on the given 'delta'}
\item{p.value}{p-value under the null hypothesis based on the given 'delta'}
\item{method}{Description of the method used ("Miettinen-Nurminen Confidence Interval")}

If \code{delta} is not provided statistic and p.value will be NULL
}
\description{
Calculates the Miettinen-Nurminen (MN) confidence interval for the difference
between two proportions. This method can be more accurate than traditional
methods, especially with small sample sizes or proportions close to 0 or 1.
}
\details{
The function implements the Miettinen-Nurminen method to compute
confidence intervals for the difference between two proportions. This approach:
\itemize{
\item Calculates the Miettinen-Nurminen score test statistic for different
possible values of the proportion difference (delta)
\item Identifies the delta values where the test statistic equals the critical
value corresponding to the desired confidence level
\item Returns these boundary values as the confidence interval limits
}

The method uses a score test with a small-sample correction factor, making it
more accurate than normal approximation methods, especially for small samples
or extreme proportions. The equation for the test statistics is as follows:

\deqn{H_0: \hat{d}-\delta <= 0 \qquad \text{vs.} \qquad H_1: \hat{d}-\delta > 0}

\deqn{ T_\delta = \frac{\hat{p_x} - \hat{p_y} - \delta}{\sigma_{mn}(\delta)}}

where \eqn{\hat{p_*} = s_*/n_*} represent the observed number of successes
divided by the number of participant in that group. The \eqn{\sigma_{mn}(\delta)} is a
function of the delta values and is create with the following equation"
\eqn{\tilde{p_*}} represent the MLE of the proportions.
\deqn{
\sigma_{mn}(\delta) = \sqrt{\left[\frac{\tilde{p_y}(1-\tilde{p_y})}{n_x}+\frac{\tilde{p_x}(1-\tilde{p_x})}{n_y} \right]\left(\frac{N}{N-1}\right)}
}
\eqn{ \tilde{p_x} = 2p\cdot{cos(a)} - \frac{L_2}{3L_3}} and \eqn{ \tilde{p_y} = \tilde{p_x} + \delta}
where:
\itemize{
\item \eqn{p = \pm \sqrt{\frac{L_2^2}{(3L_3)^2} - \frac{L_1}{3L_3}}}
\item \eqn{a = 1/3[\pi + cos^{-1}(q/p^3)]}
\item \eqn{q = \frac{L_2^3}{(3L_3)^3} - \frac{L_1L_2}{6L_3^2} + \frac{L_0}{2L_3}}
\item \eqn{L_3 = n_x + n_y }
\item \eqn{L_2 = (n_x + 2 n_y)\delta - N - (s_x + s_y)}
\item \eqn{L_1 = (n_y\delta - L_3 - 2s_y)\delta + s_x + s_y}
\item \eqn{L_0 = s_y\delta(1-\delta)}
}

For more information about these equations see Miettinen (1985)
}
\examples{
# Generate binary samples
responses <- expand(c(9, 3), c(10, 10))
arm <- rep(c("treat", "control"), times = c(10, 10))

# Calculate 95\% confidence interval for difference in proportions
ci_prop_diff_mn(x = responses, by = arm)

# Calculate 99\% confidence interval
ci_prop_diff_mn(x = responses, by = arm, conf.level = 0.99)

# Calculate the p-value under the null hypothesis delta = -0.1
ci_prop_diff_mn(x = responses, by = arm, delta = -0.1)

# Calculate from a data.frame
data <- data.frame(responses, arm)
ci_prop_diff_mn(x = responses, by = arm, data = data)
}
\references{
Miettinen, O. S., & Nurminen, M. (1985). Comparative analysis of
two rates. Statistics in Medicine, 4(2), 213-226.
}
