##
# ------------------------------------------------------------------------
#
# "fieldboot(arr,func,B,length.block,method,...)" --
#
# Bootstrap of discrete multidimensional random fields.
#
# ------------------------------------------------------------------------
##
#' @aliases fieldboot
#' @title Block Bootstrap of Random Field 
#' @description 
#' Performs a bootstrap analysis of multidimensional array representing a random 
#' field on a lattice, using various block bootstrap methods such as moving block, circular block, or 
#' nonoverlapping block bootstrap.
#' @param arr A multidimensional real-valued array; it represents a random field on a grid of dimension
#' equals to dimension of the \code{arr}.
#' @param func The function applied to each bootstrap sample.
#' @param B A positive integer; the number of bootstrap samples.
#' @param length.block 
#' A positive integer or vector of integers; it specified the block lengths for blocks.
#'  If a scalar is provided, the same block length is used for all dimensions.
#' @param method The method for array reconstruction:
#'  * \code{"movingblock"} - Moving Block Bootstrap,
#'  * \code{"nonoverlapping"} - Nonoverlapping Block Bootstrap,
#'  * \code{"circular"} - Circular Block Bootstrap (obtained by wrapping the field on the torus). 
#'  Default is \code{"movingblock"}.
#' @param ... Optional additional arguments for the \code{func} function.
#' @details 
#' The \code{fieldboot} function resamples hyper-rectangles constructed using either
#' moving blocks, nonoverlapping blocks or circular blocks to construct a bootstrap field of the same 
#' dimension as the original one. Then it
#' applies the specified \code{func} to bootstrap samples of the 
#' provided data array. The \code{length.block} parameter determines the size of the blocks 
#' used in the bootstrap method. The \code{method} parameter specifies the type of block 
#' bootstrap to use. This function is useful for assessing the variability and distribution 
#' properties of a statistic in the context of random fields.
#' @return  
#' Returns an object of class \code{boodd}.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#'  Bertail, P.  Politis, D. N. Rhomari, N. (2000). Subsampling continuous 
#' parameter random fields and a Bernstein inequality,  \emph{Statistics}, 
#' \bold{33}, 367-392.
#' 
#' Nordman, D.J.  Lahiri, S.N.(2004). On optimal spatial subsample size for 
#' variance estimation, \emph{The Annals of Statistics}, \bold{32}, 1981-2027.
#' 
#'Politis,  D.N. Romano, J.P. (1993). Nonparametric Resampling for Homogeneous 
#' Strong Mixing Random Fields, \emph{J. Multivar. Anal.}, \bold{47}, 301-328.
#' 
#' @seealso \code{\link{blockboot}},
#' \code{\link{jackVarField}},
#' \code{\link{field.sub}}, \code{\link{fieldbootP}}.
#' @keywords "Bootstrap"  "Random fields"
#' @examples 
#' \donttest{
#' set.seed(123)
#' arr <- array(rnorm(1000), dim = c(10, 10, 10))
#' res <- fieldboot(arr, mean, B = 100, length.block = c(2, 2, 2))
#' plot(res)
#' }
#' @export
#' ##
fieldboot <- function(arr,func,B,length.block,...,method=c("movingblock","nonoverlapping","circular")) {
  # Test the value returned by func
  y <- func(arr,...)
  if (!is.vector(y)) {
    stop("Function 'func' must return a vector")
  }
  len <- length(y)
  cnames <- names(y)
  res <- matrix(nrow=B,ncol=len)
  
  # Data lengths n_1, n_2, ..., n_d
  dlens <- dim(arr)
  ndims <- length(dlens)
  if (ndims < 2) {
    stop("expected at least 2-dimensional array")
  }
  if (length(length.block) == 1) {
    length.block = rep(length.block,ndims)
  }
  if (length(length.block) != ndims) {
    stop("wrong number of block lengths")
  }
  if (any(length.block > dlens)) {
    stop("block lengths must be less than data lengths in all dimensions")
  }
  if (any(length.block <= 0)) {
    stop("block lengths must be positive")
  }
  method <- match.arg(method)
  # Block lengths b_1, b_2, ..., b_d
  blens <- length.block
  last <- dlens-blens+1
  fulldims <- ((dlens-1)%/%blens +1)*blens
  
  # Array indices as string "1:n_1,1:n_2,...,1:n_d"
  indstr <- blockString(rep(1,ndims),dlens)
  
  # Build the tile indices (as strings) defining the covering
  # of the reconstructed array
  tiles <- blockTiles(blens,dlens)
  tnum <- length(tiles)
  
  # Initialize the new array (a slightly bigger array containing an
  # entire number of tiles that will be trimmed later)
  narr <- array(0,dim=fulldims)
  origs <- integer(ndims)
  if (method[1] == "circular") {
    # Ensure circularity
    arr <- completeArray(arr,blens)
  }
  
  for (i in 1:B) {
    for (j in 1:tnum) {
      tile <- tiles[j]
      
      # Draw a random block in the original array
      for (k in 1:ndims) {
        if (method[1] == "movingblock") {
          last <- dlens[k]-blens[k]+1
          origs[k] <- sample(1:last,1)
        } else if (method[1] == "nonoverlapping") {
          origs[k] <- sample(1+blens[k]*(0:(dlens[k]%/%blens[k]-1)),1)
        } else if (method[1] == "circular") {
          last <- dlens[k]
          origs[k] <- sample(1:last,1)
        }
      }
      rndblock <- blockString(origs,blens)
      
      # Build the command string
      cmd <- paste0("narr[",tile,"] <- ","arr[",rndblock,"]")
      # Eval the command
      eval(parse(text=cmd))
    }
    
    # Apply the statistics to the rebuilt array (trimmed to the
    # original dimensions)
    resCmd <- paste0("res[",i,",] <- func(narr[",indstr,"],...)")
    eval(parse(text=resCmd))
  }
  
  if (len == 1) {
    res <- as.vector(res)
  } else if (!is.null(cnames)) {
    colnames(res) <- cnames
  }
  obj <- list(s=res,Tn=y)
  class(obj) <- "boodd"
  attr(obj,"kind") <- c("randomfield",method[1])
  attr(obj,"func") <- func
  return(obj)
}