#' @title Bland-Altman plot limits for R
#'
#' @description Works out plot limits for the Bland-Altman plots. Depends on the blandr.statistics function in the package.
#'
#' @author Deepankar Datta <deepankar.datta@gmail.com>
#'
#' @param statistics.results A list of statistics generated by the blandr.statistics function: see the function's return list to see what variables are passed to this function
#' @param lowest_y_axis (Optional) Defaults to NULL If given a continuous variable will use this as the lower boundary of the y axis. Useful if need multiple plots with equivalent y-axes.
#' @param highest_y_axis (Optional) Defaults to NULL If given a continuous variable will use this as the upper boundary of the y axis. Useful if need multiple plots with equivalent y-axes.
#' @return x_upper The upper limit of the X-axis
#' @return x_lower The lower limit of the X-axis
#' @return y_upper The upper limit of the Y-axis
#' @return y_lower The lower limit of the Y-axis
#'
#' @examples
#' # Generates two random measurements
#' measurement1 <- rnorm(100)
#' measurement2 <- rnorm(100)
#'
#' # Passes data to the blandr.statistics function to generate Bland-Altman statistics
#' statistics.results <- blandr.statistics( measurement1 , measurement2 )
#'
#' # Calls the function
#' blandr.plot.limits( statistics.results )
#'
#' @export

blandr.plot.limits <- function(statistics.results, lowest_y_axis = FALSE, highest_y_axis = FALSE) {

    # Calculates a margin for error - therefore labelled bounds
    x_bounds <- (max(statistics.results$means) - min(statistics.results$means)) * 0.1  ### this is a margin of error for drawing the x axis
    y_bounds <- (max(statistics.results$differences) - min(statistics.results$differences)) *
        0.1  ### this is a margin of error for drawing the y axis

    # Sets limits for plot on the x-axis
    x_upper <- max(statistics.results$means) + x_bounds
    x_lower <- min(statistics.results$means) - x_bounds

    # Sets limits for plot on the y-axis
    y_upper <- max(statistics.results$differences) + y_bounds
    y_lower <- min(statistics.results$differences) - y_bounds

    # Ensures that y-axis includes the whole range of confidence intervals
    if (y_upper <= statistics.results$upperLOA_upperCI) {
        y_upper <- statistics.results$upperLOA_upperCI + y_bounds
    }
    if (y_lower >= statistics.results$lowerLOA_lowerCI) {
        y_lower <- statistics.results$lowerLOA_lowerCI - y_bounds
    }

    # If the user has requested specific limits to the y-axis the following 2 lines execute
    # this
    if (highest_y_axis != FALSE) {
        y_upper <- highest_y_axis
    }
    if (lowest_y_axis != FALSE) {
        y_lower <- lowest_y_axis
    }

    return(list(x_upper = x_upper, x_lower = x_lower, y_upper = y_upper, y_lower = y_lower)  #CLOSE OF LIST
)  #CLOSE OF RETURN

    # END OF FUNCTION
}


