% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RVineCopSelect.R
\name{RVineCopSelect}
\alias{RVineCopSelect}
\title{Sequential Pair-Copula Selection and Estimation for R-Vine Copula Models}
\usage{
RVineCopSelect(
  data,
  familyset = NA,
  Matrix,
  selectioncrit = "AIC",
  indeptest = FALSE,
  level = 0.05,
  trunclevel = NA,
  weights = NA,
  rotations = TRUE,
  se = FALSE,
  presel = TRUE,
  method = "mle",
  cores = 1
)
}
\arguments{
\item{data}{N x d data matrix (with uniform margins).}

\item{familyset}{integer vector of pair-copula families to select from.
The vector has to include at least one
pair-copula family that allows for positive and one that allows for negative
dependence. Not listed copula families might be included to better handle
limit cases.  If \code{familyset = NA} (default), selection among all
possible families is performed. If a vector of negative numbers is provided,
selection among all but \code{abs(familyset)} is performed. Coding of
pair copula families is the same as in \code{\link[=BiCop]{BiCop()}}.}

\item{Matrix}{lower or upper triangular d x d matrix that defines the R-vine
tree structure.}

\item{selectioncrit}{Character indicating the criterion for pair-copula
selection. Possible choices: \code{selectioncrit = "AIC"} (default),
\code{"BIC"}, or \code{"logLik"} (see \code{\link[=BiCopSelect]{BiCopSelect()}}).}

\item{indeptest}{Logical; whether a hypothesis test for the independence of
\code{u1} and \code{u2} is performed before bivariate copula selection
(default: \code{indeptest = FALSE}; see \code{\link[=BiCopIndTest]{BiCopIndTest()}}).  The
independence copula is chosen for a (conditional) pair if the null
hypothesis of independence cannot be rejected.}

\item{level}{numeric; significance level of the independence test (default:
\code{level = 0.05}).}

\item{trunclevel}{integer; level of truncation.}

\item{weights}{Numerical; weights for each observation (optional).}

\item{rotations}{logical; if \code{TRUE}, all rotations of the families in
\code{familyset} are included.}

\item{se}{Logical; whether standard errors are estimated (default: \code{se = FALSE}).}

\item{presel}{Logical; whether to exclude families before fitting based on
symmetry properties of the data. Makes the selection about 30\\% faster
(on average), but may yield slightly worse results in few special cases.}

\item{method}{indicates the estimation method: either maximum
likelihood estimation (\code{method = "mle"}; default) or inversion of
Kendall's tau (\code{method = "itau"}). For \code{method = "itau"} only
one parameter families and the Student t copula can be used (\verb{family = 1,2,3,4,5,6,13,14,16,23,24,26,33,34} or \code{36}). For the t-copula,
\code{par2} is found by a crude profile likelihood optimization over the
interval (2, 10].}

\item{cores}{integer; if \code{cores > 1}, estimation will be parallelized
within each tree (using \code{\link[parallel:clusterApply]{parallel::parLapply()}}). Note that
parallelization causes substantial overhead and may be slower than
single-threaded computation when dimension, sample size, or family set are
small or \code{method = "itau"}.}
}
\value{
An \code{\link[=RVineMatrix]{RVineMatrix()}} object with the selected families
(\code{RVM$family}) as well as sequentially
estimated parameters stored in \code{RVM$par} and \code{RVM$par2}. The object
is augmented by the following information about the fit:
\item{se, se2}{standard errors for the parameter estimates  (if
\code{se = TRUE}; note that these are only approximate since they do not
account for the sequential nature of the estimation,}
\item{nobs}{number of observations,}
\item{logLik, pair.logLik}{log likelihood (overall and pairwise)}
\item{AIC, pair.AIC}{Akaike's Informaton Criterion (overall and pairwise),}
\item{BIC, pair.BIC}{Bayesian's Informaton Criterion (overall and pairwise),}
\item{emptau}{matrix of empirical values of Kendall's tau,}
\item{p.value.indeptest}{matrix of p-values of the independence test.}#'
}
\description{
This function fits a R-vine copula model to a d-dimensional copula data set.
Pair-copula families are selected using \code{\link[=BiCopSelect]{BiCopSelect()}} and
estimated sequentially.
}
\details{
R-vine copula models with unknown structure can be specified using
\code{\link[=RVineStructureSelect]{RVineStructureSelect()}}.
}
\note{
For a comprehensive summary of the vine copula model, use
\code{summary(object)}; to see all its contents, use \code{str(object)}.
}
\examples{

# define 5-dimensional R-vine tree structure matrix
Matrix <- c(5, 2, 3, 1, 4,
            0, 2, 3, 4, 1,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 1)
Matrix <- matrix(Matrix, 5, 5)
# define R-vine pair-copula family matrix
family <- c(0, 1, 3, 4, 4,
            0, 0, 3, 4, 1,
            0, 0, 0, 4, 1,
            0, 0, 0, 0, 3,
            0, 0, 0, 0, 0)
family <- matrix(family, 5, 5)
# define R-vine pair-copula parameter matrix
par <- c(0, 0.2, 0.9, 1.5, 3.9,
         0, 0, 1.1, 1.6, 0.9,
         0, 0, 0, 1.9, 0.5,
         0, 0, 0, 0, 4.8,
         0, 0, 0, 0, 0)
par <- matrix(par, 5, 5)
# define second R-vine pair-copula parameter matrix
par2 <- matrix(0, 5, 5)

## define RVineMatrix object
RVM <- RVineMatrix(Matrix = Matrix, family = family,
                   par = par, par2 = par2,
                   names = c("V1", "V2", "V3", "V4", "V5"))

## simulate a sample of size 500 from the R-vine copula model
set.seed(123)
simdata <- RVineSim(500, RVM)

## determine the pair-copula families and parameters
RVM1 <- RVineCopSelect(simdata, familyset = c(1, 3, 4, 5 ,6), Matrix)

## see the object's content or a summary
str(RVM1)
summary(RVM1)

## inspect the fitted model using plots
\dontrun{plot(RVM1)  # tree structure}

contour(RVM1)  # contour plots of all pair-copulas

}
\references{
Brechmann, E. C., C. Czado, and K. Aas (2012). Truncated regular
vines in high dimensions with applications to financial data. Canadian
Journal of Statistics 40 (1), 68-85.

Dissmann, J. F., E. C. Brechmann, C. Czado, and D. Kurowicka (2013).
Selecting and estimating regular vine copulae and application to financial
returns. Computational Statistics & Data Analysis, 59 (1), 52-69.
}
\seealso{
\code{\link[=RVineMatrix]{RVineMatrix()}},
\code{\link[=BiCop]{BiCop()}},
\code{\link[=BiCopSelect]{BiCopSelect()}},
\code{\link[=plot.RVineMatrix]{plot.RVineMatrix()}},
\code{\link[=contour.RVineMatrix]{contour.RVineMatrix()}}
}
\author{
Eike Brechmann, Thomas Nagler
}
