// File modified from <boost/math/special_functions/digamma.hpp> to remove 
// dependency on Boost. For simplicity, only double precision is supported.
// Optimisations for integer inputs have been removed.

// The original header contains the following notice:
//  (C) Copyright John Maddock 2006.
//  Use, modification and distribution are subject to the
//  Boost Software License, Version 1.0

/*
 Boost Software License - Version 1.0 - August 17th, 2003
 
 Permission is hereby granted, free of charge, to any person or organization
 obtaining a copy of the software and accompanying documentation covered by
 this license (the "Software") to use, reproduce, display, distribute,
 execute, and transmit the Software, and to prepare derivative works of the
 Software, and to permit third-parties to whom the Software is furnished to
 do so, all subject to the following:
 
 The copyright notices in the Software and this entire statement, including
 the above license grant, this restriction and the following disclaimer,
 must be included in all copies of the Software, in whole or in part, and
 all derivative works of the Software, unless such copies or derivative
 works are solely in the form of machine-executable object code generated by
 a source language processor.
 
 THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 DEALINGS IN THE SOFTWARE.
 */

#ifndef _DIGAMMA_HPP
#define _DIGAMMA_HPP

#include <Rcpp.h>

const double PI = 3.14159265358979323846;

//
// Begin by defining the smallest value for which it is safe to
// use the asymptotic expansion for digamma:
//
const unsigned DIGAMMA_LARGE_LIM = 10;

//
// Implementations of the asymptotic expansion come next,
// the coefficients of the series have been evaluated
// in advance at high precision, and the series truncated
// at the first term that's too small to effect the result.
// Note that the series becomes divergent after a while
// so truncation is very important.
//

// Loop-unrolled polynomial evaluation
inline double evaluate_polynomial_6(const double a[6], const double x) {
  return ((((a[5]*x + a[4]) * x + a[3]) * x + a[2]) * x + a[1]) * x + a[0];
}
inline double evaluate_polynomial_7(const double a[7], const double x) {
  return (((((a[6] * x + a[5]) * x + a[4]) * x + a[3]) * x + a[2]) * x + a[1]) * x + a[0];
}
inline double evaluate_polynomial_8(const double a[7], const double x) {
  return ((((((a[7] * x + a[6]) * x + a[5]) * x + a[4]) * x + a[3]) * x + a[2]) * x + a[1]) * x + a[0];
}


//
// 17-digit precision for x >= 10:
//
template <class T>
inline T digamma_imp_large(T x)
{
  static const T P[] = {
    0.083333333333333333333333333333333333333333333333333,
    -0.0083333333333333333333333333333333333333333333333333,
    0.003968253968253968253968253968253968253968253968254,
    -0.0041666666666666666666666666666666666666666666666667,
    0.0075757575757575757575757575757575757575757575757576,
    -0.021092796092796092796092796092796092796092796092796,
    0.083333333333333333333333333333333333333333333333333,
    -0.44325980392156862745098039215686274509803921568627
  };
  x -= 1;
  T result = log(x);
  result += 1 / (2 * x);
  T z = 1 / (x*x);
  result -= z *evaluate_polynomial_8(P, z);
  return result;
}


//
// 18-digit precision:
//
template <class T>
T digamma_imp_1_2(T x)
{
  //
  // Now the approximation, we use the form:
  //
  // digamma(x) = (x - root) * (Y + R(x-1))
  //
  // Where root is the location of the positive root of digamma,
  // Y is a constant, and R is optimised for low absolute error
  // compared to Y.
  //
  // Maximum Deviation Found:               1.466e-18
  // At double precision, max error found:  2.452e-17
  
  static const float Y = 0.99558162689208984F;
  
  static const T root1 = T(1569415565) / 1073741824uL;
  static const T root2 = (T(381566830) / 1073741824uL) / 1073741824uL;
  static const T root3 = 0.9016312093258695918615325266959189453125e-19;
  
  static const T P[] = {
    0.25479851061131551,
    -0.32555031186804491,
    -0.65031853770896507,
    -0.28919126444774784,
    -0.045251321448739056,
    -0.0020713321167745952
  };
  static const T Q[] = {
    1.0,
    2.0767117023730469,
    1.4606242909763515,
    0.43593529692665969,
    0.054151797245674225,
    0.0021284987017821144,
    -0.55789841321675513e-6
  };
  
  T g = x - root1;
  g -= root2;
  g -= root3;
  T r = evaluate_polynomial_6(P, T(x-1)) / evaluate_polynomial_7(Q, T(x-1));
  T result = g * Y + g * r;
  
  return result;
}


double digamma(double x)
{
  //
  // This handles reflection of negative arguments, and all our
  // error handling, then forwards to the T-specific approximation.
  //
  
  double result = 0;
  //
  // Check for negative arguments and use reflection:
  //
  if(x <= -1)
  {
    // Reflect:
    x = 1 - x;
    // Argument reduction for tan:
    double remainder = x - floor(x);
    // Shift to negative if > 0.5:
    if(remainder > 0.5)
    {
      remainder -= 1;
    }
    //
    // check for evaluation at a negative pole:
    //
    if(remainder == 0)
      Rcpp::stop("digamma: Attempt to evaluate function at pole");
    
    result = PI / tan(PI * remainder);
  }
  if(x == 0)
    Rcpp::stop("digamma: Attempt to evaluate function at pole");
  
  //
  // If we're above the lower-limit for the
  // asymptotic expansion then use it:
  //
  if(x >= DIGAMMA_LARGE_LIM)
  {
    result += digamma_imp_large(x);
  }
  else
  {
    //
    // If x > 2 reduce to the interval [1,2]:
    //
    while(x > 2)
    {
      x -= 1;
      result += 1/x;
    }
    //
    // If x < 1 use recurrence to shift to > 1:
    //
    while(x < 1)
    {
      result -= 1/x;
      x += 1;
    }
    result += digamma_imp_1_2(x);
  }
  return result;
}

#endif // _DIGAMMA_HPP
