#' @title Local Gauss VB
#' @description
#'  A variational Bayesian algorithm, based on the Gauss Spike-and-Slab prior, is tailored for
#'    multi-source heterogeneous models and focuses on variable selection exclusively for the homogeneous
#'    covariates.
#'
#' @param X Homogeneous covariates
#' @param Z Heterogeneous covariates
#' @param Y Response covariates
#' @param max_iter Maximum number of iterations, Defaut:1000
#' @param tol Algorithm convergence tolerance, Defaut:1e-6
#' @param a A prior of Beta distribution, Defaut:1
#' @param b A prior of Beta distribution, Defaut:10
#' @param lambda A prior of Laplace distribution, Defaut:1
#'
#' @return The mean of the homogeneity coefficient:mu;
#'         The variance of homogeneity coefficient:sigma;
#'         Selection coefficient:gamma;
#'         Mean and covariance of heterogeneity coefficients:m, s2.
#'
vb_gauss_local <- function(X, Z, Y, max_iter, tol, a=1, b=10, lambda=1) {

  n <- dim(X)[1]
  p <- dim(X)[2]
  K <- dim(X)[3]
  q <- dim(Z)[2]

  mu <- matrix(rep(1,p),nrow=p,ncol=1)
  sigma <- matrix(rep(1,p),nrow=p,ncol=1)
  gamma <- matrix(rep(0.5,p),nrow=p,ncol=1)
  s2 <- array(data=diag(q),dim=c(q,q,K))
  m <- matrix(1,nrow=q,ncol=K)

  old_entr <- entropy(gamma)

  all_X <- X[,,1]
  all_Z <- Z[,,1]
  all_Zm <- Z[,,1] %*% m[,1]
  all_Y <- Y[,1]

  for (k in 2:K) {
    all_X <- rbind(all_X,X[,,k])
    all_Z <- rbind(all_Z,Z[,,k])
    all_Y <- c(all_Y,Y[,k])
    all_Zm <- c(all_Zm,Z[,,k] %*% m[,k])
  }

  YX_vec <- t(all_Y-all_Zm) %*% all_X
  half_diag <- 0.5 * gram_diag(all_X)
  approx_mean <- gamma * mu
  X_appm <- all_X %*% approx_mean

  const_lodds <- (log(a) - log(b)) + 0.5
  const_lodds <- const_lodds - log(lambda)
  sigma <- 1 / sqrt(2 * half_diag + 1 / lambda^2)

  for (i in 1:max_iter) {
    all_Zm <- Z[,,1] %*% m[,1]
    for (k in 2:K) {
      all_Zm <- c(all_Zm,Z[,,k] %*% m[,k])
    }
    YX_vec <- t(all_Y-all_Zm) %*% all_X
    for (j in 1:length(mu)) {

      X_appm <- X_appm - approx_mean[j] * all_X[, j]

      mu[j] <- sigma[j]^2 * (YX_vec[j] - as.numeric(all_X[, j] %*% X_appm))
      gamma[j] <- sigmoid(const_lodds + log(sigma[j]) + 0.5 * (mu[j] / sigma[j])^2)

      approx_mean[j] <- gamma[j] * mu[j]
      X_appm <- X_appm + approx_mean[j] * all_X[, j]
    }

    for (k in 1:K){
      s2[,,k] <- solve(t(Z[,,k]) %*% Z[,,k]+diag(ncol(Z[,,k])))
      m[,k] <- s2[,,k] %*% t(t(Y[,k] - X[,,k] %*% (mu*gamma)) %*% Z[,,k])
    }

    new_entr <- entropy(gamma)
    if (max(abs(new_entr - old_entr)) <= tol) {
      break
    } else {
      old_entr <- new_entr
    }
  }
  return(list(mu = mu, gamma = gamma, m = m, sigma=sigma, s2=s2))
}
