#' @title Lexicographic order position to binary order position
#' @description Given the lexicographic order position of a coalition, this function returns the  corresponding binary order position.
#' @param n Number of players.
#' @param Nlex A lexicographic order position, as an integer between 1 and \eqn{2^{\code{n}}-1}.
#' @return The corresponding binary order position, as an integer between 1 and \eqn{2^{\code{n}}-1}.
#' @details Lexicographic order arranges coalitions in ascending order according to size, and applies lexicographic order to break ties among coalitions of the same size. The binary order position of a coalition \eqn{S\in 2^N} is given by \eqn{\sum_{i\in S} 2^{i-1}}.
#' @examples
#' codelex2bin(5, 4)
#' @seealso \link{bin2lex}, \link{codebin2lex}, \link{lex2bin}
#' @export

codelex2bin <- function(n, Nlex) {

  ################################
  ### Comprobación datos entrada###
  ################################

  if (Nlex > (2^n - 1)) {
    stop("'Nlex' cannot be greater than 2^n-1.")
  }

  ################################
  ##### Creación de variables######
  ################################

  Nbin <- 0
  coalition <- rep(0, n)
  control <- 0
  jug <- 0
  ante <- 0
  post <- 0
  ii <- 0
  tope <- 0

  ################################
  ###### Cuerpo de la función######
  ################################

  # Bucle para saber el numero de jugadores
  # Comparo C con los acumulados de los números combinatorios
  # Los (n sobre 1) primeros, luego los (n sobre 2), etc. Todos estos estarán en la coalición.
  # En lexicográfico las coaliciones se ordenan "inicialmente" por número de jugadores.

  while (control < 1) {
    jug <- jug + 1
    ante <- ante + choose(n, jug - 1)
    post <- post + choose(n, jug)
    if (Nlex <= post) {
      control <- 1
    }
  }

  #########################################

  # Ahora calculo la coalicion correspondiente
  # como número en binario: 1 en la posición si el jugador ii pertenece a la
  # coalición y cero en otro caso


  while (jug > 0) {
    ii <- ii + 1
    # Miro si en la posición ii va un cero o un uno.
    # Si fuese un uno, el numero de la coalicion estaría entre "ante" y "ante" más el numero
    # de coaliciones de tamaño n-ii con jug-1 unos el resto ceros.
    # Permutaciones con repeticion de n-ii objetos (ceros y unos)  con (jug-1) y
    # (n-ii-jug+1) repeticiones.
    tope <- factorial(n - ii) / (factorial(jug - 1) * factorial(n - ii - jug + 1))
    if (Nlex <= ((ante - 1) + tope)) {
      coalition[ii] <- 1
      jug <- jug - 1
    } else {
      ante <- ante + tope
    }
  }

  # Ahora obtengo el numero de esa coalicion en binario

  for (ii in 1:n)
  {
    # Multiplico el coeficiente en la posición ii por la potencia 2^(ii-1).
    Nbin <- Nbin + 2^(ii - 1) * coalition[ii]
  }

  ################################
  ###### Salidas de la función#####
  ################################

  return(Nbin)
} # Fin de la función
