#' Sample size calculator for estimation of population mean under SRS
#'
#' @description The SscSrsMean function calculates the sample size required for estimation of population mean based upon the availability of prior information on coefficient of variation (CV) of the population under Simple Random Sampling (SRS) with or without replacement sampling design for a given confidence level and relative error. Further, the function provides the required sample size when population CV is known whereas it provides both required sample size and additional sample units to be surveyed when population CV is unknown under both SRS with or without replacement.
#'
#' @param CV_known Coefficient of variation (CV) of the population is known or unknown. If it is known, then write TRUE otherwise write FALSE.
#' @param replace replace=TRUE, if sampling design is SRSWR and replace=FALSE, if sampling design is SRSWOR.
#' @param alpha Level of significance value, alpha=0.01 at 1 percent level of significance and alpha=0.05 at 5 percent level of significance.
#' @param re Relative error in the estimation of population mean (e.g. 0.1, 0.5).
#' @param N Population size. When sampling design is SRSWR, then write NA.
#' @param preliminary_sample Values of the study variable for the preliminary sample. When CV of the population is known, then write NA.
#'
#' @return A list with the following components:
#' \item{Required sample size}{Sample size required for estimation of population mean when population CV is known or unknown.}
#' \item{Additional sample units to be surveyed}{Additional sample units to be surveyed over the preliminary sample for estimation of population mean when population CV is unknown. If the value of additional sample units to be surveyed is negative, then preliminary sample is considered as the final sample.}
#'
#' @details This function returns the sample size required for estimation of population mean when population CV is known. Besides this, the function returns both required sample size and additional sample units to be surveyed when population CV is unknown under both SRS with or without replacement sampling design.
#'
#' @references
#' Cochran, W. G. (1977). \emph{Sampling Techniques, 3rd Edition}. New York: John Wiley & Sons, Inc.
#'
#' Singh, D. and Chaudhary, F.S. (1986). \emph{Theory and Analysis of Sample Survey Designs}. New York: John Wiley & Sons, Inc.
#'
#' Sukhatme, P.V., Sukhatme, B.V., Sukhatme, S. and Asok, C. (1984). \emph{Sampling Theory of Surveys with Applications}. Iowa State University Press, Ames and Indian Society of Agricultural Statistics, New Delhi.
#'
#' @examples
#' # Calculate sample size for SRSWOR design when population CV is known
#' SscSrsMean(TRUE, FALSE, 0.05, 0.2, 100, NA)
#' # Calculate sample size for SRSWOR design when population CV is unknown
#' preliminary_sample =c(12, 14, 8, 5, 36, 24, 18, 17, 6, 9)
#' SscSrsMean(FALSE, FALSE, 0.05, 0.2, 100, preliminary_sample)
#' # Calculate sample size for SRSWR design when population CV is known
#' SscSrsMean(TRUE, TRUE, 0.05, 0.2, NA, NA)
#' # Calculate sample size for SRSWR design when population CV is unknown
#' preliminary_sample =c(12, 14, 8, 5, 36, 24, 18, 17, 6, 9)
#' SscSrsMean(FALSE, TRUE, 0.05, 0.2, NA, preliminary_sample)
#' @export
#' @importFrom stats qnorm
#' @importFrom stats qt
#' @importFrom stats sd
SscSrsMean=function(CV_known,replace,alpha,re,N,preliminary_sample)
{
  # Calculate sample size for SRSWOR design when population CV is known
  if(CV_known==TRUE && replace==FALSE)
  {
    cv=as.numeric(readline(prompt = "Enter the known CV value: "))
    Z=qnorm(alpha/2,lower.tail = FALSE)
    size=round((N*cv^2)/(cv^2+(N*re^2/Z^2)),0)
    output <- list("Required sample size"=size)
    return(output)
  }
  # Calculate sample size for SRSWOR design when population CV is unknown
  if(CV_known==FALSE && replace==FALSE)
  {
    cv=sd(preliminary_sample)/mean(preliminary_sample)
    t=qt(alpha/2,(length(preliminary_sample)-1),lower.tail = FALSE)
    size=round((N*cv^2)/(cv^2+(N*re^2/t^2)),0)
    additional_size=size-length(preliminary_sample)
    output <- list("Required sample size"=size, "Additional sample units to be surveyed"=additional_size)
    return(output)
  }
  # Calculate sample size for SRSWR design when population CV is known
  if(CV_known==TRUE && replace==TRUE)
  {
    cv=as.numeric(readline(prompt = "Enter the known CV value: "))
    Z=qnorm(alpha/2,lower.tail = FALSE)
    size=round(((Z*cv)/re)^2, 0)
    output <- list("Required sample size"=size)
    return(output)
  }
  # Calculate sample size for SRSWR design when population CV is unknown
  else if(CV_known==FALSE && replace==TRUE)
  {
    cv=sd(preliminary_sample)/mean(preliminary_sample)
    t=qt(alpha/2,(length(preliminary_sample)-1),lower.tail = FALSE)
    size=round(((t*cv)/re)^2, 0)
    additional_size=size-length(preliminary_sample)
    output <- list("Required sample size"=size, "Additional sample units to be surveyed"=additional_size)
    return(output)
  }
}
