#' Simulates Network Dynamics under Random Node Removal Perturbations.
#'
#' This function simulates the behavior of a network undergoing multiple uniform removal of nodes.
#' Starting with an initial healthy network, it calculates the network's trajectory by solving the system of
#' ordinary differential equations (ODEs) after each perturbation. Nodes are removed randomly and uniformly, followed by
#' dimension reduction using the specified reduction function.
#'
#'
#' @param system A function defining the system's dynamics: `system(time, x, params)` which returns a list of state deltas `dx`.
#' @param M The initial adjacency matrix of the network.
#' @param x0 Initial conditions of the network's nodes (numeric vector).
#' @param initial_params Either a list of initial parameters, or a function of type `f(M) -> list` that takes the adjacency matrix of the network as input and returns the initial parameters of the system.
#' @param update_params A function of type `f(list) -> list` that receives the list of parameters after each perturbation, and returns a new list of parameters. Defaults to `identity`.
#' @param reduction A reduction function applied to the ODE solution. This can be `identity` (for all node states) or functions like `mean` or `median`. The function signature should be either `f(numeric) -> numeric` or `f(matrix) -> matrix`, depending on whether `only.final.state` is `TRUE` or `FALSE`.
#' @param removal_order The removal order of the nodes. Leave NULL for a random removal order.
#' @param ... Additional arguments passed to the ODE solver (e.g., `method`, `atol`).
#'
#' @return Depending on `to.numeric`, returns either a list or a numeric matrix, representing the system's state across multiple perturbations.
#' If `to.numeric` is `FALSE`, the function returns a list `L`, where `L[[i]]`
#' represents the final state of the system after the `i`-th perturbation. If `TRUE`, the list is converted to a
#' numeric matrix before being returned.#'
#' @export
#' @examples
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    Rato::node.removal.thread( Rato::Michaelis.Menten
#'                              , g$M
#'                              , g$initial_values
#'                              , initial_params = list('f' = 1, 'h'=2, 'B'=0.1))
#'
node.removal.thread <- function( system                   # Dynamics of the system
                                , M                       # Adjacency matrix of the graph
                                , x0                      # Initial value
                                , initial_params = list() # Either a list, or a function: M -> list() (Defaults to an empty list)
                                , update_params = identity # A function to update the parameters after the perturbation happens (Defaults to identity)
                                , reduction = identity    # Dimension reduction function (Defaults to identity)
                                , removal_order = NULL # The removal order of the nodes. Leave NULL For random
                                , ...){

  # Sanity checks to verify that the input makes sense.

  assert(nrow(M) == ncol(M), "Error: The graph adjacency matrix needs to be a square matrix.")
  assert(is.list(initial_params) | is.function(initial_params), "Error: initial_params needs to be a List or a function: Matrix -> List. ")

  # Ok, everything seems fine.

  initial_parameter_function <- function(M) {

    if (is.list(initial_params)) {
      params <- initial_params
    }
    else if (is.function(initial_params)) {
      params <- initial_params(M)
    }
    n <- nrow(M) # The number of nodes of the graph

    if(is.null(removal_order)){
        removal_order <- sample(1:n, n) # The order in which to remove the graphs
    }

    if(is.null(params$M)){
      params$M = M
    } else {
      warning("Network adjacency matrix M is already defined in the initial parameter. Not overwriting. Are you certain this should happen?")
    }
    params$removal_order = removal_order
    return(params)
  }

  # The node perturbation functions. This receives the current params, and the current iterations.
  # If then simply removes the next node in line, and returns the updated parameters.
  perturbation <- function(params, iter) {
    removal_order <- params$removal_order
    M <- params$M

    if(length(removal_order) > 0 ) {
      index <- removal_order[1]          # Get the index of the entry that needs to be removed right now
      removal_order <- removal_order[-1] # Remove it from the list of elements that need to be removed in the future
      M[index, ] <- 0 # Remove entries
      M[, index] <- 0 # Remove entries

      params$M = M
      params$removal_order = removal_order

      # Add the list of removed indices to the parameter list.
      if(is.null(params$removed_indices)){
        params$removed_indices <- c(index)
      } else {
        removed_indices <- c(params$removed_indices, index)
        params$removed_indices <- removed_indices
      }

      output <- update_params(params)
      return(output) # Return updated parameters
    }

    # If we removed every single entry, just STOP.
    return(NULL)
  }

  perturbation.thread( system = system
                     , M = M
                     , x0 = x0
                     , perturbation_function = perturbation
                     , initial_parameter_function = initial_parameter_function
                     , reduction = reduction
                     , ...)

}



