% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/FKWC.R
\name{fkwc_multisample}
\alias{fkwc_multisample}
\title{Multisample hypothesis test for difference in covariance operators}
\usage{
fkwc_multisample(data, derivs, g, p = 20)
}
\arguments{
\item{data}{Functional data in \code{matrix} or \code{data.frame} form, where each row
is an observation/function and the columns are the grid.}

\item{derivs}{First order derivative of the functional data in \code{matrix} or
\code{data.frame} form.}

\item{g}{A \code{factor} object that indicates which sample each row of data
belongs to.}

\item{p}{Number of random projections to be generated in order to compute
random projection depths of the data.}
}
\value{
A list consisting of:
\itemize{
\item \verb{$statistic} : The observed test statistic.
\item \verb{$pvalue} : The p-value based on the null distribution.
\item \verb{$method} : A \code{string} \code{"FKWC"}
}
}
\description{
Executes a multisample hypothesis test for differences in covariance
operators using functional Kruskal–Wallis tests for covariance (FKWC) as
outlined by Ramsay and Chenouri (2024). The function requires the first order
derivative of the functional data in order to better detect changes.
}
\examples{
set.seed(111)
t <- seq(0, 1, length.out = 200)

### Generating three sets of Brownian curves with different kernels, each
### kernel generating 20 observations
# Brownian process 1
fd1 <- fda.usc::rproc2fdata(n = 20, t = t, sigma = "brownian",
                           par.list = list(scale = 10, theta = 1))
fd1_d <- fda.usc::fdata.deriv(fd1)

# Brownian process 2
fd2 <- fda.usc::rproc2fdata(n = 20, t = t, sigma = "brownian",
                           par.list = list(scale = 1, theta = 1))
fd2_d <- fda.usc::fdata.deriv(fd2)

# Brownian process 3
fd3 <- fda.usc::rproc2fdata(n = 20, t = t, sigma = "brownian",
                           par.list = list(scale = 1, theta = 5))
fd3_d <- fda.usc::fdata.deriv(fd3)

# Functional data in one matrix and first order derivatives in another matrix
funcdata <- rbind(fd1$data, fd2$data, fd3$data)
funcderivs <- rbind(fd1_d$data, fd2_d$data, fd3_d$data)

fkwc_multisample(data = funcdata,
                 derivs = funcderivs,
                 g = factor(rep(1:3, each = 20)),
                 p = 1000)
}
\references{
Ramsay, K., & Chenouri, S. (2024). Robust nonparametric
hypothesis tests for differences in the covariance structure of functional
data. Canadian Journal of Statistics, 52 (1), 43–78.
https://doi.org/10.1002/cjs.11767
}
\seealso{
\code{\link[fda.usc:fdata.deriv]{fda.usc::fdata.deriv()}}: for approximating the first order
derivative if unavailable.

\code{\link[=fkwc_posthoc]{fkwc_posthoc()}}: for a post-hoc version of this test
}
