% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HiGarrote.R
\name{HiGarrote}
\alias{HiGarrote}
\title{An Automatic Method for the Analysis of Experiments using Hierarchical Garrote}
\usage{
HiGarrote(
  D,
  y,
  heredity = "weak",
  quali_id = NULL,
  quanti_id = NULL,
  quali_sum_idx = NULL,
  user_def_coding = NULL,
  user_def_coding_idx = NULL,
  model_type = 1
)
}
\arguments{
\item{D}{An \eqn{n \times p} data frame for the unreplicated design matrix, where \eqn{n} is the run size and \eqn{p} is the number of factors.}

\item{y}{A vector for the responses corresponding to \code{D}. For replicated experiments, \code{y} should be an \eqn{n \times r} matrix, where \eqn{r} is the number of replicates.}

\item{heredity}{Specifies the heredity principles to be used. Supported options are \code{"weak"} and \code{"strong"}. The default is \code{"weak"}.}

\item{quali_id}{A vector indexing qualitative factors. 
Qualitative factors are coded using Helmert coding. 
Different coding systems are allowed by specifying \code{quali_sum_idx}, \code{user_def_coding}, \code{user_def_coding_idx}.}

\item{quanti_id}{A vector indexing quantitative factors. Quantitative factors are coded using orthogonal polynomial coding.}

\item{quali_sum_idx}{Optional. Indicating which qualitative factors should use sum coding (\code{contr.sum()}).}

\item{user_def_coding}{Optional. A list of user-defined orthogonal coding systems.
Each element must be an orthogonal contrast matrix.}

\item{user_def_coding_idx}{Optional.
A list of indices specifying which qualitative factors should use the corresponding coding systems provided in \code{user_def_coding}.}

\item{model_type}{Integer indicating the type of model to construct.
\describe{
\item{model_type = 1}{The model matrix includes all the main effects of qualitative factors, 
the first two main effects (linear and quadratic) of all the quantitative factors, 
and all the two-factor interactions generated by those main effects.
}
\item{model_type = 2}{The model matrix includes all the main effects of qualitative factors,
the linear effects of all the quantitative factors, 
all the two-factor interactions generated by those main effects, 
and the quadratic effects of all the quantitative factors.}
\item{model_type = 3}{The model matrix includes all the main effects of qualitative factors 
and the linear effects of all the quantitative factors.}
}
The default is \code{model_type = 1}.}
}
\value{
The function returns a list with:
\describe{
\item{\code{nng_estimate}}{A vector for the nonnegative garrote estimates of the identified effects.}
\item{\code{U}}{A model matrix of \code{D}.}
\item{\code{pred_info}}{A list containing information needed for future predictions.}
}
}
\description{
`HiGarrote()` provides an automatic method for analyzing experimental data. 
This function applies the nonnegative garrote method to select important effects while preserving their hierarchical structures.
It first estimates regression parameters using generalized ridge regression, where the ridge parameters are derived from a Gaussian process prior placed on the input-output relationship. 
Subsequently, the initial estimates will be used in the nonnegative garrote for effects selection.
}
\examples{
# Cast fatigue experiment
data(cast_fatigue)
X <- cast_fatigue[,1:7]
y <- cast_fatigue[,8]
fit_Hi <- HiGarrote::HiGarrote(X, y)
fit_Hi$nng_estimate

# Blood glucose experiment
data(blood_glucose)
X <- blood_glucose[,1:8]
y <- blood_glucose[,9]
fit_Hi <- HiGarrote::HiGarrote(X, y, quanti_id = 2:8) 
fit_Hi$nng_estimate

\donttest{
# Router bit experiment --- Use default Helmert coding
data(router_bit)
X <- router_bit[, 1:9]
y <- router_bit[,10]
fit_Hi <- HiGarrote::HiGarrote(X, y, quali_id = c(4,5))
fit_Hi$nng_estimate

# Router bit experiment --- Use sum coding
fit_Hi <- HiGarrote::HiGarrote(X, y, quali_id = c(4,5), quali_sum_idx = c(4,5))
fit_Hi$nng_estimate

# Router bit experiment --- Use user-defined coding system for qualitative factors
fit_Hi <- HiGarrote::HiGarrote(X, y, quali_id = c(4,5),
 user_def_coding = list(matrix(c(-1,-1,1,1,1,-1,-1,1,-1,1,-1,1), ncol = 3)),
 user_def_coding_idx = list(c(4,5)))
fit_Hi$nng_estimate

# Resin experiment --- Use model_type = 2
data(resin)
X <- resin[,1:9]
y <- log(resin$Impurity)
fit_Hi <- HiGarrote::HiGarrote(X, y, quanti_id = c(1:9), model_type = 2)
fit_Hi$nng_estimate

# Epoxy experiment --- Use model_type = 3
data(epoxy)
X <- epoxy[,1:23]
y <- epoxy[,24]
fit_Hi <- HiGarrote::HiGarrote(X, y, model_type = 3)
fit_Hi$nng_estimate

# Experiments with replicates
# Generate simulated data
data(cast_fatigue)
X <- cast_fatigue[,1:7]
U <- data.frame(model.matrix(~.^2, X)[,-1])
error <- matrix(rnorm(24), ncol = 2) # two replicates for each run
y <- 20*U$A + 10*U$A.B + 5*U$A.C + error
fit_Hi <- HiGarrote::HiGarrote(X, y)
fit_Hi$nng_estimate
}

}
\references{
Yu, W. Y. and Joseph, V. R. (2025). Automated Analysis of Experiments using Hierarchical Garrote. 
\emph{Journal of Quality Technology}, 1-15. \doi{10.1080/00224065.2025.2513508}.
}
