% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reco_lcc.R
\name{ctlcc}
\alias{ctlcc}
\title{Level conditional coherent reconciliation for cross-temporal hierarchies}
\usage{
ctlcc(base, agg_mat, nodes = "auto", agg_order, comb = "ols", res = NULL,
      CCC = TRUE, const = "exogenous", hfbts = NULL, tew = "sum",
      approach = "proj", nn = NULL, settings = NULL, ...)
}
\arguments{
\item{base}{A (\eqn{n \times h(k^\ast+m)}) numeric matrix containing the base forecasts to
be reconciled; \eqn{n} is the total number of variables, \eqn{m} is the max. order of temporal
aggregation, \eqn{k^\ast} is the sum of (a subset of) (\eqn{p-1}) factors of \eqn{m},
excluding \eqn{m}, and \eqn{h} is the forecast horizon for the lowest frequency time series.
The row identifies a time series, and the forecasts in each row are ordered from the
lowest frequency (most temporally aggregated) to the highest frequency.}

\item{agg_mat}{A (\eqn{n_a \times n_b}) numeric matrix representing the cross-sectional
aggregation matrix. It maps the \eqn{n_b} bottom-level (free)
variables into the \eqn{n_a} upper (constrained) variables.}

\item{nodes}{A (\eqn{L \times 1}) numeric vector indicating the number of variables
in each of the upper \eqn{L} levels of the hierarchy. The \emph{default}
value is the string "\code{auto}" which calculates the number of variables in each level.}

\item{agg_order}{Highest available sampling frequency per seasonal cycle (max. order
of temporal aggregation, \eqn{m}), or a vector representing a subset of \eqn{p} factors
of \eqn{m}.}

\item{comb}{A string specifying the reconciliation method. For a complete list, see \link{ctcov}.}

\item{res}{A (\eqn{n \times N(k^\ast+m)}) optional numeric matrix containing the
in-sample residuals at all the temporal frequencies ordered from the lowest frequency
to the highest frequency (columns) for each variable (rows). This matrix is used
to compute some covariance matrices.}

\item{CCC}{A logical value indicating whether the Combined Conditional Coherent reconciled
forecasts reconciliation should include bottom-up forecasts (\code{TRUE}, \emph{default}), or not.}

\item{const}{A string specifying the reconciliation constraints:
\itemize{
\item "\code{exogenous}" (\emph{default}): Fixes the top level of each sub-hierarchy.
\item "\code{endogenous}": Coherently revises both the top and bottom levels.
}}

\item{hfbts}{A (\eqn{n \times mh}) numeric matrix containing high frequency bottom base
forecasts defined by the user. This parameter can be omitted
if only base forecasts are used (see Di Fonzo and Girolimetto, 2024).}

\item{tew}{A string specifying the type of temporal aggregation. Options include:
"\code{sum}" (simple summation, \emph{default}), "\code{avg}" (average),
"\code{first}" (first value of the period), and "\code{last}"
(last value of the period).}

\item{approach}{A string specifying the approach used to compute the reconciled
forecasts. Options include:
\itemize{
\item "\code{proj}" (\emph{default}): Projection approach according to Byron (1978, 1979).
\item "\code{strc}": Structural approach as proposed by Hyndman et al. (2011).
\item "\code{proj_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for projection approach.
\item "\code{strc_osqp}": Numerical solution using \href{https://osqp.org/}{\pkg{osqp}}
for structural approach.
}}

\item{nn}{A string specifying the algorithm to compute non-negative forecasts:
\itemize{
\item "\code{osqp}": quadratic programming optimization
(\href{https://osqp.org/}{\pkg{osqp}} solver).
\item "\code{bpv}": block principal pivoting algorithm.
\item "\code{sntz}": heuristic "set-negative-to-zero" (Di Fonzo and Girolimetto, 2023).
}}

\item{settings}{A list of control parameters.
\itemize{
\item \code{nn = "osqp"} An object of class \code{osqpSettings} specifying settings
for the \href{https://osqp.org/}{\pkg{osqp}} solver. For details, refer to the
\href{https://osqp.org/}{\pkg{osqp} documentation} (Stellato et al., 2020).
\item \code{nn = "bpv"} It includes: \code{ptype} for permutation method ("\code{random}"
or "\code{fixed}", \emph{default}), \code{par} for the number of full exchange rules that
may be attempted (\code{10}, \emph{default}), \code{tol} for the tolerance in convergence
criteria (\code{sqrt(.Machine$double.eps)}, \emph{default}), \code{gtol} for the gradient
tolerance in convergence criteria (\code{sqrt(.Machine$double.eps)}, \emph{default}),
\code{itmax} for the maximum number of algorithm iterations (\code{100}, \emph{default})
}}

\item{...}{
  Arguments passed on to \code{\link[=ctcov]{ctcov}}
  \describe{
    \item{\code{mse}}{If \code{TRUE} (\emph{default}) the residuals used to compute the covariance
matrix are not mean-corrected.}
    \item{\code{shrink_fun}}{Shrinkage function of the covariance matrix, \link{shrink_estim} (\emph{default}).}
  }}
}
\value{
A (\eqn{n \times h(k^\ast+m)}) numeric matrix of cross-temporal reconciled forecasts.
}
\description{
This function implements a forecast reconciliation procedure inspired by the original
proposal by Hollyman et al. (2021) for cross-temporal hierarchies. Level conditional
coherent reconciled forecasts are conditional on (i.e., constrained by) the base
forecasts of a specific upper level in the hierarchy (exogenous constraints). It also
allows handling the linear constraints linking the variables endogenously (Di Fonzo
and Girolimetto, 2022). The function can calculate Combined Conditional Coherent (CCC)
forecasts as simple averages of Level-Conditional Coherent (LCC) and bottom-up
reconciled forecasts, with either endogenous or exogenous constraints.
}
\examples{
set.seed(123)
# Aggregation matrix for Z = X + Y, X = XX + XY and Y = YX + YY
A <- matrix(c(1,1,1,1,1,1,0,0,0,0,1,1), 3, byrow = TRUE)
# (7 x 7) base forecasts matrix (simulated), agg_order = 4
base <- rbind(rnorm(7, rep(c(40, 20, 10), c(1, 2, 4))),
              rnorm(7, rep(c(20, 10, 5), c(1, 2, 4))),
              rnorm(7, rep(c(20, 10, 5), c(1, 2, 4))),
              rnorm(7, rep(c(10, 5, 2.5), c(1, 2, 4))),
              rnorm(7, rep(c(10, 5, 2.5), c(1, 2, 4))),
              rnorm(7, rep(c(10, 5, 2.5), c(1, 2, 4))),
              rnorm(7, rep(c(10, 5, 2.5), c(1, 2, 4))))
# (7 x 70) in-sample residuals matrix (simulated)
res <- matrix(rnorm(70*7), nrow = 7)
# (4 x 4) Naive high frequency bottom base forecasts vector:
# all forecasts are set equal to 2.5
naive <- matrix(2.5, 4, 4)

## EXOGENOUS CONSTRAINTS (Hollyman et al., 2021)
# Level Conditional Coherent (LCC) reconciled forecasts
exo_LC <- ctlcc(base = base, agg_mat = A, agg_order = 4, comb = "wlsh", nn = "osqp",
                hfbts = naive, res = res, nodes = "auto", CCC = FALSE)

# Combined Conditional Coherent (CCC) reconciled forecasts
exo_CCC <- ctlcc(base = base, agg_mat = A, agg_order = 4, comb = "wlsh",
                hfbts = naive, res = res, nodes = "auto", CCC = TRUE)

# Results detailed by level:
info_exo <- recoinfo(exo_CCC, verbose = FALSE)
# info_exo$lcc

## ENDOGENOUS CONSTRAINTS (Di Fonzo and Girolimetto, 2024)
# Level Conditional Coherent (LCC) reconciled forecasts
endo_LC <- ctlcc(base = base, agg_mat = A, agg_order = 4, comb = "wlsh",
                 res = res, nodes = "auto", CCC = FALSE,
                 const = "endogenous")

# Combined Conditional Coherent (CCC) reconciled forecasts
endo_CCC <- ctlcc(base = base, agg_mat = A, agg_order = 4, comb = "wlsh",
                  res = res, nodes = "auto", CCC = TRUE,
                  const = "endogenous")

# Results detailed by level:
info_endo <- recoinfo(endo_CCC, verbose = FALSE)
# info_endo$lcc

}
\references{
Byron, R.P. (1978), The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 141, 3, 359-367.
\doi{10.2307/2344807}

Byron, R.P. (1979), Corrigenda: The estimation of large social account matrices,
\emph{Journal of the Royal Statistical Society, Series A}, 142(3), 405.
\doi{10.2307/2982515}

Di Fonzo, T. and Girolimetto, D. (2024), Forecast combination-based forecast reconciliation:
Insights and extensions, \emph{International Journal of Forecasting}, 40(2), 490–514.
\doi{10.1016/j.ijforecast.2022.07.001}

Di Fonzo, T. and Girolimetto, D. (2023b) Spatio-temporal reconciliation of solar forecasts.
\emph{Solar Energy} 251, 13–29. \doi{10.1016/j.solener.2023.01.003}

Hyndman, R.J., Ahmed, R.A., Athanasopoulos, G. and Shang, H.L. (2011),
Optimal combination forecasts for hierarchical time series,
\emph{Computational Statistics & Data Analysis}, 55, 9, 2579-2589.
\doi{10.1016/j.csda.2011.03.006}

Hollyman, R., Petropoulos, F. and Tipping, M.E. (2021), Understanding forecast reconciliation.
\emph{European Journal of Operational Research}, 294, 149–160. \doi{10.1016/j.ejor.2021.01.017}

Stellato, B., Banjac, G., Goulart, P., Bemporad, A. and Boyd, S. (2020), OSQP:
An Operator Splitting solver for Quadratic Programs,
\emph{Mathematical Programming Computation}, 12, 4, 637-672.
\doi{10.1007/s12532-020-00179-2}
}
\seealso{
Level conditional coherent reconciliation: 
\code{\link{cslcc}()},
\code{\link{telcc}()}

Cross-temporal framework: 
\code{\link{ctboot}()},
\code{\link{ctbu}()},
\code{\link{ctcov}()},
\code{\link{ctmo}()},
\code{\link{ctrec}()},
\code{\link{cttd}()},
\code{\link{cttools}()},
\code{\link{iterec}()},
\code{\link{tcsrec}()}
}
\concept{Framework: cross-temporal}
\concept{Reco: level conditional coherent}
