#' @name PPC_CoFM
#' @title Perform Projected PCA (PPC) Estimation for CoFM
#' @description
#' This function performs Projected Principal Component Analysis (PPC) on the input data to
#' estimate factor loadings and uniquenesses. It is designed to work with data generated by
#' the \code{\link{CoFM}} function and calculates error metrics (MSE and relative loss) by
#' comparing estimates against true parameters. The method projects data onto a subspace
#' (using a projection operator) before performing PCA.
#'
#' @param data A matrix or data frame of input data (n x p). Usually the \code{$data} output from \code{CoFM}.
#' @param m Integer. The number of principal components (factors) to retain.
#' @param A Matrix. The true factor loadings matrix (p x m). Usually \code{$True_Params$A} from \code{CoFM}.
#' @param D Matrix. The true uniquenesses matrix (p x p). Usually \code{$True_Params$D} from \code{CoFM}.
#'
#' @return A list containing:
#' \item{Ap2}{Estimated factor loadings matrix (Projected).}
#' \item{Dp2}{Estimated uniquenesses matrix (Projected).}
#' \item{MSESigmaA}{Mean Squared Error for factor loadings.}
#' \item{MSESigmaD}{Mean Squared Error for uniquenesses.}
#' \item{LSigmaA}{Relative loss metric for factor loadings.}
#' \item{LSigmaD}{Relative loss metric for uniquenesses.}
#'
#' @export
#'
#' @examples
#' # Examples should be fast and reproducible for CRAN checks
#' set.seed(123)
#'
#' # 1. Generate toy data using CoFM
#' sim_result <- CoFM(n = 200, p = 6, m = 2, type = "Clayton", param = 2.0)
#'
#' # 2. Extract true parameters and observed data
#' true_A <- sim_result$True_Params$A
#' true_D <- sim_result$True_Params$D
#' obs_data <- sim_result$data
#'
#' # 3. Apply PPC method and compute errors
#' ppc_result <- PPC_CoFM(data = obs_data, m = 2, A = true_A, D = true_D)
#'
#' # 4. Inspect results
#' ppc_result$MSESigmaA
#' ppc_result$MSESigmaD
#' head(ppc_result$Ap2)
PPC_CoFM <- function(data, m, A, D) {

  # 1. Input Validation
  if (!is.matrix(data) && !is.data.frame(data)) {
    stop("Data must be a matrix or data frame.")
  }
  X <- as.matrix(data)
  n <- nrow(X)
  p <- ncol(X)

  if (!is.numeric(m) || length(m) != 1L || is.na(m) || m <= 0 || m > p) {
    stop("m must be a positive integer and cannot exceed the number of variables (ncol(data)).")
  }
  m <- as.integer(m)

  if (!is.matrix(A) || nrow(A) != p || ncol(A) != m) {
    stop("A must be a matrix of dimension p x m, where p = ncol(data) and m is the number of factors.")
  }
  if (!is.matrix(D) || nrow(D) != p || ncol(D) != p) {
    stop("D must be a matrix of dimension p x p, where p = ncol(data).")
  }
  if (n < 2L || p < 2L) {
    stop("data must have at least 2 rows and 2 columns.")
  }

  # 2. Standardize Data
  X <- scale(X)

  # 3. Projection operator (avoid constructing n x n diagonal matrix)
  # Equivalent to P %*% X where P = diag(rep(c(0,1), length.out=n))
  w <- rep(c(0, 1), length.out = n)
  Xpro <- scale(X * w)

  # 4. Covariance of Projected Data
  Sigmahatpro <- stats::cov(Xpro)

  # 5. Eigen decomposition (symmetric = TRUE for covariance matrices)
  eig <- base::eigen(Sigmahatpro, symmetric = TRUE)
  ind <- order(eig$values, decreasing = TRUE)
  lambdahat <- eig$values[ind]
  Q <- eig$vectors[, ind, drop = FALSE]
  Qhat <- Q[, 1:m, drop = FALSE]

  # 6. Estimated loadings (Projected): Ap2 = Q_m * sqrt(lambda_m)
  lam_use <- pmax(lambdahat[1:m], 0)
  Apro <- Qhat %*% diag(sqrt(lam_use), nrow = m)

  # 7. Estimated uniquenesses (Projected)
  hpro <- diag(Apro %*% t(Apro))
  Dpro_vec <- diag(Sigmahatpro) - hpro
  Dpro_vec <- pmax(Dpro_vec, 0)         # clamp tiny negatives
  Dpro_mat <- diag(Dpro_vec, nrow = p)

  # 8. Error metrics
  MSESigmaA <- matrixcalc::frobenius.norm(Apro - A)^2 / (p^2)
  MSESigmaD <- matrixcalc::frobenius.norm(Dpro_mat - D)^2 / (p^2)

  denomA <- matrixcalc::frobenius.norm(A)^2
  denomD <- matrixcalc::frobenius.norm(D)^2

  LSigmaA <- if (denomA > 0) matrixcalc::frobenius.norm(Apro - A)^2 / denomA else NA_real_
  LSigmaD <- if (denomD > 0) matrixcalc::frobenius.norm(Dpro_mat - D)^2 / denomD else NA_real_

  list(
    Ap2 = Apro,
    Dp2 = Dpro_mat,
    MSESigmaA = MSESigmaA,
    MSESigmaD = MSESigmaD,
    LSigmaA = LSigmaA,
    LSigmaD = LSigmaD
  )
}
