#' plotSigExposure
#'
#' This function plots the signature exposure for all of the samples as a
#' stacked bar plot. There are a number of different options for how to sort 
#' the resulting plot. 
#' 
#' Adding data tracks to the plot:
#' One of the major features of this function is that it allows the user to add
#' in some additional data for the samples to be plotted as a track alongside 
#' the main signature exposure stacked bar plot. These additional data points 
#' can be passed in as a vector of corresponding values in the same order. If
#' you want to plot multiple tracks you can pass in a list of vectors using 
#' the trackData parameter.
#' 
#' Specifying how to sort the plot:
#' When you give the function a set of trackData, it allows you to begin to 
#' specify the sortOrder. This allows your to sort the main plot in a different 
#' order. "m" represents the main plot, and "t" followed by the number of the 
#' track (ie: "t1","t2" ...)  represents the tracks. By chaining the values 
#' together you can specify a variety of ways to sort the final plot. As an
#' example, the sortOrder of "mt1t2" specifies the the plot should be sorted by 
#' the signature exposures first followed by the first track and finally the 
#' second track. In another example, the sortOrder of "t2mt1" specifies the plot
#' to be sorted by track number 2 first followed by the signature exposures and 
#' lastly by track number 1.
#' 
#' Sorting method:
#' The two methods of sorting the signature exposure are either "hclust" or 
#' "group". The hclust uses the ward.D method to cluster the exposures and then
#' cuts the tree to split the data. The group method splits the samples into
#' groups based on which signatures they had the highest exposure to.
#'
#' @param sigExposure Signature exposure matrix to be plotted
#' @param saveRes Whether or not to save results. Default is FALSE.
#' @param saveDir Where to save plots, must be provided if using saveDir
#' @param runName Used to add a runName to the file output. Default is "".
#' @param trackData Data used to plot tracks
#' @param sort Whether or not to sort the plot 
#' @param sortOrder The order in which to sort the plot
#' @param method The method by which to sort the main plot
#' @param colors Colors used in plotting
#' @keywords sigExposure plot
#' @importFrom ggplot2 ggsave
#' @importFrom cowplot plot_grid
#' @importFrom cowplot get_legend
#' @export
#' @return Plots the signature exposure to allow visualization
#' @examples 
#' \donttest{plotSigExposure(sigExposExp)}
plotSigExposure = function(sigExposure,saveRes=FALSE,saveDir=NULL,runName = "", trackData = NULL, sort = FALSE, sortOrder = "m", method = NULL,colors = NULL){
  
  plotTitle = paste(runName,"Signature Exposure")
  
  # Check to make sure the provided sorting method is supported
  if (!is.null(method) && !(method %in% c("hclust","group"))){
    warning("Unrecognized sorting method. Using hclust. Check documentation for supported method/s.")
    method = NULL;
  }
  
  # Checks the format of the trackData
  if (!is.null(trackData) && !is.list(trackData)){
    trackData = list(trackData)
  } 
  else if (!is.null(trackData) && is.list(trackData) && length(trackData) > 3){
    warning("Max number of plot tracks is 3. Only using the first 3 elements in trackData input.")
    trackData = trackData[1:3]
  }
  
  sigExposure = as.matrix(sigExposure)
  
  # Figures out the order in which to sort the samples
  newOrder = NULL;
  if (sort) {
    # Splits sortOrder using regex
    sortOrder = unlist(strsplit(sortOrder,"(?<=[t].)|(?<=[m])",perl = T))
    newOrder = sortSection(sigExposure,trackData,sortOrder,method)
    sigExposure = sigExposure[,newOrder]
  }
  
  
  # Sorts the track Data
  if (!is.null(newOrder) && !is.null(trackData)){
    trackData = sortTrackData(trackData,newOrder)
  } 
  
  numSigs = nrow(sigExposure)
  
  # Checks the color input
  if (is.null(colors)){
    colors = colorspace::rainbow_hcl(numSigs)
  } else if (length(colors) < numSigs){
    warning(paste("Not enough colors given. nSigs =",numSigs,"nColor =",length(colors),
                  "\nSupplementing with default colors."))
    colors = c(colors,plotColors[1:(numSigs - length(colors))])
  }
  
  # Converts the data into a dataframe for plotting using ggplot
  data = data.frame()
  for(i in 1:numSigs)
  {
    sig = data.frame("var"=1:ncol(sigExposure),"value"=as.numeric(sigExposure[i,]),"comp"=i)
    data = rbind(data,sig)
  }
  data$comp = as.factor(data$comp)
  levels(data$comp) = paste0(rep("S",numSigs),1:numSigs)
  data$var = as.factor(data$var)
  
  # Creates the main plot
  mainPlot = ggplot(data,aes_string(x = "var", y = "value",fill = "comp",width = 1)) + 
    geom_bar(position = "fill",stat = "identity") +
    scale_fill_manual(values = colors) +
    theme_grey() +
    theme(axis.text.x = element_blank(),axis.ticks.x = element_blank()) + 
    scale_y_continuous(expand = c(0,0)) +
    xlab('Samples')+
    ylab('Exposure')+
    theme(axis.title.x = element_text(vjust=-1))

  # If there is any track data, add it on depending on how many tracks are given
  if (is.null(trackData)){
    finalPlot = mainPlot
  } 
  else if (length(trackData) == 1) {
    trackPlotData = data.frame("sample" = 1:length(trackData[[1]]),"track" = trackData[[1]])
    trackPlot = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
                geom_bar(position = "fill", stat="identity")+
                theme_void() + 
                scale_x_continuous(expand = c(0,0)) 
                
    legendPlot = plot_grid(get_legend(trackPlot),get_legend(mainPlot), ncol = 1)
    
    mainPlot = mainPlot + theme(legend.position = "none")
    trackPlot = trackPlot + theme(legend.position = "none")
    
    fullPlot <- plot_grid(trackPlot, mainPlot, align = "v", ncol = 1, axis = "lr", rel_heights = c(1, 10))
    finalPlot = plot_grid(fullPlot,legendPlot,nrow=1, rel_widths = c(10,1.5))
  }
  else if (length(trackData) == 2) {
    trackPlotData = data.frame("sample" = 1:length(trackData[[1]]),"track" = trackData[[1]])
    trackPlot1 = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
      geom_bar(position = "fill", stat="identity")+
      theme_void() + 
      scale_x_continuous(expand = c(0,0)) 
    
    trackPlotData = data.frame("sample" = 1:length(trackData[[2]]),"track" = trackData[[2]])
    trackPlot2 = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
      geom_bar(position = "fill", stat="identity")+
      theme_void() + 
      scale_x_continuous(expand = c(0,0)) 
    
    legendPlot = plot_grid(get_legend(trackPlot1),get_legend(trackPlot2),
                           get_legend(mainPlot), ncol = 1)
    
    mainPlot = mainPlot + theme(legend.position = "none")
    trackPlot1 = trackPlot1 + theme(legend.position = "none")
    trackPlot2 = trackPlot2 + theme(legend.position = "none")
    
    fullPlot <- plot_grid(trackPlot1,trackPlot2, mainPlot, align = "v", ncol = 1, axis = "lr", rel_heights = c(1, 1, 9))
    finalPlot = plot_grid(fullPlot,legendPlot,nrow=1, rel_widths = c(10,1.5))
  }
  else if (length(trackData) == 3) {
    trackPlotData = data.frame("sample" = 1:length(trackData[[1]]),"track" = trackData[[1]])
    trackPlot1 = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
      geom_bar(position = "fill", stat="identity")+
      theme_void() + 
      scale_x_continuous(expand = c(0,0)) 
    
    trackPlotData = data.frame("sample" = 1:length(trackData[[2]]),"track" = trackData[[2]])
    trackPlot2 = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
      geom_bar(position = "fill", stat="identity")+
      theme_void() + 
      scale_x_continuous(expand = c(0,0)) 
    
    trackPlotData = data.frame("sample" = 1:length(trackData[[3]]),"track" = trackData[[3]])
    trackPlot3 = ggplot(trackPlotData, aes_string(x = "sample",y=.5,fill = "track",width = 1)) +
      geom_bar(position = "fill", stat="identity")+
      theme_void() + 
      scale_x_continuous(expand = c(0,0)) 
    
    legendPlot = plot_grid(get_legend(trackPlot1),get_legend(trackPlot2),
                           get_legend(trackPlot3),get_legend(mainPlot), ncol = 1)
    
    mainPlot = mainPlot + theme(legend.position = "none")
    trackPlot1 = trackPlot1 + theme(legend.position = "none")
    trackPlot2 = trackPlot2 + theme(legend.position = "none")
    trackPlot3 = trackPlot3 + theme(legend.position = "none")
    
    fullPlot <- plot_grid(trackPlot1,trackPlot2,trackPlot3, mainPlot, align = "v", ncol = 1, axis = "lr", 
                          rel_heights = c(1, 1, 1, 8))
    finalPlot = plot_grid(fullPlot,legendPlot,nrow=1, rel_widths = c(10,1.5))
  }
  
  
  
  
  
  if (runName != "")
    runName = paste0(runName,"_")
  
  if(saveRes){
    if (is.null(saveDir)){
	stop('No saveDir provided for saveRes. Please set saveDir.')
    }
    ggplot2::ggsave(paste0(saveDir,plotTitle,".tiff"),plot,width = 4,
           height = 3, units = "in", dpi = 300)
  }
  
  return(finalPlot)
}


# This is a recursive subfunction that is used in order to sort the data. It 
# grabs the first piece of the sortOrder and sorts the data based off that
# instruction. Then it calls sort again based on the subsections found  and
# the remaining sortOrder
sortSection = function(sigExp,trackData,sortOrder,method){
  # Recursive checks
  if (is.null(ncol(sigExp)))
    return(1)
  
  if (length(sortOrder)==0)
    return(1:nrow(sigExp))
  
  nextSort = sortOrder[1]
  sortOrder = sortOrder[-1]
  
  #Sorts the data
  if (startsWith(nextSort,"t")){
    trackNum = as.numeric(substr(nextSort,2,2))
    
    if (trackNum > length(trackData)){
      stop("Specified track number in sort order larger than length of given trackData.")
    }
    
    groups = trackData[[trackNum]]
    newOrder = order(groups)
    groups = groups[newOrder]
  }
  else if (nextSort == "m"){
    if (is.null(method) || method == "hclust") {
      dist = dist(t(sigExp))
      clust = hclust(dist,method="ward.D")
      groups = cutree(clust,k=min(c(5,ncol(sigExp))))
      newOrder = clust$order
      groups = groups[newOrder]
    }
    else if (method == "group") {
      groups = numeric(ncol(sigExp))
      for (i in 1:nrow(sigExp)){
        groups[sigExp[i,] == apply(sigExp,2,max)] = i
      }
      newOrder = order(groups)
      groups = groups[newOrder]
    }
  }
  else {
    stop("Invalid Sort Order. Check function documentation.")
  }

  # Subsets the data based on the groups found and calls sortSection again
  finalOrder = NULL
  for (i in unique(groups)){
    sub = newOrder[groups==i] 
    trackSub = subsetTrackData(trackData,sub)
    subOrder = sortSection(sigExp[,sub],trackSub,sortOrder,method)
    finalOrder = c(finalOrder, sub[subOrder])
  }

  
  return(finalOrder);
}

#Method to subset each set of track data returning a list containing the subset 
# track data
subsetTrackData = function(trackData,indices){
  for (i in 1:length(trackData)){
    sub = trackData[[i]]
    sub = sub[indices]
    trackData[[i]] = sub
  }
  return(trackData)
}

# Method used to go through the list of trackData and apply the sorted order
sortTrackData = function(trackData,order){
  for (i in 1:length(trackData)){
    sub = trackData[[i]]
    sub = sub[order]
    trackData[[i]] = sub
  }
  return(trackData)
}
