% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitted_BKP.R, R/fitted_DKP.R
\name{fitted}
\alias{fitted}
\alias{fitted.BKP}
\alias{fitted.DKP}
\title{Extract BKP or DKP Model Fitted Values}
\usage{
\method{fitted}{BKP}(object, ...)

\method{fitted}{DKP}(object, ...)
}
\arguments{
\item{object}{An object of class \code{BKP} or \code{DKP}, typically the
result of a call to \code{\link{fit_BKP}} or \code{\link{fit_DKP}}.}

\item{...}{Additional arguments (currently unused).}
}
\value{
A numeric vector (for \code{BKP}) or a numeric matrix (for
\code{DKP}) containing posterior mean estimates at the training inputs.
}
\description{
Compute the posterior fitted values from a fitted \code{BKP} or
\code{DKP} object. For a \code{BKP} object, this returns the posterior mean
probability of the positive class. For a \code{DKP} object, this returns
the posterior mean probabilities for each class.
}
\details{
For a \code{BKP} model, the fitted values correspond to the
posterior mean probability of the positive class, computed from the Beta
Kernel Process. For a \code{DKP} model, the fitted values correspond to the
posterior mean probabilities for each class, derived from the posterior
Dirichlet distribution of the class probabilities.
}
\examples{
# -------------------------- BKP ---------------------------
set.seed(123)

# Define true success probability function
true_pi_fun <- function(x) {
  (1 + exp(-x^2) * cos(10 * (1 - exp(-x)) / (1 + exp(-x)))) / 2
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(100, n, replace = TRUE)
y <- rbinom(n, size = m, prob = true_pi)

# Fit BKP model
model <- fit_BKP(X, y, m, Xbounds = Xbounds)

# Extract fitted values
fitted(model)

# -------------------------- DKP ---------------------------
#' set.seed(123)

# Define true class probability function (3-class)
true_pi_fun <- function(X) {
  p1 <- 1/(1+exp(-3*X))
  p2 <- (1 + exp(-X^2) * cos(10 * (1 - exp(-X)) / (1 + exp(-X)))) / 2
  return(matrix(c(p1/2, p2/2, 1 - (p1+p2)/2), nrow = length(p1)))
}

n <- 30
Xbounds <- matrix(c(-2, 2), nrow = 1)
X <- tgp::lhs(n = n, rect = Xbounds)
true_pi <- true_pi_fun(X)
m <- sample(150, n, replace = TRUE)

# Generate multinomial responses
Y <- t(sapply(1:n, function(i) rmultinom(1, size = m[i], prob = true_pi[i, ])))

# Fit DKP model
model <- fit_DKP(X, Y, Xbounds = Xbounds)

# Extract fitted values
fitted(model)

}
\references{
Zhao J, Qing K, Xu J (2025). \emph{BKP: An R Package for Beta
Kernel Process Modeling}.  arXiv.
https://doi.org/10.48550/arXiv.2508.10447.
}
\keyword{BKP}
\keyword{DKP}
