#' Quick computation of Gaussian Process Classification (GPC) covariances
#'
#' Computes conditional covariance matrix between some new points and many integration points.
#'
#' @param object an object of class \code{\link[GPCsign]{gpcm}}.
#' @param integration.points \code{p*d} matrix of fixed integration points in the \code{X} space.
#' @param X.new \code{q*d} matrix of new points.
#' @param precalc.data list containing precalculated data. This list can be generated using the \code{precomputeUpdateData} function. 
#' @param c.newdata the (unconditional) covariance between \code{X.new} and the design points.
#'
#' @return Conditional covariance matrix between \code{integration.points} and \code{X.new}.
#'
#' @importMethodsFrom DiceKriging covMat1Mat2
#' @export
#'
#' @references Menz, M., Munoz-Zuniga, M., Sinoquet, D. Estimation of simulation failure set with active learning based on Gaussian Process classifiers and random set theory (2023). \url{https://hal.science/hal-03848238}.
#' @references Bachoc, F., Helbert, C. & Picheny, V. Gaussian process optimization with failures: classification and convergence proof. \emph{J Glob Optim} \bold{78}, 483–506 (2020). \doi{10.1007/s10898-020-00920-0}.
#'
#' @author Morgane MENZ, Delphine SINOQUET, Miguel MUNOZ-ZUNIGA. Contributors: Naoual SERRAJI.
#'
#' @examples
#' #-------------------------------------------------------------------
#' #------------------- computeQuickgpccov ----------------------------
#' #-------------------------------------------------------------------
#'
#' ## 20-points DoE, and the corresponding response
#' d <- 2
#' nb_PX <- 20
#' x <- matrix(c(0.205293785978832, 0.0159983370750337,
#'               0.684774733109666, 0.125251417595962,
#'               0.787208786290006, 0.700475706055049,
#'               0.480507717105934, 0.359730889653793,
#'               0.543665267336735, 0.565974761807069,
#'               0.303412043992361, 0.471502352650857,
#'               0.839505250127309, 0.504914690245002,
#'               0.573294917143728, 0.784444726564573,
#'               0.291681289223421, 0.255053812451938,
#'               0.87233450888786, 0.947168337730927,
#'               0.648262257638515, 0.973264712407035,
#'               0.421877310273815, 0.0686662506387988,
#'               0.190976166753807, 0.810964668176754,
#'               0.918527262507395, 0.161973686467513,
#'               0.0188128700859558, 0.43522031347403,
#'               0.99902788789426, 0.655561821513544,
#'               0.741113863862512, 0.321050086076934,
#'               0.112003007565305, 0.616551317575545,
#'               0.383511473487687, 0.886611679106771,
#'               0.0749211435982952, 0.205805968972305),
#'             byrow = TRUE, ncol = d)
#' require(DiceKriging)
#' fx <- apply(x, 1, branin)
#' f <- ifelse(fx < 14, -1, 1)
#' Xf <- as.matrix(x)
#'
#' require(future) # load future package for parallelization while building gpcm model 
#' plan(multisession) # activate parallel calculations (with available cores automatic detection)
#'
#' ## gpcm object
#' require(GPCsign)
#' model <- gpcm(f, Xf, coef.m = -1.25, coef.cov = c(1.17,0.89))
#'
#' ## prediction at X.new
#' X.new <-  matrix(c(0.1,0.2),ncol=2,byrow=TRUE)
#' gpc <- predict(object = model, newdata = X.new)
#' c.newdata <- gpc$c
#'
#' ## prediction at integration points
#' require(randtoolbox)
#' nb.integration <- d * 100
#' integration.points <- sobol(n = nb.integration, dim = d, scrambling = 0)
#' gpc_int <- predict(object = model, newdata = integration.points)
#'
#' precalc.data <- list(c.K = crossprod(gpc_int$c , model@invK))
#' c.newdata <- gpc$c
#'
#' integration.points <- scale(x = integration.points, center = model@X.mean, scale = model@X.std)
#' kn <-  computeQuickgpccov(object = model, integration.points = integration.points,
#'                           X.new = X.new, precalc.data = precalc.data,
#'                           c.newdata = c.newdata )
#' plan(sequential) # deactivate parallel calculations: back to sequential mode 
computeQuickgpccov <- function(object, integration.points, X.new,
                               precalc.data, c.newdata){

  colnames(integration.points) <- colnames(object@X)

	c.xnew.integpoints <- DiceKriging::covMat1Mat2(X1=integration.points,X2=X.new, object=object@covariance, nugget.flag=object@covariance@nugget.flag)

	kn <- c.xnew.integpoints - precalc.data$c.K %*% c.newdata

	return(kn)
}

