







use alloc::slice;
use alloc::vec;
#[cfg(not(feature = "std"))]
use alloc::vec::Vec;
#[allow(unused_imports)]
use rawpointer::PointerExt;
use std::mem::{size_of, ManuallyDrop};

use crate::imp_prelude::*;

use crate::argument_traits::AssignElem;
use crate::dimension;
use crate::dimension::broadcast::co_broadcast;
use crate::dimension::reshape_dim;
use crate::dimension::IntoDimension;
use crate::dimension::{
    abs_index,
    axes_of,
    do_slice,
    merge_axes,
    move_min_stride_axis_to_last,
    offset_from_low_addr_ptr_to_logical_ptr,
    size_of_shape_checked,
    stride_offset,
    Axes,
};
use crate::error::{self, from_kind, ErrorKind, ShapeError};
use crate::itertools::zip;
use crate::math_cell::MathCell;
use crate::order::Order;
use crate::shape_builder::ShapeArg;
use crate::zip::{IntoNdProducer, Zip};
use crate::AxisDescription;
use crate::{arraytraits, DimMax};

use crate::iter::{
    AxisChunksIter,
    AxisChunksIterMut,
    AxisIter,
    AxisIterMut,
    AxisWindows,
    ExactChunks,
    ExactChunksMut,
    IndexedIter,
    IndexedIterMut,
    Iter,
    IterMut,
    Lanes,
    LanesMut,
    Windows,
};
use crate::slice::{MultiSliceArg, SliceArg};
use crate::stacking::concatenate;
use crate::{NdIndex, Slice, SliceInfoElem};

/// # Methods For All Array Types
impl<A, S, D> ArrayBase<S, D>
where
    S: RawData<Elem = A>,
    D: Dimension,
{
    
    pub fn len(&self) -> usize
    {
        self.dim.size()
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn len_of(&self, axis: Axis) -> usize
    {
        self.dim[axis.index()]
    }

    
    pub fn is_empty(&self) -> bool
    {
        self.len() == 0
    }

    
    pub fn ndim(&self) -> usize
    {
        self.dim.ndim()
    }

    
    
    
    pub fn dim(&self) -> D::Pattern
    {
        self.dim.clone().into_pattern()
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn raw_dim(&self) -> D
    {
        self.dim.clone()
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn shape(&self) -> &[usize]
    {
        self.dim.slice()
    }

    
    pub fn strides(&self) -> &[isize]
    {
        let s = self.strides.slice();
        
        unsafe { slice::from_raw_parts(s.as_ptr() as *const _, s.len()) }
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn stride_of(&self, axis: Axis) -> isize
    {
        
        self.strides[axis.index()] as isize
    }

    
    pub fn view(&self) -> ArrayView<'_, A, D>
    where S: Data
    {
        debug_assert!(self.pointer_is_inbounds());
        unsafe { ArrayView::new(self.ptr, self.dim.clone(), self.strides.clone()) }
    }

    
    pub fn view_mut(&mut self) -> ArrayViewMut<'_, A, D>
    where S: DataMut
    {
        self.ensure_unique();
        unsafe { ArrayViewMut::new(self.ptr, self.dim.clone(), self.strides.clone()) }
    }

    
    
    
    
    
    
    
    pub fn cell_view(&mut self) -> ArrayView<'_, MathCell<A>, D>
    where S: DataMut
    {
        self.view_mut().into_cell_view()
    }

    
    
    
    
    
    
    
    
    
    
    /// # use ndarray::prelude::*;
    /// # let arr = Array::from_shape_vec((2, 2).f(), vec![1, 2, 3, 4]).unwrap();
    /// # let owned = {
    
    /// # };
    /// # assert!(owned.is_standard_layout());
    /// # assert_eq!(arr, owned);
    
    
    
    
    
    /// # use ndarray::prelude::*;
    /// # let arr = Array::from_shape_vec((2, 2), vec![1, 2, 3, 4]).unwrap();
    /// # let owned = {
    
    /// # };
    /// # assert!(owned.t().is_standard_layout());
    /// # assert_eq!(arr, owned);
    
    pub fn to_owned(&self) -> Array<A, D>
    where
        A: Clone,
        S: Data,
    {
        if let Some(slc) = self.as_slice_memory_order() {
            unsafe { Array::from_shape_vec_unchecked(self.dim.clone().strides(self.strides.clone()), slc.to_vec()) }
        } else {
            self.map(A::clone)
        }
    }

    
    
    pub fn to_shared(&self) -> ArcArray<A, D>
    where
        A: Clone,
        S: Data,
    {
        S::to_shared(self)
    }

    
    
    pub fn into_owned(self) -> Array<A, D>
    where
        A: Clone,
        S: Data,
    {
        S::into_owned(self)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn try_into_owned_nocopy(self) -> Result<Array<A, D>, Self>
    where S: Data
    {
        S::try_into_owned_nocopy(self)
    }

    
    
    
    
    
    
    pub fn into_shared(self) -> ArcArray<A, D>
    where
        A: Clone,
        S: DataOwned,
    {
        S::into_shared(self)
    }

    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    pub fn first(&self) -> Option<&A>
    where S: Data
    {
        if self.is_empty() {
            None
        } else {
            Some(unsafe { &*self.as_ptr() })
        }
    }

    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    pub fn first_mut(&mut self) -> Option<&mut A>
    where S: DataMut
    {
        if self.is_empty() {
            None
        } else {
            Some(unsafe { &mut *self.as_mut_ptr() })
        }
    }

    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    pub fn last(&self) -> Option<&A>
    where S: Data
    {
        if self.is_empty() {
            None
        } else {
            let mut index = self.raw_dim();
            for ax in 0..index.ndim() {
                index[ax] -= 1;
            }
            Some(unsafe { self.uget(index) })
        }
    }

    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    pub fn last_mut(&mut self) -> Option<&mut A>
    where S: DataMut
    {
        if self.is_empty() {
            None
        } else {
            let mut index = self.raw_dim();
            for ax in 0..index.ndim() {
                index[ax] -= 1;
            }
            Some(unsafe { self.uget_mut(index) })
        }
    }

    
    
    
    
    
    
    pub fn iter(&self) -> Iter<'_, A, D>
    where S: Data
    {
        debug_assert!(self.pointer_is_inbounds());
        self.view().into_iter_()
    }

    
    
    
    
    
    
    pub fn iter_mut(&mut self) -> IterMut<'_, A, D>
    where S: DataMut
    {
        self.view_mut().into_iter_()
    }

    
    
    
    
    
    
    
    
    pub fn indexed_iter(&self) -> IndexedIter<'_, A, D>
    where S: Data
    {
        IndexedIter::new(self.view().into_elements_base())
    }

    
    
    
    
    
    
    pub fn indexed_iter_mut(&mut self) -> IndexedIterMut<'_, A, D>
    where S: DataMut
    {
        IndexedIterMut::new(self.view_mut().into_elements_base())
    }

    
    
    /// See [*Slicing*](#slicing) for full documentation.
    
    
    
    
    #[track_caller]
    pub fn slice<I>(&self, info: I) -> ArrayView<'_, A, I::OutDim>
    where
        I: SliceArg<D>,
        S: Data,
    {
        self.view().slice_move(info)
    }

    
    
    /// See [*Slicing*](#slicing) for full documentation.
    
    
    
    
    #[track_caller]
    pub fn slice_mut<I>(&mut self, info: I) -> ArrayViewMut<'_, A, I::OutDim>
    where
        I: SliceArg<D>,
        S: DataMut,
    {
        self.view_mut().slice_move(info)
    }

    
    
    /// See [*Slicing*](#slicing) for full documentation. See also
    
    
    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn multi_slice_mut<'a, M>(&'a mut self, info: M) -> M::Output
    where
        M: MultiSliceArg<'a, A, D>,
        S: DataMut,
    {
        info.multi_slice_move(self.view_mut())
    }

    
    
    /// See [*Slicing*](#slicing) for full documentation.
    
    
    
    
    #[track_caller]
    pub fn slice_move<I>(mut self, info: I) -> ArrayBase<S, I::OutDim>
    where I: SliceArg<D>
    {
        assert_eq!(
            info.in_ndim(),
            self.ndim(),
            "The input dimension of `info` must match the array to be sliced.",
        );
        let out_ndim = info.out_ndim();
        let mut new_dim = I::OutDim::zeros(out_ndim);
        let mut new_strides = I::OutDim::zeros(out_ndim);

        let mut old_axis = 0;
        let mut new_axis = 0;
        info.as_ref().iter().for_each(|&ax_info| match ax_info {
            SliceInfoElem::Slice { start, end, step } => {
                
                self.slice_axis_inplace(Axis(old_axis), Slice { start, end, step });
                
                new_dim[new_axis] = self.dim[old_axis];
                new_strides[new_axis] = self.strides[old_axis];
                old_axis += 1;
                new_axis += 1;
            }
            SliceInfoElem::Index(index) => {
                
                let i_usize = abs_index(self.len_of(Axis(old_axis)), index);
                self.collapse_axis(Axis(old_axis), i_usize);
                
                
                
                
                old_axis += 1;
            }
            SliceInfoElem::NewAxis => {
                
                new_dim[new_axis] = 1;
                new_strides[new_axis] = 0;
                new_axis += 1;
            }
        });
        debug_assert_eq!(old_axis, self.ndim());
        debug_assert_eq!(new_axis, out_ndim);

        
        unsafe { self.with_strides_dim(new_strides, new_dim) }
    }

    
    
    
    
    
    
    
    
    
    
    /// See [*Slicing*](#slicing) for full documentation.
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn slice_collapse<I>(&mut self, info: I)
    where I: SliceArg<D>
    {
        assert_eq!(
            info.in_ndim(),
            self.ndim(),
            "The input dimension of `info` must match the array to be sliced.",
        );
        let mut axis = 0;
        info.as_ref().iter().for_each(|&ax_info| match ax_info {
            SliceInfoElem::Slice { start, end, step } => {
                self.slice_axis_inplace(Axis(axis), Slice { start, end, step });
                axis += 1;
            }
            SliceInfoElem::Index(index) => {
                let i_usize = abs_index(self.len_of(Axis(axis)), index);
                self.collapse_axis(Axis(axis), i_usize);
                axis += 1;
            }
            SliceInfoElem::NewAxis => panic!("`slice_collapse` does not support `NewAxis`."),
        });
        debug_assert_eq!(axis, self.ndim());
    }

    
    
    
    
    #[track_caller]
    #[must_use = "slice_axis returns an array view with the sliced result"]
    pub fn slice_axis(&self, axis: Axis, indices: Slice) -> ArrayView<'_, A, D>
    where S: Data
    {
        let mut view = self.view();
        view.slice_axis_inplace(axis, indices);
        view
    }

    
    
    
    
    #[track_caller]
    #[must_use = "slice_axis_mut returns an array view with the sliced result"]
    pub fn slice_axis_mut(&mut self, axis: Axis, indices: Slice) -> ArrayViewMut<'_, A, D>
    where S: DataMut
    {
        let mut view_mut = self.view_mut();
        view_mut.slice_axis_inplace(axis, indices);
        view_mut
    }

    
    
    
    
    #[track_caller]
    pub fn slice_axis_inplace(&mut self, axis: Axis, indices: Slice)
    {
        let offset =
            do_slice(&mut self.dim.slice_mut()[axis.index()], &mut self.strides.slice_mut()[axis.index()], indices);
        unsafe {
            self.ptr = self.ptr.offset(offset);
        }
        debug_assert!(self.pointer_is_inbounds());
    }

    
    
    
    
    #[must_use = "slice_axis_move returns an array with the sliced result"]
    pub fn slice_axis_move(mut self, axis: Axis, indices: Slice) -> Self
    {
        self.slice_axis_inplace(axis, indices);
        self
    }

    
    
    
    
    
    
    
    #[track_caller]
    pub fn slice_each_axis<F>(&self, f: F) -> ArrayView<'_, A, D>
    where
        F: FnMut(AxisDescription) -> Slice,
        S: Data,
    {
        let mut view = self.view();
        view.slice_each_axis_inplace(f);
        view
    }

    
    
    
    
    
    
    
    #[track_caller]
    pub fn slice_each_axis_mut<F>(&mut self, f: F) -> ArrayViewMut<'_, A, D>
    where
        F: FnMut(AxisDescription) -> Slice,
        S: DataMut,
    {
        let mut view = self.view_mut();
        view.slice_each_axis_inplace(f);
        view
    }

    
    
    
    
    
    
    
    #[track_caller]
    pub fn slice_each_axis_inplace<F>(&mut self, mut f: F)
    where F: FnMut(AxisDescription) -> Slice
    {
        for ax in 0..self.ndim() {
            self.slice_axis_inplace(
                Axis(ax),
                f(AxisDescription {
                    axis: Axis(ax),
                    len: self.dim[ax],
                    stride: self.strides[ax] as isize,
                }),
            )
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn get<I>(&self, index: I) -> Option<&A>
    where
        S: Data,
        I: NdIndex<D>,
    {
        unsafe { self.get_ptr(index).map(|ptr| &*ptr) }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn get_ptr<I>(&self, index: I) -> Option<*const A>
    where I: NdIndex<D>
    {
        let ptr = self.ptr;
        index
            .index_checked(&self.dim, &self.strides)
            .map(move |offset| unsafe { ptr.as_ptr().offset(offset) as *const _ })
    }

    
    
    pub fn get_mut<I>(&mut self, index: I) -> Option<&mut A>
    where
        S: DataMut,
        I: NdIndex<D>,
    {
        unsafe { self.get_mut_ptr(index).map(|ptr| &mut *ptr) }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn get_mut_ptr<I>(&mut self, index: I) -> Option<*mut A>
    where
        S: RawDataMut,
        I: NdIndex<D>,
    {
        
        
        let ptr = self.as_mut_ptr();
        index
            .index_checked(&self.dim, &self.strides)
            .map(move |offset| unsafe { ptr.offset(offset) })
    }

    
    
    
    
    
    
    /// # Safety
    
    
    #[inline]
    pub unsafe fn uget<I>(&self, index: I) -> &A
    where
        S: Data,
        I: NdIndex<D>,
    {
        arraytraits::debug_bounds_check(self, &index);
        let off = index.index_unchecked(&self.strides);
        &*self.ptr.as_ptr().offset(off)
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    
    
    
    #[inline]
    pub unsafe fn uget_mut<I>(&mut self, index: I) -> &mut A
    where
        S: DataMut,
        I: NdIndex<D>,
    {
        debug_assert!(self.data.is_unique());
        arraytraits::debug_bounds_check(self, &index);
        let off = index.index_unchecked(&self.strides);
        &mut *self.ptr.as_ptr().offset(off)
    }

    
    
    
    
    
    #[track_caller]
    pub fn swap<I>(&mut self, index1: I, index2: I)
    where
        S: DataMut,
        I: NdIndex<D>,
    {
        let ptr = self.as_mut_ptr();
        let offset1 = index1.index_checked(&self.dim, &self.strides);
        let offset2 = index2.index_checked(&self.dim, &self.strides);
        if let Some(offset1) = offset1 {
            if let Some(offset2) = offset2 {
                unsafe {
                    std::ptr::swap(ptr.offset(offset1), ptr.offset(offset2));
                }
                return;
            }
        }
        panic!("swap: index out of bounds for indices {:?} {:?}", index1, index2);
    }

    
    
    
    
    
    
    /// # Safety
    
    
    
    
    
    
    
    pub unsafe fn uswap<I>(&mut self, index1: I, index2: I)
    where
        S: DataMut,
        I: NdIndex<D>,
    {
        debug_assert!(self.data.is_unique());
        arraytraits::debug_bounds_check(self, &index1);
        arraytraits::debug_bounds_check(self, &index2);
        let off1 = index1.index_unchecked(&self.strides);
        let off2 = index2.index_unchecked(&self.strides);
        std::ptr::swap(self.ptr.as_ptr().offset(off1), self.ptr.as_ptr().offset(off2));
    }

    
    
    fn get_0d(&self) -> &A
    where S: Data
    {
        assert!(self.ndim() == 0);
        unsafe { &*self.as_ptr() }
    }

    
    
    
    /// See [*Subviews*](#subviews) for full documentation.
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn index_axis(&self, axis: Axis, index: usize) -> ArrayView<'_, A, D::Smaller>
    where
        S: Data,
        D: RemoveAxis,
    {
        self.view().index_axis_move(axis, index)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn index_axis_mut(&mut self, axis: Axis, index: usize) -> ArrayViewMut<'_, A, D::Smaller>
    where
        S: DataMut,
        D: RemoveAxis,
    {
        self.view_mut().index_axis_move(axis, index)
    }

    
    
    /// See [`.index_axis()`](Self::index_axis) and [*Subviews*](#subviews) for full documentation.
    
    
    #[track_caller]
    pub fn index_axis_move(mut self, axis: Axis, index: usize) -> ArrayBase<S, D::Smaller>
    where D: RemoveAxis
    {
        self.collapse_axis(axis, index);
        let dim = self.dim.remove_axis(axis);
        let strides = self.strides.remove_axis(axis);
        
        unsafe { self.with_strides_dim(strides, dim) }
    }

    
    
    
    #[track_caller]
    pub fn collapse_axis(&mut self, axis: Axis, index: usize)
    {
        let offset = dimension::do_collapse_axis(&mut self.dim, &self.strides, axis.index(), index);
        self.ptr = unsafe { self.ptr.offset(offset) };
        debug_assert!(self.pointer_is_inbounds());
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn select(&self, axis: Axis, indices: &[Ix]) -> Array<A, D>
    where
        A: Clone,
        S: Data,
        D: RemoveAxis,
    {
        if self.ndim() == 1 {
            
            let axis_len = self.len_of(axis);
            
            if let Some(max_index) = indices.iter().cloned().max() {
                if max_index >= axis_len {
                    panic!("ndarray: index {} is out of bounds in array of len {}",
                           max_index, self.len_of(axis));
                }
            } 
            let view = self.view().into_dimensionality::<Ix1>().unwrap();
            Array::from_iter(indices.iter().map(move |&index| {
                
                unsafe { view.uget(index).clone() }
            }))
            .into_dimensionality::<D>()
            .unwrap()
        } else {
            let mut subs = vec![self.view(); indices.len()];
            for (&i, sub) in zip(indices, &mut subs[..]) {
                sub.collapse_axis(axis, i);
            }
            if subs.is_empty() {
                let mut dim = self.raw_dim();
                dim.set_axis(axis, 0);
                unsafe { Array::from_shape_vec_unchecked(dim, vec![]) }
            } else {
                concatenate(axis, &subs).unwrap()
            }
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn rows(&self) -> Lanes<'_, A, D::Smaller>
    where S: Data
    {
        let mut n = self.ndim();
        if n == 0 {
            n += 1;
        }
        Lanes::new(self.view(), Axis(n - 1))
    }

    
    
    
    
    pub fn rows_mut(&mut self) -> LanesMut<'_, A, D::Smaller>
    where S: DataMut
    {
        let mut n = self.ndim();
        if n == 0 {
            n += 1;
        }
        LanesMut::new(self.view_mut(), Axis(n - 1))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn columns(&self) -> Lanes<'_, A, D::Smaller>
    where S: Data
    {
        Lanes::new(self.view(), Axis(0))
    }

    
    
    
    
    pub fn columns_mut(&mut self) -> LanesMut<'_, A, D::Smaller>
    where S: DataMut
    {
        LanesMut::new(self.view_mut(), Axis(0))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn lanes(&self, axis: Axis) -> Lanes<'_, A, D::Smaller>
    where S: Data
    {
        Lanes::new(self.view(), axis)
    }

    
    
    
    
    pub fn lanes_mut(&mut self, axis: Axis) -> LanesMut<'_, A, D::Smaller>
    where S: DataMut
    {
        LanesMut::new(self.view_mut(), axis)
    }

    
    
    
    
    
    
    #[allow(deprecated)]
    pub fn outer_iter(&self) -> AxisIter<'_, A, D::Smaller>
    where
        S: Data,
        D: RemoveAxis,
    {
        self.view().into_outer_iter()
    }

    
    
    
    
    
    
    #[allow(deprecated)]
    pub fn outer_iter_mut(&mut self) -> AxisIterMut<'_, A, D::Smaller>
    where
        S: DataMut,
        D: RemoveAxis,
    {
        self.view_mut().into_outer_iter()
    }

    
    
    
    
    
    
    
    
    
    
    /// See [*Subviews*](#subviews) for full documentation.
    
    
    
    /// <img src="https://rust-ndarray.github.io/ndarray/images/axis_iter_3_4_5.svg" height="250px">
    #[track_caller]
    pub fn axis_iter(&self, axis: Axis) -> AxisIter<'_, A, D::Smaller>
    where
        S: Data,
        D: RemoveAxis,
    {
        AxisIter::new(self.view(), axis)
    }

    
    
    
    
    
    
    
    #[track_caller]
    pub fn axis_iter_mut(&mut self, axis: Axis) -> AxisIterMut<'_, A, D::Smaller>
    where
        S: DataMut,
        D: RemoveAxis,
    {
        AxisIterMut::new(self.view_mut(), axis)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn axis_chunks_iter(&self, axis: Axis, size: usize) -> AxisChunksIter<'_, A, D>
    where S: Data
    {
        AxisChunksIter::new(self.view(), axis, size)
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn axis_chunks_iter_mut(&mut self, axis: Axis, size: usize) -> AxisChunksIterMut<'_, A, D>
    where S: DataMut
    {
        AxisChunksIterMut::new(self.view_mut(), axis, size)
    }

    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn exact_chunks<E>(&self, chunk_size: E) -> ExactChunks<'_, A, D>
    where
        E: IntoDimension<Dim = D>,
        S: Data,
    {
        ExactChunks::new(self.view(), chunk_size)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn exact_chunks_mut<E>(&mut self, chunk_size: E) -> ExactChunksMut<'_, A, D>
    where
        E: IntoDimension<Dim = D>,
        S: DataMut,
    {
        ExactChunksMut::new(self.view_mut(), chunk_size)
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn windows<E>(&self, window_size: E) -> Windows<'_, A, D>
    where
        E: IntoDimension<Dim = D>,
        S: Data,
    {
        Windows::new(self.view(), window_size)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn windows_with_stride<E>(&self, window_size: E, stride: E) -> Windows<'_, A, D>
    where
        E: IntoDimension<Dim = D>,
        S: Data,
    {
        Windows::new_with_stride(self.view(), window_size, stride)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn axis_windows(&self, axis: Axis, window_size: usize) -> AxisWindows<'_, A, D>
    where S: Data
    {
        let axis_index = axis.index();

        ndassert!(
            axis_index < self.ndim(),
            concat!(
                "Window axis {} does not match array dimension {} ",
                "(with array of shape {:?})"
            ),
            axis_index,
            self.ndim(),
            self.shape()
        );

        AxisWindows::new(self.view(), axis, window_size)
    }

    
    fn diag_params(&self) -> (Ix, Ixs)
    {
        /* empty shape has len 1 */
        let len = self.dim.slice().iter().cloned().min().unwrap_or(1);
        let stride = self.strides().iter().sum();
        (len, stride)
    }

    
    
    
    
    pub fn diag(&self) -> ArrayView1<'_, A>
    where S: Data
    {
        self.view().into_diag()
    }

    
    pub fn diag_mut(&mut self) -> ArrayViewMut1<'_, A>
    where S: DataMut
    {
        self.view_mut().into_diag()
    }

    
    pub fn into_diag(self) -> ArrayBase<S, Ix1>
    {
        let (len, stride) = self.diag_params();
        
        unsafe { self.with_strides_dim(Ix1(stride as Ix), Ix1(len)) }
    }

    
    
    
    
    
    fn try_ensure_unique(&mut self)
    where S: RawDataMut
    {
        debug_assert!(self.pointer_is_inbounds());
        S::try_ensure_unique(self);
        debug_assert!(self.pointer_is_inbounds());
    }

    
    
    
    fn ensure_unique(&mut self)
    where S: DataMut
    {
        debug_assert!(self.pointer_is_inbounds());
        S::ensure_unique(self);
        debug_assert!(self.pointer_is_inbounds());
    }

    
    
    
    
    
    pub fn is_standard_layout(&self) -> bool
    {
        dimension::is_layout_c(&self.dim, &self.strides)
    }

    
    pub(crate) fn is_contiguous(&self) -> bool
    {
        D::is_contiguous(&self.dim, &self.strides)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn as_standard_layout(&self) -> CowArray<'_, A, D>
    where
        S: Data<Elem = A>,
        A: Clone,
    {
        if self.is_standard_layout() {
            CowArray::from(self.view())
        } else {
            let v = crate::iterators::to_vec_mapped(self.iter(), A::clone);
            let dim = self.dim.clone();
            debug_assert_eq!(v.len(), dim.size());

            unsafe {
                
                
                CowArray::from(Array::from_shape_vec_unchecked(dim, v))
            }
        }
    }

    
    
    
    
    
    
    
    
    
    #[inline(always)]
    pub fn as_ptr(&self) -> *const A
    {
        self.ptr.as_ptr() as *const A
    }

    
    
    
    
    
    /// # Warning
    
    
    
    
    #[inline(always)]
    pub fn as_mut_ptr(&mut self) -> *mut A
    where S: RawDataMut
    {
        self.try_ensure_unique(); 
        self.ptr.as_ptr()
    }

    
    #[inline]
    pub fn raw_view(&self) -> RawArrayView<A, D>
    {
        unsafe { RawArrayView::new(self.ptr, self.dim.clone(), self.strides.clone()) }
    }

    
    
    
    
    #[inline]
    pub fn raw_view_mut(&mut self) -> RawArrayViewMut<A, D>
    where S: RawDataMut
    {
        self.try_ensure_unique(); 
        unsafe { RawArrayViewMut::new(self.ptr, self.dim.clone(), self.strides.clone()) }
    }

    
    
    
    #[inline]
    pub(crate) unsafe fn raw_view_mut_unchecked(&mut self) -> RawArrayViewMut<A, D>
    where S: DataOwned
    {
        RawArrayViewMut::new(self.ptr, self.dim.clone(), self.strides.clone())
    }

    
    
    
    
    
    pub fn as_slice(&self) -> Option<&[A]>
    where S: Data
    {
        if self.is_standard_layout() {
            unsafe { Some(slice::from_raw_parts(self.ptr.as_ptr(), self.len())) }
        } else {
            None
        }
    }

    
    
    pub fn as_slice_mut(&mut self) -> Option<&mut [A]>
    where S: DataMut
    {
        if self.is_standard_layout() {
            self.ensure_unique();
            unsafe { Some(slice::from_raw_parts_mut(self.ptr.as_ptr(), self.len())) }
        } else {
            None
        }
    }

    
    
    
    
    
    pub fn as_slice_memory_order(&self) -> Option<&[A]>
    where S: Data
    {
        if self.is_contiguous() {
            let offset = offset_from_low_addr_ptr_to_logical_ptr(&self.dim, &self.strides);
            unsafe { Some(slice::from_raw_parts(self.ptr.sub(offset).as_ptr(), self.len())) }
        } else {
            None
        }
    }

    
    
    
    
    
    
    pub fn as_slice_memory_order_mut(&mut self) -> Option<&mut [A]>
    where S: DataMut
    {
        self.try_as_slice_memory_order_mut().ok()
    }

    
    
    pub(crate) fn try_as_slice_memory_order_mut(&mut self) -> Result<&mut [A], &mut Self>
    where S: DataMut
    {
        if self.is_contiguous() {
            self.ensure_unique();
            let offset = offset_from_low_addr_ptr_to_logical_ptr(&self.dim, &self.strides);
            unsafe { Ok(slice::from_raw_parts_mut(self.ptr.sub(offset).as_ptr(), self.len())) }
        } else {
            Err(self)
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn to_shape<E>(&self, new_shape: E) -> Result<CowArray<'_, A, E::Dim>, ShapeError>
    where
        E: ShapeArg,
        A: Clone,
        S: Data,
    {
        let (shape, order) = new_shape.into_shape_and_order();
        self.to_shape_order(shape, order.unwrap_or(Order::RowMajor))
    }

    fn to_shape_order<E>(&self, shape: E, order: Order) -> Result<CowArray<'_, A, E>, ShapeError>
    where
        E: Dimension,
        A: Clone,
        S: Data,
    {
        let len = self.dim.size();
        if size_of_shape_checked(&shape) != Ok(len) {
            return Err(error::incompatible_shapes(&self.dim, &shape));
        }

        
        if len == 0 {
            unsafe {
                return Ok(CowArray::from(ArrayView::from_shape_ptr(shape, self.as_ptr())));
            }
        }

        
        match reshape_dim(&self.dim, &self.strides, &shape, order) {
            Ok(to_strides) => unsafe {
                return Ok(CowArray::from(ArrayView::new(self.ptr, shape, to_strides)));
            },
            Err(err) if err.kind() == ErrorKind::IncompatibleShape => {
                return Err(error::incompatible_shapes(&self.dim, &shape));
            }
            _otherwise => {}
        }

        
        unsafe {
            let (shape, view) = match order {
                Order::RowMajor => (shape.set_f(false), self.view()),
                Order::ColumnMajor => (shape.set_f(true), self.t()),
            };
            Ok(CowArray::from(Array::from_shape_trusted_iter_unchecked(shape, view.into_iter(), A::clone)))
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn into_shape_with_order<E>(self, shape: E) -> Result<ArrayBase<S, E::Dim>, ShapeError>
    where E: ShapeArg
    {
        let (shape, order) = shape.into_shape_and_order();
        self.into_shape_with_order_impl(shape, order.unwrap_or(Order::RowMajor))
    }

    fn into_shape_with_order_impl<E>(self, shape: E, order: Order) -> Result<ArrayBase<S, E>, ShapeError>
    where E: Dimension
    {
        let shape = shape.into_dimension();
        if size_of_shape_checked(&shape) != Ok(self.dim.size()) {
            return Err(error::incompatible_shapes(&self.dim, &shape));
        }

        
        unsafe {
            
            match order {
                Order::RowMajor if self.is_standard_layout() =>
                    Ok(self.with_strides_dim(shape.default_strides(), shape)),
                Order::ColumnMajor if self.raw_view().reversed_axes().is_standard_layout() =>
                    Ok(self.with_strides_dim(shape.fortran_strides(), shape)),
                _otherwise => Err(error::from_kind(error::ErrorKind::IncompatibleLayout)),
            }
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[deprecated(note = "Use `.into_shape_with_order()` or `.to_shape()`", since = "0.16.0")]
    pub fn into_shape<E>(self, shape: E) -> Result<ArrayBase<S, E::Dim>, ShapeError>
    where E: IntoDimension
    {
        let shape = shape.into_dimension();
        if size_of_shape_checked(&shape) != Ok(self.dim.size()) {
            return Err(error::incompatible_shapes(&self.dim, &shape));
        }
        
        unsafe {
            
            if self.is_standard_layout() {
                Ok(self.with_strides_dim(shape.default_strides(), shape))
            } else if self.ndim() > 1 && self.raw_view().reversed_axes().is_standard_layout() {
                Ok(self.with_strides_dim(shape.fortran_strides(), shape))
            } else {
                Err(error::from_kind(error::ErrorKind::IncompatibleLayout))
            }
        }
    }

    
    
    
    
    
    
    /// # `.to_shape` vs `.into_shape_clone`
    
    
    
    
    
    
    pub fn into_shape_clone<E>(self, shape: E) -> Result<ArrayBase<S, E::Dim>, ShapeError>
    where
        S: DataOwned,
        A: Clone,
        E: ShapeArg,
    {
        let (shape, order) = shape.into_shape_and_order();
        let order = order.unwrap_or(Order::RowMajor);
        self.into_shape_clone_order(shape, order)
    }

    fn into_shape_clone_order<E>(self, shape: E, order: Order) -> Result<ArrayBase<S, E>, ShapeError>
    where
        S: DataOwned,
        A: Clone,
        E: Dimension,
    {
        let len = self.dim.size();
        if size_of_shape_checked(&shape) != Ok(len) {
            return Err(error::incompatible_shapes(&self.dim, &shape));
        }

        
        if len == 0 {
            unsafe {
                return Ok(self.with_strides_dim(shape.default_strides(), shape));
            }
        }

        
        match reshape_dim(&self.dim, &self.strides, &shape, order) {
            Ok(to_strides) => unsafe {
                return Ok(self.with_strides_dim(to_strides, shape));
            },
            Err(err) if err.kind() == ErrorKind::IncompatibleShape => {
                return Err(error::incompatible_shapes(&self.dim, &shape));
            }
            _otherwise => {}
        }

        
        unsafe {
            let (shape, view) = match order {
                Order::RowMajor => (shape.set_f(false), self.view()),
                Order::ColumnMajor => (shape.set_f(true), self.t()),
            };

            Ok(ArrayBase::from_shape_trusted_iter_unchecked(shape, view.into_iter(), A::clone))
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    #[deprecated(note = "Use `.into_shape_with_order()` or `.to_shape()`", since = "0.16.0")]
    pub fn reshape<E>(&self, shape: E) -> ArrayBase<S, E::Dim>
    where
        S: DataShared + DataOwned,
        A: Clone,
        E: IntoDimension,
    {
        let shape = shape.into_dimension();
        if size_of_shape_checked(&shape) != Ok(self.dim.size()) {
            panic!(
                "ndarray: incompatible shapes in reshape, attempted from: {:?}, to: {:?}",
                self.dim.slice(),
                shape.slice()
            )
        }
        
        if self.is_standard_layout() {
            let cl = self.clone();
            
            unsafe { cl.with_strides_dim(shape.default_strides(), shape) }
        } else {
            let v = self.iter().cloned().collect::<Vec<A>>();
            unsafe { ArrayBase::from_shape_vec_unchecked(shape, v) }
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    pub fn flatten(&self) -> CowArray<'_, A, Ix1>
    where
        A: Clone,
        S: Data,
    {
        self.flatten_with_order(Order::RowMajor)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn flatten_with_order(&self, order: Order) -> CowArray<'_, A, Ix1>
    where
        A: Clone,
        S: Data,
    {
        self.to_shape((self.len(), order)).unwrap()
    }

    
    
    
    
    
    
    
    
    
    
    
    
    pub fn into_flat(self) -> ArrayBase<S, Ix1>
    where
        A: Clone,
        S: DataOwned,
    {
        let len = self.len();
        self.into_shape_clone(Ix1(len)).unwrap()
    }

    
    
    
    
    
    
    
    
    
    pub fn into_dyn(self) -> ArrayBase<S, IxDyn>
    {
        
        unsafe {
            ArrayBase::from_data_ptr(self.data, self.ptr).with_strides_dim(self.strides.into_dyn(), self.dim.into_dyn())
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn into_dimensionality<D2>(self) -> Result<ArrayBase<S, D2>, ShapeError>
    where D2: Dimension
    {
        unsafe {
            if D::NDIM == D2::NDIM {
                
                let dim = unlimited_transmute::<D, D2>(self.dim);
                let strides = unlimited_transmute::<D, D2>(self.strides);
                return Ok(ArrayBase::from_data_ptr(self.data, self.ptr).with_strides_dim(strides, dim));
            } else if D::NDIM.is_none() || D2::NDIM.is_none() {
                
                
                if let Some(dim) = D2::from_dimension(&self.dim) {
                    if let Some(strides) = D2::from_dimension(&self.strides) {
                        return Ok(self.with_strides_dim(strides, dim));
                    }
                }
            }
        }
        Err(ShapeError::from_kind(ErrorKind::IncompatibleShape))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn broadcast<E>(&self, dim: E) -> Option<ArrayView<'_, A, E::Dim>>
    where
        E: IntoDimension,
        S: Data,
    {
        
        
        
        
        
        
        
        
        fn upcast<D: Dimension, E: Dimension>(to: &D, from: &E, stride: &E) -> Option<D>
        {
            
            
            
            
            let _ = size_of_shape_checked(to).ok()?;

            let mut new_stride = to.clone();
            
            
            if to.ndim() < from.ndim() {
                return None;
            }

            {
                let mut new_stride_iter = new_stride.slice_mut().iter_mut().rev();
                for ((er, es), dr) in from
                    .slice()
                    .iter()
                    .rev()
                    .zip(stride.slice().iter().rev())
                    .zip(new_stride_iter.by_ref())
                {
                    /* update strides */
                    if *dr == *er {
                        /* keep stride */
                        *dr = *es;
                    } else if *er == 1 {
                        /* dead dimension, zero stride */
                        *dr = 0
                    } else {
                        return None;
                    }
                }

                /* set remaining strides to zero */
                for dr in new_stride_iter {
                    *dr = 0;
                }
            }
            Some(new_stride)
        }
        let dim = dim.into_dimension();

        
        let broadcast_strides = match upcast(&dim, &self.dim, &self.strides) {
            Some(st) => st,
            None => return None,
        };
        unsafe { Some(ArrayView::new(self.ptr, dim, broadcast_strides)) }
    }

    
    
    
    
    #[allow(clippy::type_complexity)]
    pub(crate) fn broadcast_with<'a, 'b, B, S2, E>(
        &'a self, other: &'b ArrayBase<S2, E>,
    ) -> Result<(ArrayView<'a, A, DimMaxOf<D, E>>, ArrayView<'b, B, DimMaxOf<D, E>>), ShapeError>
    where
        S: Data<Elem = A>,
        S2: Data<Elem = B>,
        D: Dimension + DimMax<E>,
        E: Dimension,
    {
        let shape = co_broadcast::<D, E, <D as DimMax<E>>::Output>(&self.dim, &other.dim)?;
        let view1 = if shape.slice() == self.dim.slice() {
            self.view()
                .into_dimensionality::<<D as DimMax<E>>::Output>()
                .unwrap()
        } else if let Some(view1) = self.broadcast(shape.clone()) {
            view1
        } else {
            return Err(from_kind(ErrorKind::IncompatibleShape));
        };
        let view2 = if shape.slice() == other.dim.slice() {
            other
                .view()
                .into_dimensionality::<<D as DimMax<E>>::Output>()
                .unwrap()
        } else if let Some(view2) = other.broadcast(shape) {
            view2
        } else {
            return Err(from_kind(ErrorKind::IncompatibleShape));
        };
        Ok((view1, view2))
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn swap_axes(&mut self, ax: usize, bx: usize)
    {
        self.dim.slice_mut().swap(ax, bx);
        self.strides.slice_mut().swap(ax, bx);
    }

    
    
    
    
    
    
    
    
    
    
    
    /// # Examples
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn permuted_axes<T>(self, axes: T) -> ArrayBase<S, D>
    where T: IntoDimension<Dim = D>
    {
        let axes = axes.into_dimension();
        
        let mut usage_counts = D::zeros(self.ndim());
        for axis in axes.slice() {
            usage_counts[*axis] += 1;
        }
        for count in usage_counts.slice() {
            assert_eq!(*count, 1, "each axis must be listed exactly once");
        }
        
        let mut new_dim = usage_counts; 
        let mut new_strides = D::zeros(self.ndim());
        {
            let dim = self.dim.slice();
            let strides = self.strides.slice();
            for (new_axis, &axis) in axes.slice().iter().enumerate() {
                new_dim[new_axis] = dim[axis];
                new_strides[new_axis] = strides[axis];
            }
        }
        
        unsafe { self.with_strides_dim(new_strides, new_dim) }
    }

    
    
    
    
    pub fn reversed_axes(mut self) -> ArrayBase<S, D>
    {
        self.dim.slice_mut().reverse();
        self.strides.slice_mut().reverse();
        self
    }

    
    
    
    
    
    pub fn t(&self) -> ArrayView<'_, A, D>
    where S: Data
    {
        self.view().reversed_axes()
    }

    
    pub fn axes(&self) -> Axes<'_, D>
    {
        axes_of(&self.dim, &self.strides)
    }

    /*
    
    pub fn min_stride_axis(&self) -> Axis {
        self.dim.min_stride_axis(&self.strides)
    }
    */

    
    
    pub fn max_stride_axis(&self) -> Axis
    {
        self.dim.max_stride_axis(&self.strides)
    }

    
    
    
    #[track_caller]
    pub fn invert_axis(&mut self, axis: Axis)
    {
        unsafe {
            let s = self.strides.axis(axis) as Ixs;
            let m = self.dim.axis(axis);
            if m != 0 {
                self.ptr = self.ptr.offset(stride_offset(m - 1, s as Ix));
            }
            self.strides.set_axis(axis, (-s) as Ix);
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn merge_axes(&mut self, take: Axis, into: Axis) -> bool
    {
        merge_axes(&mut self.dim, &mut self.strides, take, into)
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn insert_axis(self, axis: Axis) -> ArrayBase<S, D::Larger>
    {
        assert!(axis.index() <= self.ndim());
        
        unsafe {
            let strides = self.strides.insert_axis(axis);
            let dim = self.dim.insert_axis(axis);
            self.with_strides_dim(strides, dim)
        }
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn remove_axis(self, axis: Axis) -> ArrayBase<S, D::Smaller>
    where D: RemoveAxis
    {
        self.index_axis_move(axis, 0)
    }

    pub(crate) fn pointer_is_inbounds(&self) -> bool
    {
        self.data._is_pointer_inbounds(self.as_ptr())
    }

    
    
    
    
    
    #[track_caller]
    pub fn assign<E: Dimension, S2>(&mut self, rhs: &ArrayBase<S2, E>)
    where
        S: DataMut,
        A: Clone,
        S2: Data<Elem = A>,
    {
        self.zip_mut_with(rhs, |x, y| x.clone_from(y));
    }

    
    
    
    
    
    
    #[track_caller]
    pub fn assign_to<P>(&self, to: P)
    where
        S: Data,
        P: IntoNdProducer<Dim = D>,
        P::Item: AssignElem<A>,
        A: Clone,
    {
        Zip::from(self).map_assign_into(to, A::clone);
    }

    
    pub fn fill(&mut self, x: A)
    where
        S: DataMut,
        A: Clone,
    {
        self.map_inplace(move |elt| elt.clone_from(&x));
    }

    pub(crate) fn zip_mut_with_same_shape<B, S2, E, F>(&mut self, rhs: &ArrayBase<S2, E>, mut f: F)
    where
        S: DataMut,
        S2: Data<Elem = B>,
        E: Dimension,
        F: FnMut(&mut A, &B),
    {
        debug_assert_eq!(self.shape(), rhs.shape());

        if self.dim.strides_equivalent(&self.strides, &rhs.strides) {
            if let Some(self_s) = self.as_slice_memory_order_mut() {
                if let Some(rhs_s) = rhs.as_slice_memory_order() {
                    for (s, r) in self_s.iter_mut().zip(rhs_s) {
                        f(s, r);
                    }
                    return;
                }
            }
        }

        
        self.zip_mut_with_by_rows(rhs, f);
    }

    
    #[inline(always)]
    fn zip_mut_with_by_rows<B, S2, E, F>(&mut self, rhs: &ArrayBase<S2, E>, mut f: F)
    where
        S: DataMut,
        S2: Data<Elem = B>,
        E: Dimension,
        F: FnMut(&mut A, &B),
    {
        debug_assert_eq!(self.shape(), rhs.shape());
        debug_assert_ne!(self.ndim(), 0);

        
        let n = self.ndim();
        let dim = self.raw_dim();
        Zip::from(LanesMut::new(self.view_mut(), Axis(n - 1)))
            .and(Lanes::new(rhs.broadcast_assume(dim), Axis(n - 1)))
            .for_each(move |s_row, r_row| Zip::from(s_row).and(r_row).for_each(&mut f));
    }

    fn zip_mut_with_elem<B, F>(&mut self, rhs_elem: &B, mut f: F)
    where
        S: DataMut,
        F: FnMut(&mut A, &B),
    {
        self.map_inplace(move |elt| f(elt, rhs_elem));
    }

    
    
    
    
    
    
    #[track_caller]
    #[inline]
    pub fn zip_mut_with<B, S2, E, F>(&mut self, rhs: &ArrayBase<S2, E>, f: F)
    where
        S: DataMut,
        S2: Data<Elem = B>,
        E: Dimension,
        F: FnMut(&mut A, &B),
    {
        if rhs.dim.ndim() == 0 {
            
            self.zip_mut_with_elem(rhs.get_0d(), f);
        } else if self.dim.ndim() == rhs.dim.ndim() && self.shape() == rhs.shape() {
            self.zip_mut_with_same_shape(rhs, f);
        } else {
            let rhs_broadcast = rhs.broadcast_unwrap(self.raw_dim());
            self.zip_mut_with_by_rows(&rhs_broadcast, f);
        }
    }

    
    
    
    
    pub fn fold<'a, F, B>(&'a self, init: B, f: F) -> B
    where
        F: FnMut(B, &'a A) -> B,
        A: 'a,
        S: Data,
    {
        if let Some(slc) = self.as_slice_memory_order() {
            slc.iter().fold(init, f)
        } else {
            let mut v = self.view();
            move_min_stride_axis_to_last(&mut v.dim, &mut v.strides);
            v.into_elements_base().fold(init, f)
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn map<'a, B, F>(&'a self, f: F) -> Array<B, D>
    where
        F: FnMut(&'a A) -> B,
        A: 'a,
        S: Data,
    {
        unsafe {
            if let Some(slc) = self.as_slice_memory_order() {
                ArrayBase::from_shape_trusted_iter_unchecked(
                    self.dim.clone().strides(self.strides.clone()),
                    slc.iter(),
                    f,
                )
            } else {
                ArrayBase::from_shape_trusted_iter_unchecked(self.dim.clone(), self.iter(), f)
            }
        }
    }

    
    
    
    
    
    
    pub fn map_mut<'a, B, F>(&'a mut self, f: F) -> Array<B, D>
    where
        F: FnMut(&'a mut A) -> B,
        A: 'a,
        S: DataMut,
    {
        let dim = self.dim.clone();
        if self.is_contiguous() {
            let strides = self.strides.clone();
            let slc = self.as_slice_memory_order_mut().unwrap();
            unsafe { ArrayBase::from_shape_trusted_iter_unchecked(dim.strides(strides), slc.iter_mut(), f) }
        } else {
            unsafe { ArrayBase::from_shape_trusted_iter_unchecked(dim, self.iter_mut(), f) }
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn mapv<B, F>(&self, mut f: F) -> Array<B, D>
    where
        F: FnMut(A) -> B,
        A: Clone,
        S: Data,
    {
        self.map(move |x| f(x.clone()))
    }

    
    
    
    
    pub fn mapv_into<F>(mut self, f: F) -> Self
    where
        S: DataMut,
        F: FnMut(A) -> A,
        A: Clone,
    {
        self.mapv_inplace(f);
        self
    }

    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn mapv_into_any<B, F>(self, mut f: F) -> Array<B, D>
    where
        S: DataMut,
        F: FnMut(A) -> B,
        A: Clone + 'static,
        B: 'static,
    {
        if core::any::TypeId::of::<A>() == core::any::TypeId::of::<B>() {
            
            
            let f = |a| {
                let b = f(a);
                
                unsafe { unlimited_transmute::<B, A>(b) }
            };
            
            
            let output = self.mapv_into(f).into_owned();
            
            
            unsafe { unlimited_transmute::<Array<A, D>, Array<B, D>>(output) }
        } else {
            
            
            self.mapv(f)
        }
    }

    
    
    
    pub fn map_inplace<'a, F>(&'a mut self, f: F)
    where
        S: DataMut,
        A: 'a,
        F: FnMut(&'a mut A),
    {
        match self.try_as_slice_memory_order_mut() {
            Ok(slc) => slc.iter_mut().for_each(f),
            Err(arr) => {
                let mut v = arr.view_mut();
                move_min_stride_axis_to_last(&mut v.dim, &mut v.strides);
                v.into_elements_base().for_each(f);
            }
        }
    }

    
    
    
    
    
    
    /// # #[cfg(feature = "approx")] {
    
    
    
    
    
    
    
    
    
    
    
    
    /// # }
    
    pub fn mapv_inplace<F>(&mut self, mut f: F)
    where
        S: DataMut,
        F: FnMut(A) -> A,
        A: Clone,
    {
        self.map_inplace(move |x| *x = f(x.clone()));
    }

    
    
    
    pub fn for_each<'a, F>(&'a self, mut f: F)
    where
        F: FnMut(&'a A),
        A: 'a,
        S: Data,
    {
        self.fold((), move |(), elt| f(elt))
    }

    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn fold_axis<B, F>(&self, axis: Axis, init: B, mut fold: F) -> Array<B, D::Smaller>
    where
        D: RemoveAxis,
        F: FnMut(&B, &A) -> B,
        B: Clone,
        S: Data,
    {
        let mut res = Array::from_elem(self.raw_dim().remove_axis(axis), init);
        for subview in self.axis_iter(axis) {
            res.zip_mut_with(&subview, |x, y| *x = fold(x, y));
        }
        res
    }

    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn map_axis<'a, B, F>(&'a self, axis: Axis, mut mapping: F) -> Array<B, D::Smaller>
    where
        D: RemoveAxis,
        F: FnMut(ArrayView1<'a, A>) -> B,
        A: 'a,
        S: Data,
    {
        if self.len_of(axis) == 0 {
            let new_dim = self.dim.remove_axis(axis);
            Array::from_shape_simple_fn(new_dim, move || mapping(ArrayView::from(&[])))
        } else {
            Zip::from(self.lanes(axis)).map_collect(mapping)
        }
    }

    
    
    
    
    
    
    
    
    
    
    #[track_caller]
    pub fn map_axis_mut<'a, B, F>(&'a mut self, axis: Axis, mut mapping: F) -> Array<B, D::Smaller>
    where
        D: RemoveAxis,
        F: FnMut(ArrayViewMut1<'a, A>) -> B,
        A: 'a,
        S: DataMut,
    {
        if self.len_of(axis) == 0 {
            let new_dim = self.dim.remove_axis(axis);
            Array::from_shape_simple_fn(new_dim, move || mapping(ArrayViewMut::from(&mut [])))
        } else {
            Zip::from(self.lanes_mut(axis)).map_collect(mapping)
        }
    }

    
    
    
    
    
    
    
    
    
    
    
    pub fn remove_index(&mut self, axis: Axis, index: usize)
    where S: DataOwned + DataMut
    {
        assert!(index < self.len_of(axis), "index {} must be less than length of Axis({})",
                index, axis.index());
        let (_, mut tail) = self.view_mut().split_at(axis, index);
        
        Zip::from(tail.lanes_mut(axis)).for_each(|mut lane| lane.rotate1_front());
        
        self.slice_axis_inplace(axis, Slice::new(0, Some(-1), 1));
    }

    
    
    
    
    
    
    
    /// # Example
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    
    pub fn accumulate_axis_inplace<F>(&mut self, axis: Axis, mut f: F)
    where
        F: FnMut(&A, &mut A),
        S: DataMut,
    {
        if self.len_of(axis) <= 1 {
            return;
        }
        let mut curr = self.raw_view_mut(); 
        let mut prev = curr.raw_view(); 
        prev.slice_axis_inplace(axis, Slice::from(..-1));
        curr.slice_axis_inplace(axis, Slice::from(1..));
        
        Zip::from(prev).and(curr).for_each(|prev, curr| unsafe {
            
            
            
            
            
            
            
            
            
            
            f(&*prev, &mut *curr)
        });
    }
}







#[track_caller]
#[inline]
unsafe fn unlimited_transmute<A, B>(data: A) -> B
{
    
    assert_eq!(size_of::<A>(), size_of::<B>());
    let old_data = ManuallyDrop::new(data);
    (&*old_data as *const A as *const B).read()
}

type DimMaxOf<A, B> = <A as DimMax<B>>::Output;

#[cfg(test)]
mod tests
{
    use super::*;
    use crate::arr3;

    #[test]
    fn test_flatten()
    {
        let array = arr3(&[[[1, 2], [3, 4]], [[5, 6], [7, 8]]]);
        let flattened = array.flatten();
        assert_eq!(flattened, arr1(&[1, 2, 3, 4, 5, 6, 7, 8]));
    }

    #[test]
    fn test_flatten_with_order()
    {
        let array = arr2(&[[1, 2], [3, 4], [5, 6], [7, 8]]);
        let flattened = array.flatten_with_order(Order::RowMajor);
        assert_eq!(flattened, arr1(&[1, 2, 3, 4, 5, 6, 7, 8]));
        let flattened = array.flatten_with_order(Order::ColumnMajor);
        assert_eq!(flattened, arr1(&[1, 3, 5, 7, 2, 4, 6, 8]));
    }

    #[test]
    fn test_into_flat()
    {
        let array = arr3(&[[[1, 2], [3, 4]], [[5, 6], [7, 8]]]);
        let flattened = array.into_flat();
        assert_eq!(flattened, arr1(&[1, 2, 3, 4, 5, 6, 7, 8]));
    }
}
