% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_adi.R
\name{get_adi}
\alias{get_adi}
\title{Get Area Deprivation Index (ADI) and Berg Indices (ADI-3)}
\usage{
get_adi(
  geography = c("state", "county", "tract", "block group", "zcta",
    "zip code tabulation area"),
  state = NULL,
  county = NULL,
  geoid = NULL,
  zcta = NULL,
  year,
  dataset = c("acs5", "acs3", "acs1", "decennial"),
  geometry = FALSE,
  keep_indicators = FALSE,
  raw_data_only = FALSE,
  cache_tables = TRUE,
  key = NULL,
  evaluator = purrr::insistently(eval, rate = purrr::rate_delay(), quiet = FALSE),
  ...
)
}
\arguments{
\item{geography}{A character string denoting the level of census geography
whose ADIs and ADI-3s you'd like to obtain. Must be one of \code{c("state", "county", "tract", "block group", "zcta")} (\code{"zip code tabulation area"}
will be changed to \code{"zcta"}).}

\item{state}{A character string specifying states whose ADI and ADI-3 data is
desired. Defaults to \code{NULL}. Can contain full state names, two-letter state
abbreviations, or a two-digit FIPS code/GEOID (must be a vector of strings,
so use quotation marks and leading zeros if necessary). Must be left as
\code{NULL} if using the \code{geoid}.

If \code{geography = "zcta"} and \code{year} is earlier than 2020, this argument can
be used to only include ZCTAs associated with specific states, as
determined by the US Census bureau. If year is 2020 or later, this argument
will be ignored with a \link{warning}.}

\item{county}{A vector of character strings specifying the counties whose ADI
and ADI-3 data you're requesting. Defaults to \code{NULL}. If not \code{NULL}, the
\code{state} parameter must have a length of 1. County names and three-digit
FIPS codes are accepted (must contain strings, so use quotation marks and
leading zeros if necessary). Must be \code{NULL} if using the \code{geoid} parameter
or if \code{geography = "zcta"}.}

\item{geoid}{A character vector of GEOIDs (use quotation marks and leading
zeros). Defaults to \code{NULL}. Must be blank if \code{state}, \code{county}, or \code{zcta}
is used. Can contain different levels of geography (see details).}

\item{zcta}{A character vector of ZCTAs or the leading digit(s) of ZCTAs (use
quotation marks and leading zeros). Defaults to \code{NULL}. Must be blank if
\code{county} or \code{geoid} is used.

Strings under 5 digits long will yield all ZCTAs that begin with those
digits.

Requires that \code{geography = "zcta"}. If \code{geography = "zcta"} and \code{zcta = NULL}, all ZCTAs in the 50 states plus DC and Puerto Rico will be used.}

\item{year}{Single integer specifying the year of US Census data to use.}

\item{dataset}{The data set used to calculate ADIs and ADI-3s. Must be one of
\code{c("acs5", "acs3", "acs1", "decennial")}, denoting the 5-, 3-, and 1-year
ACS along with the decennial census. Defaults to \code{"acs5"}.

When \code{dataset = "decennial"}, \code{year} must be in \code{c(2000, 2010, 2020)}.

The 2010 and 2020 decennial censuses did not include the long-form
questionnaire used in the 1990 and 2000 censuses, so this function uses the
5-year estimates from the 2010/2020 ACS to supply the data not included in
the decennial censuses.

Important: data are not always available depending on the level of
geography and data set chosen. See \code{\link{dataset_year_geography_availability}}
and \url{https://www.census.gov/programs-surveys/acs/guidance/estimates.html}.}

\item{geometry}{Logical value indicating whether or not shapefile data should
be included in the result, making the result an \code{\link[sf:sf]{sf}} object
instead of a plain \code{\link[tibble:tibble]{tibble}}. Defaults to \code{FALSE}.

The shapefile data that is returned is somewhat customizable by passing
certain arguments along to the \code{tidycensus} functions via \code{...}.}

\item{keep_indicators}{Logical value indicating whether or not the resulting
\code{\link[tibble:tibble]{tibble}} or \code{\link[sf:sf]{sf}} object will contain the
socioeconomic measures used to calculate the ADI and ADI-3 values. Defaults
to \code{FALSE}.

See \code{\link{acs_vars}} and \code{\link{decennial_vars}} for basic descriptions of the raw
census variables.}

\item{raw_data_only}{Logical, indicating whether or not to skip calculation
of the ADI and ADI-3 and only return the census variables. Defaults to
\code{FALSE}.}

\item{cache_tables}{The plural version of the \code{cache_table} argument in
\code{\link[tidycensus:get_acs]{tidycensus::get_acs()}} or \code{\link[tidycensus:get_decennial]{tidycensus::get_decennial()}}. (\code{get_adi()}
calls the necessary \code{tidycensus} function many times in order to return
ADIs and ADI-3s, so many tables are cached if \code{TRUE}). Defaults to \code{TRUE}.}

\item{key}{Your Census API key as a character string. Obtain one at
\url{http://api.census.gov/data/key_signup.html}. Defaults to \code{NULL}. Not
necessary if you have already loaded your key with \code{\link[=census_api_key]{census_api_key()}}.}

\item{evaluator}{A function that will evaluate the calls to the tidycensus
functions. Defaults to \code{\link[purrr:insistently]{purrr::insistently}}\code{(}\code{\link{eval}}\verb{, rate = }\code{\link[purrr:rate-helpers]{purrr::rate_delay()}}\verb{, quiet = FALSE)}.}

\item{...}{Additional arguments to be passed onto \code{\link[tidycensus:get_acs]{tidycensus::get_acs()}} or
\code{\link[tidycensus:get_decennial]{tidycensus::get_decennial()}}. These must all be named. Must not match any
of the \code{tidycensus} formal arguments that \code{sociome} needs to set
explicitly.

This may be found to be helpful when setting \code{geometry = TRUE}, since the
\code{tidycensus} functions pass \code{...} onto the appropriate \code{tigris} function
(namely, one of \code{\link[tigris:states]{tigris::states()}}, \code{\link[tigris:counties]{tigris::counties()}},
\code{\link[tigris:tracts]{tigris::tracts()}}, \code{\link[tigris:block_groups]{tigris::block_groups()}}, or \code{\link[tigris:zctas]{tigris::zctas()}},
according to the the value of \code{geography}). This enables the user to
somewhat customize the shapefile data obtained.}
}
\value{
If \code{geometry = FALSE}, (the default) a \code{\link[tibble:tibble]{tibble}}. If
\code{geometry = TRUE} is specified, an \code{\link[sf:sf]{sf}}.
}
\description{
Returns the ADI and ADI-3 of user-specified areas.
}
\details{
Returns a \code{\link[tibble:tibble]{tibble}} or \code{\link[sf:sf]{sf}} object of the Area
Deprivation Indices (ADIs) and Berg Indices (ADI-3s) of user-specified
locations in the United States, utilizing US Census data. Locations that are
listed as having zero households are excluded from ADI and ADI-3 calculation:
their ADI and ADI-3 values will be \code{NA}.
}
\section{Reference area}{


\strong{The concept of "reference area" is important to
understand when using this function.} The algorithm that produced the
original ADIs employs factor analysis. As a result, the ADI is a relative
measure; the ADI of a particular location is dynamic, varying depending on
which other locations were supplied to the algorithm. In other words, \strong{ADI
will vary depending on the reference area you specify.}

For example, the ADI of Orange County, California is \emph{x} when calculated
alongside all other counties in California, but it is \emph{y} when calculated
alongside all counties in the US. The \code{get_adi()} function enables the user
to define a \strong{reference area} by feeding a vector of GEOIDs to its \code{geoid}
parameter (or alternatively for convenience, states and/or counties to
\code{state} and \code{county}). The function then gathers data from those specified
locations and performs calculations using their data alone.

The Berg Indices (ADI-3) were developed with this principle of relativity
in mind, and as such there is no set of seminal ADI-3 values. Thus, the
terms "Berg Indices" and "ADI-3" refer more nearly to any values generated
using the algorithm employed in this package.

Areas listed as having zero households are excluded from the reference
area, and their ADI and ADI-3 values will be \code{NA}.
}

\section{The \code{geoid} parameter}{
 Elements of \code{geoid} can represent different
levels of geography, but they all must be either 2 digits (for states), 5
digits (for counties), 11 digits (for tracts), or 12 digits (for block
groups). It must contain character strings, so use quotation marks as well
as leading zeros where applicable.
}

\section{ADI and ADI-3 factor loadings}{
 The returned
\code{\link[tibble:tibble]{tibble}} or \code{\link[sf:sf]{sf}} is of class \code{adi}, and it
contains an attribute called \code{loadings}, which contains a tibble of the PCA
loadings of each factor. This is accessible through
\code{\link{attr}}\verb{(name_of_tibble, "loadings")}.
}

\section{Missingness and imputation}{
 While this function allows flexibility
in specifying reference areas (see the \strong{Reference area} section above),
data from the US Census are masked for sparsely populated places, resulting
in many missing values.

Imputation is attempted via \code{\link[mice:mice]{mice::mice}}\verb{(m = 1, maxit = 50, method = "pmm")}. If imputation is unsuccessful, an error is thrown, but the dataset
of indicators on which imputation was unsuccessful is available via
\code{\link[rlang:last_error]{rlang::last_error()}}\verb{$adi_indicators} and the raw census data are
available via \code{\link[rlang:last_error]{rlang::last_error()}}\verb{$adi_raw_data}. The former excludes
areas with zero households, but the latter includes them.

One of the indicators of both ADI and the Financial Strength component of
ADI-3 is median family income, but methodological issues with the 2015 and
2016 ACS have rendered this variable unavailable at the block group level
for those years. When requested, this function will use median household
income in its place, with a \code{warning()}. See
\url{https://www.census.gov/programs-surveys/acs/technical-documentation/user-notes/2016-01.html}.
}

\section{API-related error handling}{
 Depending on user input, this function
may call its underlying functions (\code{\link[tidycensus:get_acs]{tidycensus::get_acs()}} or
\code{\link[tidycensus:get_decennial]{tidycensus::get_decennial()}}) many times in order to accommodate their
behavior. When these calls are broken up by state or by state and county, a
message is printed indicating the state or state and county whose data is
being pulled. These calls are wrapped in
\code{\link[purrr:insistently]{purrr::insistently}}\code{(}\code{\link[purrr:rate-helpers]{purrr::rate_delay()}}\verb{, quiet = FALSE)}, meaning
that they are attempted over and over until success, and \code{tidycensus} error
messages are printed as they occur.
}

\section{Warnings and disclaimers}{
 Please note that this function calls data
from US Census servers, so execution may take a long time depending on the
user's internet connection and the amount of data requested.

For advanced users, if changing the \code{dataset} argument, be sure to know the
advantages and limitations of the 1-year and 3-year ACS estimates. See
\url{https://www.census.gov/programs-surveys/acs/guidance/estimates.html} for
details.
}

\examples{
\dontrun{
# Wrapped in \dontrun{} because all these examples take >5 seconds
# and require a Census API key.

# ADI of all census tracts in Cuyahoga County, Ohio
get_adi(geography = "tract", year = 2017, state = "OH", county = "Cuyahoga")

# ADI and ADI-3 of all counties in Connecticut, using the 2014 ACS1 survey.
# Returns a warning because there are only 8 counties.
# A minimum of 30 locations is recommended.
get_adi(geography = "county", state = "CT", year = 2014, dataset = "acs1")

# Areas with zero households will have an ADI and ADI-3 of NA:
queens <-
  get_adi(
    "tract",
    year = 2017,
    state = "NY",
    county = "Queens",
    keep_indicators = TRUE,
    geometry = TRUE
  )
queens \%>\%
  dplyr::as_tibble() \%>\%
  dplyr::select(GEOID, NAME, ADI, households = B11005_001) \%>\%
  dplyr::filter(is.na(ADI) | households == 0) \%>\%
  print(n = Inf)

# geoid argument allows for highly customized reference populations.
# ADI of all census tracts in the GEOIDs stored in "delmarva" below:
# Notice the mixing of state- ("10") and county-level GEOIDs (the others).
delmarva_geoids <- c("10", "51001", "51131", "24015", "24029", "24035",
                     "24011", "24041", "24019", "24045", "24039", "24047")
delmarva <-
  get_adi(
    geography = "tract",
    geoid = delmarva_geoids,
    dataset = "acs5",
    year = 2009,
    geometry = TRUE
  )

# Demonstration of geom_sf() integration:
require(ggplot2)

# The na.value argument changes the fill of NA ADI areas.
delmarva \%>\% ggplot() + geom_sf(aes(fill = ADI), lwd = 0)

# Setting direction = -1 makes the less deprived areas the lighter ones
# The argument na.value changes the color of zero-household areas
queens \%>\%
  ggplot() +
  geom_sf(aes(fill = ADI), lwd = 0) +
  scale_fill_viridis_c(na.value = "red", direction = -1)

# Obtain factor loadings:
attr(queens, "loadings")
}
}
\seealso{
\code{\link{dataset_year_geography_availability}} for usable combinations of
\code{geography}, \code{year}, and \code{dataset}.
}
