% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survdiff_ci.R
\name{survdiff_ci}
\alias{survdiff_ci}
\title{Estimate Difference in Survival or Cumulative Incidence and Confidence Interval}
\usage{
survdiff_ci(
  formula,
  data,
  time,
  estimand = c("survival", "cuminc"),
  type = c("diff", "ratio"),
  approach = c("mover", "squareadd"),
  conf.level = 0.95,
  event_type = NULL,
  id_variable = NULL,
  weighted = FALSE
)
}
\arguments{
\item{formula}{Formula of a survival object using
\code{\link[survival]{Surv}} of the form, \code{Surv(time, event) ~ group}.
The exposure variable (here, \code{group}) must be categorical with at
least 2 categories.}

\item{data}{Data set.}

\item{time}{Time point to estimate survival difference at.}

\item{estimand}{Optional. Estimate difference in survival (\code{"survival"})
or cumulative incidence (\code{"cuminc"})? This parameter affects the
sign of the differences. Only \code{"cuminc"} is available if competing
events are present, i.e., \code{event_type} is not \code{NULL}.
Defaults to \code{"survival"}.}

\item{type}{Optional. Estimate differences (\code{"diff"}) or ratio
(\code{"ratio"}) of survival or cumulative incidence? Defaults to
\code{"diff"}.}

\item{approach}{Optional. For estimating confidence limits of differences,
use the MOVER approach based on upper and lower confidence limits of each
group (\code{"mover"}), or square-and-add standard errors
(\code{"squareadd"})? Defaults to \code{"mover"}. (For confidence limits of
ratios, this argument is ignored and MOVER is used.)}

\item{conf.level}{Optional. Confidence level. Defaults to \code{0.95}.}

\item{event_type}{Optional. Event type (level) for event variable with
competing events. Defaults to \code{NULL}.}

\item{id_variable}{Optional. Identifiers for individual oberversations,
required if data are clustered, or if competing events and time/time2
notation are used concomitantly.}

\item{weighted}{Optional. Weigh survival curves, e.g. for inverse-probability
weighting, before estimating differences or ratios? If \code{TRUE}, the
\code{data} must contain a variable called \code{.weights}. Defaults to
\code{FALSE}.}
}
\value{
Tibble in \code{\link[broom]{tidy}} format:
\itemize{
\item \code{term} Name of the exposure stratum.
\item \code{estimate} Difference or ratio.
\item \code{std.error} Large-sample standard error of the difference in survival
functions (see References). For each survival function, Greenwood
standard errors with log transformation are used, the default of the
survival package/\code{\link[survival]{survfit}}).
\item \code{statistic} z statistic.
\item \code{p.value} From the z statistic.
\item \code{conf.low} Lower confidence limit
\item \code{conf.high} Upper confidence limit
}
}
\description{
This function estimates the unadjusted difference or ratio in survival or
cumulative incidence (risk) at a given time point based on the difference
between per-group Kaplan-Meier estimates or, if competing events are present,
Aalen-Johansen estimates of the cumulative incidence.

For constructing confidence limits, the MOVER approach described by Zou and
Donner (2008) is used, with estimation on the log scale for ratios.
}
\examples{
# Load 'cancer' dataset from survival package (Used in all examples)
data(cancer, package = "survival")

cancer <- cancer |>
  dplyr::mutate(
    sex = factor(
      sex,
      levels = 1:2,
      labels = c("Male", "Female")
    ),
    status = status - 1
  )

survdiff_ci(
  formula = survival::Surv(time = time, event = status) ~ sex,
  data = cancer,
  time = 365.25
)
# Females have 19 percentage points higher one-year survival than males
# (95\% CI, 5 to 34 percentage points).
}
\references{
Com-Nougue C, Rodary C, Patte C. How to establish equivalence when data are
censored: a randomized trial of treatments for B non-Hodgkin lymphoma.
Stat Med 1993;12:1353–64. \doi{10.1002/sim.4780121407}

Altman DG, Andersen PK. Calculating the number needed to treat for trials
where the outcome is time to an event. BMJ 1999;319:1492–5.
\doi{10.1136/bmj.319.7223.1492}

Zou GY, Donner A. Construction of confidence limits about effect measures:
A general approach. Statist Med 2008;27:1693–1702.
\doi{10.1002/sim.3095}
}
