% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils-projects.R
\name{project_path}
\alias{project_path}
\title{Get path relative to project root (Quarto-aware)}
\usage{
project_path(..., root = NULL)
}
\arguments{
\item{...}{Character vectors of path components to be joined}

\item{root}{Project root directory. If \code{NULL} (default), automatic detection
is used following the hierarchy described above}
}
\value{
A character vector of the normalized file path relative to the project root.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}

This function constructs file paths relative to the project root when
running in a Quarto context (using \code{QUARTO_PROJECT_ROOT} or \code{QUARTO_PROJECT_DIR}
environment variables), or falls back to intelligent project root detection
when not in a Quarto context.

It is experimental and subject to change in future releases. The automatic
project root detection may not work reliably in all contexts, especially when
projects have complex directory structures or when running in non-standard
environments. For a more explicit and potentially more robust approach,
consider using \code{\link[here:i_am]{here::i_am()}} to declare your project structure,
followed by \code{\link[here:here]{here::here()}} for path construction. See examples for comparison.
}
\details{
The function uses the following fallback hierarchy to determine the project root:
\itemize{
\item Quarto environment variables set during Quarto commands (e.g., \verb{quarto render}):
\itemize{
\item \code{QUARTO_PROJECT_ROOT} environment variable (set by Quarto commands)
\item \code{QUARTO_PROJECT_DIR} environment variable (alternative Quarto variable)
}
\item Fallback to intelligent project root detection using \code{\link[xfun:proj_root]{xfun::proj_root()}} for interactive sessions:
\itemize{
\item \verb{_quarto.yml} or \verb{_quarto.yaml} (Quarto project files)
\item \code{DESCRIPTION} file with \verb{Package:} field (R package or Project)
\item \code{.Rproj} files with \verb{Version:} field (RStudio projects)
}
}

Last fallback is the current working directory if no project root can be determined.
A warning is issued to alert users that behavior may differ between interactive use and Quarto rendering,
as in this case the computed path may be wrong.
}
\section{Use in Quarto document cells}{


This function is particularly useful in Quarto document cells where you want to
use a path relative to the project root dynamically during rendering.

\if{html}{\out{<div class="sourceCode markdown">}}\preformatted{```\{r\}`r ''`
 # Get a csv path from data directory in the Quarto project root
 data <- project_path("data", "my_data.csv")
```
}\if{html}{\out{</div>}}
}

\examples{
\dontrun{
# Create a dummy Quarto project structure for example
tmpdir <- tempfile("quarto_project")
dir.create(tmpdir)
quarto::quarto_create_project(
  'test project', type = 'blog',
  dir = tmpdir, no_prompt = TRUE, quiet = TRUE
)
project_dir <- file.path(tmpdir, "test project")

# Simulate working within a blog post
xfun::in_dir(
  dir = file.path(project_dir, "posts", "welcome"), {

  # Reference a data file from project root
  # ../../data/my_data.csv
  quarto::project_path("data", "my_data.csv")

  # Reference a script from project root
  # ../../R/analysis.R
  quarto::project_path("R", "analysis.R")

  # Explicitly specify root (overrides automatic detection)
  # ../../data/file.csv
  quarto::project_path("data", "file.csv", root = "../..")

  # Alternative approach using here::i_am() (potentially more robust)
  # This approach requires you to declare where you are in the project:
  if (requireNamespace("here", quietly = TRUE)) {
    # Declare that this document is in the project root or subdirectory
    here::i_am("posts/welcome/index.qmd")

    # Now here::here() will work reliably from the project root
    here::here("data", "my_data.csv")
    here::here("R", "analysis.R")
  }
})

}

}
\seealso{
\itemize{
\item \code{\link[here:here]{here::here()}} and \code{\link[here:i_am]{here::i_am()}} for a similar function that works with R projects
\item \code{\link[=find_project_root]{find_project_root()}} to search for Quarto Project configuration in parents directories
\item \code{\link[=get_running_project_root]{get_running_project_root()}} for detecting the project root in Quarto commands
\item \code{\link[xfun:from_root]{xfun::from_root()}} for the underlying path construction
\item \code{\link[xfun:proj_root]{xfun::proj_root()}} for project root detection logic
}
}
