\name{choose.pweibull}
\alias{choose.pweibull}
\alias{choose2.pweibull}

\title{
Choose a Partition for a Power Piecewise Weibull Model
}

\description{
Selects a time partition for the power piecewise Weibull model, given a maximum
number of allowed partitions. For \code{choose.pweibull}, the user may specify
whether the \eqn{\lambda}'s are equal (default \code{FALSE}), whether the
\eqn{\alpha}'s are equal (default \code{FALSE}), and whether \eqn{\alpha} is
fixed (default \code{FALSE}).  
For \code{choose2.pweibull}, the procedure selects the best combination among:

\itemize{
\item \eqn{\lambda}'s different and \eqn{\alpha}'s different,
\item \eqn{\lambda}'s equal and \eqn{\alpha}'s different,
\item \eqn{\lambda}'s different and \eqn{\alpha}'s equal,
\item \eqn{\lambda}'s different and \eqn{\alpha = 1} (piecewise exponential distribution),
\item \eqn{\lambda}'s different and \eqn{\alpha = 2} (piecewise Rayleigh distribution).
}
}

\usage{
choose.pweibull(formula, data, criteria = "AIC", L.max = 5, t = NULL, 
    prec = 1e-04, max.iter = 1000, lambda.identical = FALSE, 
    alpha.identical = FALSE, alpha.fixed = FALSE)

choose2.pweibull(formula, data, criteria = "AIC", L.max = 5, t = NULL, 
    prec = 1e-04, max.iter = 1000, alpha.fixed = c(1, 2))
}

\arguments{
  \item{formula}{
A model formula of class \code{"formula"} describing the survival model
to be fitted. Details about model specification are given in \emph{Details}.
}

  \item{data}{
An optional data frame, list, or environment containing the variables in the
model. If not found in \code{data}, variables are taken from
\code{environment(formula)}.
}

  \item{criteria}{
Model selection criterion: \code{"AIC"} (default) or \code{"BIC"}.
}

  \item{L.max}{
Maximum number of partitions to consider (default \code{5}).
}

  \item{t}{
Optional fixed time partition. If provided, both \code{choose.pweibull} and
\code{choose2.pweibull} evaluate only the model combinations with the specified
partition.
}

  \item{prec}{
Numerical tolerance for the estimation algorithm (default \code{1e-4}).
}

  \item{max.iter}{
Maximum number of iterations for the estimation algorithm (default \code{1000}).
}

  \item{lambda.identical}{
Logical; should the \eqn{\lambda}'s be constrained to be equal?
(default \code{FALSE}).
}

  \item{alpha.identical}{
Logical; should the \eqn{\alpha}'s be constrained to be equal?
(default \code{FALSE}).
}

  \item{alpha.fixed}{
If \code{FALSE} (default), \eqn{\alpha} is estimated.  
If a positive numeric value is supplied, all \eqn{\alpha}'s are fixed at that value.
For \code{choose2.pweibull}, this may be a vector of fixed values.
}
}

\details{
The hazard function of the power piecewise Weibull model is

\deqn{
h(t \mid \boldsymbol{\lambda}, \boldsymbol{\alpha}) 
  = \lambda_\ell \alpha_\ell t^{\alpha_\ell - 1}, 
  \qquad t \in (a_{\ell-1}, a_\ell),\; \ell = 1,\ldots,L,
}

where \eqn{0 = a_0 < a_1 < \cdots < a_L < \infty} is the time partition,
\eqn{\boldsymbol{\lambda} = (\lambda_1,\ldots,\lambda_L)} and
\eqn{\boldsymbol{\alpha} = (\alpha_1,\ldots,\alpha_L)}.

The special cases include:

\itemize{
\item \eqn{\alpha_1 = \cdots = \alpha_L = 1}: the piecewise exponential model
(Feigl and Zelen, 1965; Friedman, 1982),
\item \eqn{\alpha_1 = \cdots = \alpha_L = 2}: a piecewise Rayleigh model.
}
}

\value{
A list with components:

\item{estimate}{A matrix of parameter estimates and standard errors for the selected partition.}
\item{logLik}{Log-likelihood evaluated at the parameter estimates.}
\item{t}{Selected time partition.}
\item{AIC}{Akaike Information Criterion.}
\item{BIC}{Bayesian Information Criterion.}
\item{L.sel}{Number of selected partitions.}
\item{AIC.L}{AIC values for \eqn{L = 1, \ldots, L.max}.}
\item{BIC.L}{BIC values for \eqn{L = 1, \ldots, L.max}.}
}

\references{
Feigl P., Zelen M. (1965). Estimation of exponential survival probabilities with concomitant information. \emph{Biometrics}, \bold{21}, 826-838.

Friedman M. (1982). Piecewise exponential models for survival data with covariates. \emph{Annals of Statistics}, \bold{10}, 101-113.

Gomez Y. M., Gallardo D. I., Arnold B. C. (2018). The power piecewise exponential model. \emph{Journal of Statistical Computation and Simulation}, \bold{88}, 825-840.
}


\author{
Diego I. Gallardo, Yolanda M. Gomez, Hector W. Gomez, and Barry C. Arnold.
}

\examples{
\donttest{

library(survival)
set.seed(3100)

n  <- 200
x1 <- rnorm(n)
x2 <- rnorm(n)

## drawing covariates
lambda <- c(0.05, 0.03)
rate   <- exp(cbind(x1, x2) \%*\% c(0.5, -0.5))

time2 = c()
for (i in 1:n)
  time2[i] <- rpweibull(1, rate = lambda * rate[i], alpha = c(1, 1), t = c(0, 10))

delta <- rbinom(n, size = 1, prob = 0.75)
cc    <- runif(n, 0, max(time2))
time  <- ifelse(delta == 1, time2, cc)

data  <- data.frame(time = time, x1 = x1, x2 = x2, delta = delta)

choose.pweibull(survival::Surv(time, delta) ~ x1 + x2, data = data, L.max = 3)
}
}
