#' @title First-Order Drug Release Kinetic Model
#' @name first_order_release
#' @description
#' Fits experimental cumulative drug release data to a first-order kinetic model
#' using linear regression on the log-transformed unreleased fraction. The function
#' supports optional grouping (formulation/batch) and pH-dependent analysis. It
#' can generate plots with straight lines and annotations for first-order rate
#' constant (k1), intercept, coefficient of determination (R^2), and time required
#' for 50-percent drug release (t50).
#'
#' @param data A data frame containing experimental drug release data.
#' @param time_col Character string specifying the column name for time.
#' @param log_remain_col Column name for log cumulative percent drug remaining.
#' @param group_col Optional character string specifying a column for grouping
#' (e.g., formulation/batch).
#' @param pH_col Optional character string specifying a column containing pH values.
#' @param plot Logical; if TRUE, generates a plot of experimental data with
#' first-order fitted curves.
#' @param annotate Logical; if TRUE, annotates the plot with k1, intercept, R^2,
#' and t50 (only if <= 2 groups).
#'
#' @import stats
#' @import ggplot2
#' @importFrom stats lm na.omit
#' @importFrom ggplot2 ggplot aes geom_point geom_line geom_smooth
#'   geom_text labs theme theme_bw element_text element_blank
#'
#' @return A list containing:
#' \describe{
#'   \item{\code{fitted_parameters}}{Data frame with k1, intercept, R^2, and t50
#'         values for each group or pH condition.}
#'   \item{\code{data}}{The processed data used for model fitting and plotting.}
#' }
#' @examples
#' # Example I: Single formulation
#' df_1 <- data.frame(
#'   time = c(0, 15, 30, 45, 60, 90, 120, 150, 180),
#'   log_remain = c(2, 1.947, 1.899, 1.840, 1.780, 1.625, 1.447, 1.182, 0.813)
#' )
#' first_order_release(
#'   data = df_1,
#'   time_col = "time",
#'   log_remain_col = "log_remain"
#' )
#'
#' # Example II: Two formulations (grouped, not pH-dependent)
#' df_2 <- data.frame(
#'   time = rep(c(0, 30, 60, 90, 120, 150), 2),
#'   log_remain = c(
#'     2.00, 1.84, 1.73, 1.53, 1.37, 1.25,  # Formulation A
#'     2.00, 1.88, 1.76, 1.67, 1.53, 1.39   # Formulation B
#'   ),
#'   formulation = rep(c("Formulation A", "Formulation B"), each = 6)
#' )
#' first_order_release(
#'   data = df_2,
#'   time_col = "time",
#'   log_remain_col = "log_remain",
#'   group_col = "formulation"
#' )
#'
#' # Example III: pH-dependent first-order release
#' df_pH <- data.frame(
#'   time = rep(c(0, 60, 120, 180), 2),
#'   log_remain = c(
#'     2.00, 1.74, 1.38, 1.11,  # pH 7.4
#'     2.00, 1.84, 1.66, 1.48   # pH 4.5
#'   ),
#'   pH = rep(c(7.4, 4.5), each = 4)
#' )
#' first_order_release(
#'   data = df_pH,
#'   time_col = "time",
#'   log_remain_col = "log_remain",
#'   pH_col = "pH"
#' )
#'
#' # Example IV: Two formulations under two pH conditions
#' df1 <- data.frame(
#'   time = rep(c(0, 30, 60, 90, 120, 150, 180), 2),
#'   log_remain = c(
#'     2.000, 1.918, 1.842, 1.755, 1.685, 1.598, 1.520,
#'     2.000, 1.865, 1.748, 1.612, 1.488, 1.352, 1.225
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 7)
#' )
#' df2 <- data.frame(
#'   time = rep(c(0, 20, 40, 60, 80, 100, 120), 2),
#'   log_remain = c(
#'     2.000, 1.936, 1.872, 1.806, 1.742, 1.675, 1.610,
#'     2.000, 1.882, 1.760, 1.645, 1.522, 1.408, 1.295
#'   ),
#'   pH = rep(c(4.5, 7.6), each = 7)
#' )
#' df_all <- rbind(
#'   cbind(formulation = "Dataset 1", df1),
#'   cbind(formulation = "Dataset 2", df2)
#' )
#' first_order_release(
#'   data = df_all,
#'   time_col = "time",
#'   log_remain_col = "log_remain",
#'   group_col = "formulation",
#'   pH_col = "pH"
#' )
#' @references Ostwald, W. (1884) <doi:10.1002/prac.18840290139> Studien zur
#' chemischen Dynamik. Journal für Praktische Chemie, 29(1), 385–408.
#' @references Noyes, A. A., & Whitney, W. R. (1897) <doi:10.1021/ja02086a003> The
#' rate of solution of solid substances in their own solutions. Journal of the
#' American Chemical Society, 19(12), 930–934.
#' @author Paul Angelo C. Manlapaz
#' @export

utils::globalVariables(c("time", "log_remain", "group", "fitted", "k1", "intercept",
                         "R2", "t50", "label", "x_pos", "y_pos", "hjust", "vjust"))

first_order_release <- function(data,
                                time_col = "time",
                                log_remain_col = "log_remain",
                                group_col = NULL,
                                pH_col = NULL,
                                plot = TRUE,
                                annotate = TRUE) {

  if (!requireNamespace("ggplot2", quietly = TRUE)) {
    stop("Package 'ggplot2' is required.")
  }

  # -------------------------
  # Prepare data
  # -------------------------
  df <- data[, c(time_col, log_remain_col, group_col, pH_col), drop = FALSE]
  df <- stats::na.omit(df)
  colnames(df)[1:2] <- c("time", "log_remain")

  if (!is.null(group_col) && !is.null(pH_col)) {
    df$group <- paste0(df[[group_col]], " | pH ", df[[pH_col]])
  } else if (!is.null(group_col)) {
    df$group <- as.factor(df[[group_col]])
  } else if (!is.null(pH_col)) {
    df$group <- paste0("pH ", df[[pH_col]])
  } else {
    df$group <- "Experimental"
  }

  df$group <- as.factor(df$group)

  # -------------------------
  # First-order fitting
  # -------------------------
  fit_results <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fit <- stats::lm(log_remain ~ time, data = d)
    s <- summary(fit)

    k1 <- -coef(fit)[2]
    intercept <- coef(fit)[1]
    R2 <- s$r.squared
    t50 <- if (k1 > 0) log(2) / k1 else NA_real_

    data.frame(group = unique(d$group), k1 = k1, intercept = intercept, R2 = R2, t50 = t50)
  }))

  # -------------------------
  # Fitted values for plotting
  # -------------------------
  df_fit <- do.call(rbind, lapply(split(df, df$group), function(d) {
    fr <- fit_results[fit_results$group == unique(d$group), ]
    d$fitted <- fr$intercept - fr$k1 * d$time
    d
  }))

  # -------------------------
  # Plot
  # -------------------------
  if (plot) {
    p <- ggplot2::ggplot(df, ggplot2::aes(x = time, y = log_remain, color = group)) +
      ggplot2::geom_point(size = 3) +
      ggplot2::geom_line(linewidth = 1.2) +
      ggplot2::geom_smooth(method = "lm", se = FALSE, color = "black", linewidth = 1) +
      ggplot2::labs(
        title = "First-Order Drug Release (Log % Remaining)",
        subtitle = "Experimental data with first-order linear fit",
        x = "Time (minutes)",
        y = "Log Cumulative % Drug Remaining",
        color = "Group"
      ) +
      ggplot2::theme_bw(base_size = 14) +
      ggplot2::theme(
        plot.title = ggplot2::element_text(face = "bold", hjust = 0.5),
        plot.subtitle = ggplot2::element_text(hjust = 0.5),
        panel.grid.major = ggplot2::element_blank(),
        panel.grid.minor = ggplot2::element_blank()
      )

    num_groups <- nlevels(df$group)
    if (annotate && num_groups <= 2) {
      ann <- fit_results
      ann$label <- paste0(
        "k1 = ", round(ann$k1, 3), " /min\n",
        "Intercept = ", round(ann$intercept, 2), "\n",
        "R^2 = ", round(ann$R2, 3), "\n",
        "t[50] = ", round(ann$t50, 1), " min"
      )

      x_min <- min(df$time, na.rm = TRUE)
      x_max <- max(df$time, na.rm = TRUE)
      y_min <- min(df$log_remain, na.rm = TRUE)
      y_max <- max(df$log_remain, na.rm = TRUE)

      ann$x_pos <- c(
        x_max - 0.05 * (x_max - x_min),
        x_min + 0.05 * (x_max - x_min)
      )[seq_len(nrow(ann))]

      ann$y_pos <- c(
        y_max - 0.05 * (y_max - y_min),
        y_min + 0.10 * (y_max - y_min)
      )[seq_len(nrow(ann))]

      ann$hjust <- c(1, 0)[seq_len(nrow(ann))]
      ann$vjust <- c(1, 0)[seq_len(nrow(ann))]

      p <- p +
        ggplot2::geom_text(
          data = ann,
          ggplot2::aes(x = x_pos, y = y_pos, label = label, color = group),
          hjust = ann$hjust,
          vjust = ann$vjust,
          size = 4,
          show.legend = FALSE
        )

    } else if (num_groups > 2 && annotate) {
      message("More than 2 groups detected - annotations disabled to prevent overlap.")
    }

    print(p)
  }

  return(list(
    fitted_parameters = fit_results,
    data = df
  ))
}
